# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008, 2009 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import gtk
import gobject

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Windows constants
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
(CATEGORY_ICON, CATEGORY_NAME, CATEGORY_OBJ) = range (3)
(ATTR_ID, ATTR_NAME, ATTR_INITVALUE) = range (3)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Widget
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Widget (gtk.VBox):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self, mediator):
    gtk.VBox.__init__ (self)
    self.mediator = mediator.new_client_mediator ()
    self.set_border_width (5)
    self.set_spacing (5)
    self.show ()

    hpaned = gtk.HPaned ()
    hpaned.show ()
    self.pack_start (hpaned)

    # category listview
    frame = gtk.Frame ()
    frame.show ()
    hpaned.pack1 (frame, True)

    datastore = gtk.ListStore (gtk.gdk.Pixbuf, str, object)
    datastore.set_sort_column_id (CATEGORY_NAME, gtk.SORT_ASCENDING)

    for category in self.mediator.call ('category.list.iter'):
      icon = self.mediator.call ('ui.render-icon', category.icon_data, 32, 32)
      datastore.append ((icon, category.name, category))

    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_NEVER, gtk.POLICY_AUTOMATIC)
    sw.show ()
    frame.add (sw)

    self.category_listview = gtk.TreeView (datastore)
    self.category_listview.set_rules_hint (True)
    self.category_listview.set_headers_visible (False)
    self.category_listview.show ()
    sw.add (self.category_listview)

    selection = self.category_listview.get_selection ()
    selection.connect ('changed', self.on_category_selection_changed)

    renderer = gtk.CellRendererPixbuf ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'pixbuf', CATEGORY_ICON)
    self.category_listview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', CATEGORY_NAME)
    self.category_listview.append_column (tvcolumn)

    # notebook
    notebook = gtk.Notebook ()
    notebook.show ()
    hpaned.pack2 (notebook)

    # page: general
    vbox1 = gtk.VBox ()
    vbox1.set_border_width (5)
    vbox1.show ()
    notebook.append_page (vbox1, gtk.Label ('General'))

    table = gtk.Table (3, 3)
    table.set_row_spacings (10)
    table.set_col_spacings (5)
    table.show ()
    vbox1.pack_start (table, False, False)

    label = gtk.Label ()
    label.set_markup ('<b>Name</b>')
    label.set_alignment (1.0, -1)
    label.show ()
    table.attach (label, 0, 1, 0, 1, gtk.FILL, 0)

    self.category_name_entry = gtk.Entry ()
    self.category_name_entry.connect ('changed', self.on_category_name_changed)
    self.category_name_entry.show ()
    table.attach (self.category_name_entry, 1, 3, 0, 1, yoptions=0)

    label = gtk.Label ()
    label.set_markup ('<b>Icon</b>')
    label.set_alignment (1.0, -1)
    label.show ()
    table.attach (label, 0, 1, 1, 2, gtk.FILL, 0)

    self.category_icon_button = gtk.Button ()
    self.category_icon_button.connect ('clicked', self.on_category_icon_edit)
    self.category_icon_button.show ()

    self.category_image = gtk.Image ()
    self.category_image.set_from_stock (gtk.STOCK_MISSING_IMAGE, gtk.ICON_SIZE_BUTTON)
    self.category_image.show ()
    self.category_icon_button.add (self.category_image)

    table.attach (self.category_icon_button, 1, 2, 1, 2, 0, 0)

    placeholder = gtk.Label (' ')
    placeholder.show ()
    table.attach (placeholder, 2, 3, 1, 2, yoptions=0)

    self.category_is_dataholder_cb = gtk.CheckButton ('Is dataholder')
    self.category_is_dataholder_cb.connect ('toggled', self.on_category_is_dataholder_toggled)
    self.category_is_dataholder_cb.show ()
    table.attach (self.category_is_dataholder_cb, 1, 3, 2, 3, yoptions=0)

    # page: attributes
    vbox_page = gtk.VBox ()
    vbox_page.set_spacing (5)
    vbox_page.show ()
    notebook.append_page (vbox_page, gtk.Label ('Attributes'))
    
    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    vbox_page.pack_start (sw)

    # attributes listview
    datastore = gtk.ListStore (str, str, str)
    self.attribute_listview = gtk.TreeView (datastore)
    self.attribute_listview.set_rules_hint (True)
    self.attribute_listview.set_enable_search (False)

    selection = self.attribute_listview.get_selection ()
    selection.connect ('changed', self.on_attribute_selection_changed)

    renderer = gtk.CellRendererText ()
    renderer.set_property ('editable', True)
    renderer.connect ('edited', self.on_attribute_id_edited)

    tvcolumn = gtk.TreeViewColumn ('ID')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', ATTR_ID)
    tvcolumn.set_resizable (True)
    self.attribute_listview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    renderer.set_property ('editable', True)
    renderer.connect ('edited', self.on_attribute_name_edited)

    tvcolumn = gtk.TreeViewColumn ('Name')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', ATTR_NAME)
    self.attribute_listview.append_column (tvcolumn)
    self.attribute_listview.show ()
    sw.add (self.attribute_listview)

    # initvalue
    hbox = gtk.HBox ()
    hbox.show ()
    vbox_page.pack_start (hbox, False)

    label = gtk.Label ()
    label.set_markup ('<b>Init value</b>')
    label.show ()
    hbox.pack_start (label, False)

    self.initvalue_entry = gtk.Entry ()
    self.initvalue_entry.connect ('changed', self.on_initvalue_changed)
    self.initvalue_entry.show ()
    hbox.pack_start (self.initvalue_entry)

    # attribute buttons
    hbox = gtk.HBox ()
    hbox.show ()
    vbox_page.pack_start (hbox, False)

    self.add_attr_button = gtk.Button (stock=gtk.STOCK_ADD)
    self.add_attr_button.connect ('clicked', self.on_attribute_add)
    self.add_attr_button.show ()
    hbox.pack_start (self.add_attr_button, False)
    
    self.remove_attr_button = gtk.Button (stock=gtk.STOCK_REMOVE)
    self.remove_attr_button.set_sensitive (False)
    self.remove_attr_button.connect ('clicked', self.on_attribute_remove)
    self.remove_attr_button.show ()
    hbox.pack_start (self.remove_attr_button, False)

    self.up_attr_button = gtk.Button (stock=gtk.STOCK_GO_UP)
    self.up_attr_button.set_sensitive (False)
    self.up_attr_button.connect ('clicked', self.on_attribute_up)
    self.up_attr_button.show ()
    hbox.pack_start (self.up_attr_button, False)

    self.down_attr_button = gtk.Button (stock=gtk.STOCK_GO_DOWN)
    self.down_attr_button.set_sensitive (False)
    self.down_attr_button.connect ('clicked', self.on_attribute_down)
    self.down_attr_button.show ()
    hbox.pack_start (self.down_attr_button, False)

    # buttons
    hbox = gtk.HBox ()
    hbox.show ()
    self.pack_start (hbox, False)

    button = gtk.Button (stock=gtk.STOCK_ADD)
    button.connect ('clicked', self.on_category_add)
    button.show ()
    hbox.pack_start (button, False)
    
    self.remove_button = gtk.Button (stock=gtk.STOCK_REMOVE)
    self.remove_button.connect ('clicked', self.on_category_remove)
    self.remove_button.set_sensitive (False)
    self.remove_button.show ()
    hbox.pack_start (self.remove_button, False)

    button = gtk.Button (stock=gtk.STOCK_CLOSE)
    button.connect ('clicked', self.on_extension_close)
    button.show ()
    hbox.pack_end (button, False)
    
    self.save_button = gtk.Button (stock=gtk.STOCK_SAVE)
    self.save_button.set_sensitive (False)
    self.save_button.connect ('clicked', self.on_model_save)
    self.save_button.show ()
    hbox.pack_end (self.save_button, False)

    # flags
    self.is_modified = False		# categories are modified
    self.is_selecting = False		# selecting new category
    self.is_setting_initvalue = False	# setting initvalue text
    self.category = None		# current selected category

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Populate attribute listview
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def populate_attribute_listview (self, category):
    model = self.attribute_listview.get_model ()
    model.clear ()

    for a in self.mediator.call ('category.iter-attributes', category.id):
      model.append ((a.id, a.name, a.initvalue))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle close button
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_extension_close (self, widget, *args):
    self.mediator.call ('ui.working-area.close', self.working_area.id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief show save/ignore/cancel dialog if there are modified items
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_widget_stopped (self):
    model = self.category_listview.get_model ()

    # if model was modified, show save/ignore/cancel dialog
    if self.is_modified:
      dialog = gtk.MessageDialog (None,
                  gtk.DIALOG_MODAL,
                  gtk.MESSAGE_QUESTION,
                  gtk.BUTTONS_YES_NO,
                  "Save changes before closing?")
      dialog.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
      rc = dialog.run ()
      dialog.destroy ()

      if rc == gtk.RESPONSE_CANCEL:
        return True

      elif rc == gtk.RESPONSE_YES:
        self.on_model_save ()

    # clean-up code
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle model modification
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_model_modified (self):
    self.is_modified = True
    self.save_button.set_sensitive (True)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief save model
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_model_save (self, *args):
    self.mediator.call ('category.list.save')
    self.is_modified = False
    self.save_button.set_sensitive (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category selection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_selection_changed (self, selection, *args):
    model, iter = selection.get_selected ()
    self.is_selecting = True

    # fill info
    if iter:
      icon = model.get_value (iter, CATEGORY_ICON)
      category = model.get_value (iter, CATEGORY_OBJ)

      self.category_name_entry.set_text (category.name or '')
      self.category_image.set_from_pixbuf (icon)
      self.category_is_dataholder_cb.set_active (category.is_dataholder)
      self.populate_attribute_listview (category)
      self.category = category

    else:
      self.category_name_entry.set_text ('')
      self.category_image.clear ()
      self.category_is_dataholder_cb.set_active (False)
      attr_model = self.attribute_listview.get_model ()
      attr_model.clear ()
      self.category = None

    self.is_selecting = False

    # enable/disable widgets
    enabled = iter != None
    self.remove_button.set_sensitive (enabled)
    self.category_name_entry.set_sensitive (enabled)
    self.category_icon_button.set_sensitive (enabled)
    self.add_attr_button.set_sensitive (enabled)
    self.category_is_dataholder_cb.set_sensitive (enabled)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category add
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_add (self, widget, *args):

    # show dialog to enter category ID and Name
    dialog = gtk.Dialog ('Creating category', None, gtk.DIALOG_MODAL,
      (gtk.STOCK_OK, gtk.RESPONSE_OK, gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL))
    dialog.set_position (gtk.WIN_POS_CENTER)
    dialog.set_type_hint (gtk.gdk.WINDOW_TYPE_HINT_DIALOG)
    dialog.set_border_width (10)

    table = gtk.Table (2, 2)
    table.set_row_spacings (5)
    table.set_col_spacings (5)
    table.show ()
    dialog.vbox.pack_start (table)

    label = gtk.Label ('ID')
    label.set_alignment (1, -1)
    label.show ()
    table.attach (label, 0, 1, 0, 1, 0, 0)

    dialog.id_entry = gtk.Entry ()
    dialog.id_entry.show ()
    table.attach (dialog.id_entry, 1, 2, 0, 1, yoptions=0)

    label = gtk.Label ('Name')
    label.set_alignment (1, -1)
    label.show ()
    table.attach (label, 0, 1, 1, 2, 0, 0)

    dialog.name_entry = gtk.Entry ()
    dialog.name_entry.show ()
    table.attach (dialog.name_entry, 1, 2, 1, 2, yoptions=0)

    rc = dialog.run ()
    cat_id = dialog.id_entry.get_text ()
    cat_name = dialog.name_entry.get_text ()
    dialog.destroy ()

    # if user pressed OK, create category
    if rc == gtk.RESPONSE_OK and cat_id:

      # create category
      category = self.mediator.call ('category.new', cat_id)
      category.name = cat_name

      # update window
      icon = self.mediator.call ('ui.render-icon', category.icon_data, 32, 32)
      model = self.category_listview.get_model ()
      iter = model.append ((icon, category.name, category))

      selection = self.category_listview.get_selection ()
      selection.select_iter (iter)

      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category remove
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_remove (self, widget, *args):
    selection = self.category_listview.get_selection ()
    model, iter = selection.get_selected ()

    if iter:
      category = model.get_value (iter, CATEGORY_OBJ)

      dialog = gtk.MessageDialog (None,
        gtk.DIALOG_MODAL,
        gtk.MESSAGE_QUESTION,
        gtk.BUTTONS_YES_NO,
        "You are about to remove category '%s'. Are you sure?" % category.name)

      rc = dialog.run ()
      dialog.destroy ()

      if rc != gtk.RESPONSE_YES:
        return

      # remove category
      self.mediator.call ('category.remove', category.id)

      # update window
      has_next = model.remove (iter)

      if has_next:
        selection.select_iter (iter)
      else:
        self.on_category_unselected ()

      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category name edition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_name_changed (self, widget, *args):
    model, iter = self.category_listview.get_selection ().get_selected ()

    if self.is_selecting == False and iter:
      value = self.category_name_entry.get_text ()

      category = model.get_value (iter, CATEGORY_OBJ)
      category.name = value
      self.mediator.call ('category.set', category)
      model.set_value (iter, CATEGORY_NAME, value)

      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category icon edition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_icon_edit (self, widget, *args):

    # choose file
    fs = gtk.FileChooserDialog ('Choose 32x32 icon filename')
    fs.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
    fs.add_button (gtk.STOCK_OK, gtk.RESPONSE_OK)

    filter = gtk.FileFilter ()
    filter.add_pattern ('*.png')
    filter.add_pattern ('*.jpg')
    filter.add_pattern ('*.gif')
    filter.add_pattern ('*.svg')
    fs.set_filter (filter)

    rc = fs.run ()
    filename = fs.get_filename ()
    fs.destroy ()

    if rc != gtk.RESPONSE_OK:
      return

    # read file
    fp = open (filename)
    data = fp.read ()
    fp.close ()

    # set new icon
    model, iter = self.category_listview.get_selection ().get_selected ()

    if iter:
      category = model.get_value (iter, CATEGORY_OBJ)
      category.icon_data = data.encode ('base64')
      self.mediator.call ('category.set', category)

      icon = self.mediator.call ('ui.render-icon', category.icon_data, 32, 32)
      self.category_image.set_from_pixbuf (icon)
      model.set_value (iter, CATEGORY_ICON, icon)

      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle category is dataholder toggled
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_category_is_dataholder_toggled (self, widget, *args):
    model, iter = self.category_listview.get_selection ().get_selected ()

    if self.is_selecting == False and iter:
      category = model.get_value (iter, CATEGORY_OBJ)
      category.is_dataholder = widget.get_active ()
      self.mediator.call ('category.set', category)
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute selection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_selection_changed (self, selection, *args):
    model, iter = selection.get_selected ()

    # get initvalue
    if iter:
      initvalue = model.get_value (iter, ATTR_INITVALUE) or ''
      row = model.get_path (iter)[0]
    else:
      initvalue = ''

    # set entry text
    self.is_setting_initvalue = True
    self.initvalue_entry.set_text (initvalue)
    self.is_setting_initvalue = False

    # enable/disable widgets
    enabled = iter != None
    up_enabled = enabled and row > 0
    down_enabled = enabled and model.iter_next (iter) != None

    self.initvalue_entry.set_sensitive (enabled)
    self.remove_attr_button.set_sensitive (enabled)
    self.up_attr_button.set_sensitive (up_enabled)
    self.down_attr_button.set_sensitive (down_enabled)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute add
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_add (self, widget, *args):
    model = self.attribute_listview.get_model ()
    iter = model.append (('<ID>', '<NAME>', ''))
    path = model.get_path (iter)
    column = self.attribute_listview.get_column (ATTR_ID)
    self.attribute_listview.set_cursor (path, column, True)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute remotion
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_remove (self, widget, *args):
    selection = self.attribute_listview.get_selection ()
    model, iter = selection.get_selected ()

    if iter:
      # remove attribute
      attr_id = model.get_value (iter, ATTR_ID)
      self.mediator.call ('category.attribute.remove', self.category.id, attr_id)

      # update window
      has_next = model.remove (iter)

      if has_next:
        selection.select_iter (iter)

      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle move up
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_up (self, widget, *args):
    model, iter = self.attribute_listview.get_selection ().get_selected ()

    if iter:
      # update attribute
      dest_row = model.get_path (iter)[0] - 1
      attr_id = model.get_value (iter, ATTR_ID)
      self.mediator.call ('category.attribute.move', self.category.id, attr_id, dest_row)

      # update window
      dest_iter = model.get_iter (dest_row)
      model.move_before (iter, dest_iter)
      self.attribute_listview.scroll_to_cell (dest_row)

      if dest_row == 0:
        self.up_attr_button.set_sensitive (False)

      self.down_attr_button.set_sensitive (True)
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle move down
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_down (self, widget, *args):
    model, iter = self.attribute_listview.get_selection ().get_selected ()

    if iter:
      # update attribute
      dest_row = model.get_path (iter)[0] + 1
      attr_id = model.get_value (iter, ATTR_ID)
      self.mediator.call ('category.attribute.move', self.category.id, attr_id, dest_row)

      # update window
      dest_iter = model.get_iter (dest_row)
      model.move_after (iter, dest_iter)
      self.attribute_listview.scroll_to_cell (dest_row)

      if dest_row + 1 >= len (model):
        self.down_attr_button.set_sensitive (False)

      self.up_attr_button.set_sensitive (True)
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute ID edition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_id_edited (self, cell, path, new_text, *args):
    model = self.attribute_listview.get_model ()
    iter = model.get_iter_from_string (path)
    text = model.get_value (iter, ATTR_ID)

    if text == '<ID>' and new_text.strip () != '' and new_text != '<ID>':
      model.set_value (iter, ATTR_ID, new_text)
      attr = self.mediator.call ('category.attribute.new', self.category.id, new_text)
      self.on_model_modified ()

    elif text == '<ID>':
      dialog = gtk.MessageDialog (None, gtk.DIALOG_MODAL, gtk.MESSAGE_ERROR,
                  gtk.BUTTONS_OK, 'attribute ID is mandatory')
      dialog.run ()
      dialog.destroy ()

      column = self.attribute_listview.get_column (ATTR_ID)
      gobject.idle_add (self.attribute_listview.set_cursor, path, column, True)

    elif text != new_text:
      dialog = gtk.MessageDialog (None, gtk.DIALOG_MODAL, gtk.MESSAGE_ERROR,
                  gtk.BUTTONS_OK, 'cannot changed attribute ID')
      dialog.run ()
      dialog.destroy ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle attribute name edition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_attribute_name_edited (self, cell, path, new_text, *args):
    model = self.attribute_listview.get_model ()
    iter = model.get_iter_from_string (path)
    text = model.get_value (iter, ATTR_NAME)

    if text != new_text:
      attr_id = model.get_value (iter, ATTR_ID) 
      model.set_value (iter, ATTR_NAME, new_text)
      self.mediator.call ('category.attribute.set-name', self.category.id, attr_id, new_text)
      self.on_model_modified ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle initvalue edition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_initvalue_changed (self, widget, *args):
    selection = self.attribute_listview.get_selection ()
    model, iter = selection.get_selected ()

    if not self.is_setting_initvalue and iter:
      attr_id = model.get_value (iter, ATTR_ID) 
      value = self.initvalue_entry.get_text ()
      model.set_value (iter, ATTR_INITVALUE, value)

      self.mediator.call ('category.attribute.set-init-value', self.category.id, attr_id, value)
      self.on_model_modified ()

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief icon data
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ICON_DATA = '''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'''

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Extension class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Extension (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator):
    self.id = 'category-manager'
    self.name = 'Category Manager'
    self.author = 'Eduardo Aguiar'
    self.version = '0.1.2'
    self.description = 'Category manager'
    self.mediator = mediator.copy ()
    self.icon_data = ICON_DATA

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Start extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def start (self):
    self.mediator.call ('toolbox.add', self.id, self.icon_data, 'Category\nManager', self.on_activate)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Stop extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def stop (self):
    self.mediator.call ('ui.working-area.del', self.id)
    #self.mediator.call ('toolbox.remove', self.id)
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: on_activate
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_activate (self, item_id):
    working_area = self.mediator.call ('ui.working-area.get', self.id)

    if not working_area:
      widget = Widget (self.mediator)

      working_area = self.mediator.call ('ui.working-area.new', self.id)
      working_area.set_default_size (600, 400)
      working_area.set_title (self.name)
      working_area.set_icon (self.icon_data)
      working_area.set_widget (widget)

    working_area.show ()
