# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import email.utils
import email.header
import datetime

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Decode .eml mail files
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Decoder (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Message class
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  class Message (object):
    def __init__ (self):
      self.id = None
      self.timestamp = None
      self.subject = ''
      self.sender = ''
      self.sender_email = ''
      self.to = []
      self.cc = []
      self.recipients = []
      self.text = ''
      self.parts = []

    def __iter__ (self):
      return iter (self.parts)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Message part
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  class MessagePart (object):
    def __init__ (self):
      self.data = ''
      self.encoding = ''
      self.content_type = ''

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Decode data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def decode (self, data):
    msg = self.Message ()

    try:
      py_msg = email.message_from_string (data)

      timestamp = email.utils.parsedate (py_msg.get ('date'))
      if timestamp:
        msg.timestamp = datetime.datetime (*(timestamp[:6]))

      msg.subject = self.decode_string (py_msg.get ('subject'))
      sender, sender_email = email.utils.parseaddr (py_msg.get ('from'))
      msg.sender = self.decode_string (sender)
      msg.sender_email = self.decode_string (sender_email)

      msg.to = self.decode_addresses (py_msg.get_all ('to', []))
      msg.cc = self.decode_addresses (py_msg.get_all ('cc', []))
      msg.bcc = self.decode_addresses (py_msg.get_all ('bcc', []))
      msg.recipients = msg.to + msg.cc + msg.bcc

      charset = py_msg.get_content_charset () or 'iso-8859-15'
      if charset in ('default_charset', 'x-user-defined'):
        charset = 'iso-8859-15'

      msg.data = py_msg.as_string ()
      msg.text = data.decode (charset)

      for part in py_msg.walk ():
        p = self.MessagePart ()
        p.content_type = part.get_content_type ()
        p.encoding = part.get_content_charset ()
        p.data = part.get_payload (decode=True)
        msg.parts.append (p)

    except Exception, e:
      print 'Invalid message: %s' % e # @TODO handle error

    return msg

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Decode rfc 2047 string
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def decode_string (self, str):
    text = u''

    try:
      for str, encoding in email.header.decode_header (str):
        try:
          text += str.decode (encoding or 'iso-8859-15')

        except:
          # try another encoding
          try:
            text += str.decode ('iso-8859-1')
          except:
            print 'encoding=<iso-8859-1>, text=<%s>' % str

    except:
      return text

    return text

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Decode addresses
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def decode_addresses (self, value):
    addresses = email.utils.getaddresses (value)
    decoded = []

    for name, address in addresses:
      decoded.append ((self.decode_string (name), self.decode_string (address)))
 
    return decoded
