# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import mobius
import mobius.config
import mobius.model.case
import mobius.ui.manager
import mobius.pickle.application
import mobius.pickle.case
import glob
import imp
import os
import shutil

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Application class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Application (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize application
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self):
    self.__cases = {}
    self.__resources = {}
    self.__next_uid = 0		# session case id

    self.ui_manager = None
    self.next_new_id = 0	# untitled case id

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add a resource (category, filesystem, etc) to application
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_resource (self, type, id, value):
    self.__resources.setdefault (type, {})[id] = value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get a resource
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_resource (self, type, id):
    return self.__resources.setdefault (type, {}).get (id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Pop resource from application
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def pop_resource (self, type, id):
    return self.__resources.setdefault (type, {}).pop (id, None)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get a list of resources
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_resource_list (self, type):
    return self.__resources.setdefault (type, {}).values ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Create a case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def new_case (self):
    case = mobius.model.case.Case ()
    case.uid = self.__next_uid
    case.app = self
    case.set_attribute ('id', 'untitled%02d' % self.next_new_id)
    case.set_attribute ('name', 'Untitled Case #%02d' % self.next_new_id)
    case.filename = 'untitled%02d.case' % self.next_new_id

    self.__cases[case.uid] = case
    self.__next_uid += 1
    self.next_new_id += 1

    return case

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Open case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def open_case (self, path):
    pickle = mobius.pickle.case.Pickle ()

    case = pickle.load (path)
    case.uid = self.__next_uid
    case.app = self

    self.__cases[case.uid] = case
    self.__next_uid += 1

    return case

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_case (self, case):
    pickle = mobius.pickle.case.Pickle ()
    pickle.save (case)
    case.set_modified (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Close case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def close_case (self, case):
    case = self.__cases.pop (case.uid)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save all cases
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_all_cases (self):
    for case in self.__cases.itervalues ():
      if case.is_modified ():
        self.save_case (case)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return iterator to cases
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def iter_cases (self):
    return self.__cases.itervalues ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Check if there is at least a case modified
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def is_modified (self):
    for case in self.__cases.itervalues ():
      if case.is_modified ():
        return True
    return False

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return path to application file/directory
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_path (self, sub_path=None):
    path = os.path.expandvars ("$HOME/.mobiusft")

    if sub_path:
      path = os.path.join (path, sub_path)

    return path
      
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return application directory/subdirectory
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_directory (self, dirname=None):
    dir = os.path.expandvars ("$HOME/.mobiusft")

    if dirname:
      dir = os.path.join (dir, dirname)

    return dir
      
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Create application subdirectory
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def make_directory (self, dirname=None):
    dir = self.get_path (dirname)

    if not os.path.isdir (dir):
      os.makedirs (dir)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Import extension from path
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def import_extension (self, path):
    module = imp.new_module ('extension')
    execfile (path, module.__dict__)

    extension = module.Extension (mobius.mediator)
    extension.path = path
    extension.module = module
    return extension

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Start extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def start_extension (self, extension):
    extension.start ()
    self.add_resource ('extension', extension.id, extension)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Install an extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def install_extension (self, extension):

    # run extension install method, if it exists
    install_method = getattr (extension, 'install', None)
    if install_method:
      install_method ()

    # copy file to extensions directory
    extension_dir = self.get_path ('extensions')
    shutil.copy (extension.path, extension_dir)
    extension.path = os.path.join (extension_dir, os.path.basename (extension.path))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Uninstall an extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def uninstall_extension (self, id):

    # stop extension
    extension = self.pop_resource ('extension', id)
    extension.stop ()

    # run extension uninstall method, if it exists
    uninstall_method = getattr (extension, 'uninstall', None)
    if uninstall_method:
      uninstall_method ()

    # remove file from extensions directory
    os.remove (extension.path)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Run application
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def run (self):

    # advertise services
    mobius.mediator.advertise ('app.get-path', self.svc_get_path)
    mobius.mediator.advertise ('app.get-name', self.svc_get_name)
    mobius.mediator.advertise ('app.get-version', self.svc_get_version)
    mobius.mediator.advertise ('app.get-window-title', self.svc_get_window_title)
    mobius.mediator.advertise ('item.set-attribute', self.svc_item_set_attribute)

    # start extensions
    #self.__start_extensions ()

    # configure user account
    self.__configure_user ()

    # start UI
    self.ui_manager = mobius.ui.manager.UIManager (self)
    # @todo better toolbox mechanism to allow extensions start before UI
    self.__start_extensions ()

    # load configuration
    config_path = os.path.join (self.get_directory (), 'mobius.xml')
    app_pickle = mobius.pickle.application.Pickle ()
    app_pickle.load (self, config_path)

    # start application
    self.ui_manager.start ()

    # stop application
    self.__stop_extensions ()
    app_pickle.save (self, config_path)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return path to application file/directory
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_get_path (self, sub_path=None):
    path = os.path.expandvars ("$HOME/.mobiusft")

    if sub_path:
      path = os.path.join (path, sub_path)

    return path

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return application name
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_get_name (self):
    return mobius.config.APP_NAME

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return application version
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_get_version (self):
    return mobius.config.APP_VERSION

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Return standard window title
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_get_window_title (self, subtitle=None):
    title = '%s v%s' % (mobius.config.APP_NAME, mobius.config.APP_VERSION)
    if subtitle:
      title += ' - ' + subtitle
    return title

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set attribute from item
  # @todo Find a better place for this service :-(
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_item_set_attribute (self, item, attr, value):
    old_value = item.get_attribute (attr)

    if old_value != value:
      item.set_attribute (attr, value)
      mobius.mediator.emit ('item.attribute-modified', item, attr, old_value, value)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Configure application for current user
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __configure_user (self):
    self.make_directory ()

    # install extensions, if necessary
    if not os.path.isdir (self.get_path ('extensions')):
      self.make_directory ('extensions')
      extension_dir = mobius.config.get_data_path ('extensions')

      for path in glob.iglob ('%s/*.py' % extension_dir):
        extension = self.import_extension (path)
        self.install_extension (extension)

    # create category.xml, if necessary
    # @todo move code below to CategoryManager
    path = self.get_path ('category.xml')
    if not os.path.exists (path):
      shutil.copy (mobius.config.get_data_path ('category.xml'), path)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Start extensions
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __start_extensions (self):
    extension_dir = self.get_directory ('extensions')

    for path in glob.iglob ('%s/*.py' % extension_dir):
      extension = self.import_extension (path)
      extension.start ()
      self.add_resource ('extension', extension.id, extension)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Stop extensions
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __stop_extensions (self):
    for extension in self.get_resource_list ('extension'):
      extension.stop ()
