# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import gtk
import gtk.gdk
import mobius
import mobius.config
import mobius.ui.about_dialog
import mobius.ui.extension_manager_dialog
import mobius.ui.case_window

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Main window
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Window (gtk.Window):

  def __init__ (self, toolbox_model):
    self.case = None

    gtk.Window.__init__ (self)
    self.connect ('delete-event', self.on_file_quit)
    self.set_size_request (400, 600)
    self.set_title ('%s v%s' % (mobius.config.APP_NAME, mobius.config.APP_VERSION))

    # accel_group
    accel_group = gtk.AccelGroup ()
    self.add_accel_group (accel_group)

    # accel_group
    accel_group = gtk.AccelGroup ()
    self.add_accel_group (accel_group)

    # vbox
    vbox = gtk.VBox (False, 1)
    vbox.set_border_width (1)
    self.add (vbox)
    vbox.show ()

    # menubar
    menubar = gtk.MenuBar ()
    menubar.show ()
    vbox.pack_start (menubar, False, False)

    item = gtk.MenuItem ('_File')
    item.show ()
    menubar.append (item)

    menu = gtk.Menu ()
    menu.show ()
    item.set_submenu (menu)

    item = gtk.ImageMenuItem (gtk.STOCK_NEW, accel_group)
    item.connect ("activate", self.on_file_new)
    item.show ()
    menu.append (item)

    item = gtk.ImageMenuItem (gtk.STOCK_OPEN, accel_group)
    item.connect ("activate", self.on_file_open)
    item.show ()
    menu.append (item)

    item = gtk.SeparatorMenuItem ()
    item.show ()
    menu.append (item)

    item = gtk.ImageMenuItem (gtk.STOCK_QUIT, accel_group)
    item.connect ("activate", self.on_file_quit)
    item.show ()
    menu.append (item)

    item = gtk.MenuItem ('_Tools')
    item.show ()
    menubar.append (item)

    menu = gtk.Menu ()
    menu.show ()
    item.set_submenu (menu)

    item = gtk.MenuItem ('Extensions...')
    item.connect ("activate", self.on_extension_manager)
    item.show ()
    menu.append (item)

    item = gtk.MenuItem ('_Help')
    item.show ()
    menubar.append (item)

    menu = gtk.Menu ()
    menu.show ()
    item.set_submenu (menu)

    item = gtk.ImageMenuItem (gtk.STOCK_ABOUT, accel_group)
    item.connect ("activate", self.on_help_about)
    item.show ()
    menu.append (item)

    # toolbar
    self.tooltips = gtk.Tooltips ()

    handlebox = gtk.HandleBox ()
    handlebox.show ()
    vbox.pack_start (handlebox, False, False)

    toolbar = gtk.Toolbar ()
    toolbar.set_style (gtk.TOOLBAR_ICONS)
    toolbar.set_tooltips (True)
    toolbar.show ()
    handlebox.add (toolbar)

    toolitem = gtk.ToolButton (gtk.STOCK_NEW)
    toolitem.connect ("clicked", self.on_file_new)
    toolitem.show ()
    toolitem.set_tooltip (self.tooltips, "Start new case")
    toolbar.insert (toolitem, -1)

    toolitem = gtk.ToolButton (gtk.STOCK_OPEN)
    toolitem.connect ("clicked", self.on_file_open)
    toolitem.show ()
    toolitem.set_tooltip (self.tooltips, "Open an existing case")
    toolbar.insert (toolitem, -1)

    toolitem = gtk.SeparatorToolItem ()
    toolitem.show ()
    toolbar.insert (toolitem, -1)

    toolitem = gtk.ToolButton (gtk.STOCK_QUIT)
    toolitem.connect ("clicked", self.on_file_quit)
    toolitem.show ()
    toolitem.set_tooltip (self.tooltips, "Exit from Mobius")
    toolbar.insert (toolitem, -1)

    # toolbox
    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_NEVER, gtk.POLICY_AUTOMATIC)
    sw.show ()
    vbox.pack_start (sw)

    self.iconview = gtk.IconView (toolbox_model)
    self.iconview.set_pixbuf_column (1)
    self.iconview.set_text_column (2)
    self.iconview.set_selection_mode (gtk.SELECTION_NONE)
    self.iconview.connect ('item-activated', self.on_toolitem_activated)
    self.iconview.show ()
    sw.add (self.iconview)

    # status bar
    frame = gtk.Frame ()
    frame.set_shadow_type (gtk.SHADOW_IN)
    frame.show ()
    vbox.pack_end (frame, False, False)

    self.status_label = gtk.Label ()
    self.status_label.set_alignment (0, -1)
    self.status_label.show ()
    frame.add (self.status_label)

    # @deprecated
    self.manager = mobius.app.ui_manager

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief add case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def add_case (self, case):

    # create a new window for case
    window = mobius.ui.case_window.Window ()
    window.show ()
    self.manager.add_window (window)

    # configure window
    window.case = case
    window.update_title ()
    window.case_treeview.set_case (case)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief new case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_file_new (self, widget, *args):
    case = mobius.app.new_case ()
    self.add_case (case)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief open case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_file_open (self, widget, *args):

    # choose file
    fs = gtk.FileChooserDialog ('Open Case', parent=self)
    fs.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
    fs.add_button (gtk.STOCK_OK, gtk.RESPONSE_OK)

    filter = gtk.FileFilter ()
    filter.add_pattern ('*.case')
    fs.set_filter (filter)

    rc = fs.run ()
    filename = fs.get_filename ()
    fs.destroy ()

    if rc != gtk.RESPONSE_OK:
      return

    # populate case
    case = mobius.app.open_case (filename)
    self.add_case (case)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief quit application
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_file_quit (self, widget, *args):

    # if application is modified, show save/ignore/cancel dialog
    if mobius.app.is_modified ():
      dialog = gtk.MessageDialog (self,
                  gtk.DIALOG_MODAL,
                  gtk.MESSAGE_QUESTION,
                  gtk.BUTTONS_YES_NO,
                  "There are unsaved cases. Save them before quitting?")
      dialog.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
      rc = dialog.run ()
      dialog.destroy ()

      if rc == gtk.RESPONSE_CANCEL:
        return

      elif rc == gtk.RESPONSE_YES:
        mobius.app.save_all_cases ()

    # else show 'do you want to quit' dialog
    else:
      dialog = gtk.MessageDialog (self,
                  gtk.DIALOG_MODAL,
                  gtk.MESSAGE_QUESTION,
                  gtk.BUTTONS_YES_NO,
                  "Do you really want to quit?")

      rc = dialog.run ()
      dialog.destroy ()

      if rc != gtk.RESPONSE_YES:
        return True

    # stop UI
    mobius.app.ui_manager.stop ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief toolitem activated
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_toolitem_activated (self, widget, path, *args):
    model = widget.get_model ()
    iter = model.get_iter (path)
    id = model.get_value (iter, 0)
    callback = model.get_value (iter, 3)
    callback (id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief call extension manager dialog
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_extension_manager (self, widget, *args):
    dialog = mobius.ui.extension_manager_dialog.Dialog ()
    rc = dialog.run ()
    dialog.destroy ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief call 'about' dialog
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_help_about (self, widget, *args):
    dialog = mobius.ui.about_dialog.Dialog ()
    dialog.run ()
    dialog.destroy ()
