/*
 *   This file is part of the MLV Library.
 *
 *   Copyright (C) 2010 Adrien Boussicault, Marc Zipstein
 *
 *
 *    This Library is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This Library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this Library.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \file MLV_image.h
 *
 * \author Adrien Boussicault
 * \author Marc Zipstein
 *
 * \brief Ce fichier définit tous les prototypes concernant les fonctions
 * permettant d'insérer des images.
 */

#ifndef __MLV_IMAGE_H__
#define __MLV_IMAGE_H__

/**
 * \brief Définit le type Image dans la librairie MLV.
 */
typedef struct _MLV_Image MLV_Image;

/**
 * \brief Charge en mémoire une image contenue dans un fichier au format BMP.
 *
 * \param file_image Chemin d'accès du fichier au format BMP contenant l'image.
 * \return L'image chargée.
 */
MLV_Image* MLV_load_image_bmp( const char* file_image );

/**
 * \brief Libère la mémoire utilisée par l'image.
 *
 * \param image Image à supprimer de la mémoire
 */
void MLV_free_image( MLV_Image *image );

/**
 * \brief Copie une image.
 *
 * Cette fonction créé une nuovelle image qui est la copie conforme de l'image
 * passée en paramètre.
 * La gestion de la mémoire de l'image créée est laissée à l'utilisateur.
 *
 * \param image L'image à copier.
 * \return La copie de l'image.
 */
MLV_Image* MLV_copy_image( MLV_Image* image );

/**
 * \brief Retourne la taille d'une image donnée.
 *
 * \param image L'image pour laquelle la fonction doit retourner sa taille.
 * \param width La largeur de l'image.
 * \param height La hauteur de l'image.
 */
void MLV_get_size_image( MLV_Image* image, int* width, int* height );

/**
 * \brief Redimensionne l'image en ne repsectant pas les proportions de l'image originale.
 *
 * La hauteur et la largeur de la nouvelle image deviennent respectivement la 
 * hauteur et la largeur passées en paramètres.
 *
 * \param image Image à redimentionner.
 * \param width Largeur de l'image.
 * \param height Hauteur de l'image.
 */
void MLV_resize_image( MLV_Image* image, int width, int height );

/**
 * \brief Redimensionne l'image en repsectant les proportions de l'image originale.
 *
 * Cette fonction redimensionne l'image de telle sorte que les proportions 
 * d'origines soient respctées et que la hauteur et la largeur de la nouvelle
 * image soient à la fois maximales et à la fois inférieures ou égales à la hauteur et la largeur
 * données en paramètres.
 *
 * L'utilisateur peut entrer un entier négatif pour les champs width et height.
 * Ces champs ne sont alors pas pris en compte dans le redimensionnement de 
 * l'image.
 *
 * \param image Image à redimentionner.
 * \param width Nouvelle largeur de l'image.
 * \param height Nouvlle hauteur de l'image.
 */ 
void MLV_proportionnal_resize_image( MLV_Image* image, int width, int height );

/**
 * \brief Modifie l'image à l'aide d'une homotétie ( Etire l'image )
 *
 * \param image L'image à modifier
 * \param scalar Coefficient de proportionalité de l'homotétie ( >1 l'image est agrandie, <1 l'image est rétrécie ). 
 */
void MLV_scale_image( MLV_Image* image, double scalar );

/**
 * \brief Effectue une rotation sur une image donnée.
 *
 * \param image L'image à modifier
 * \param rotation L'angle de la rotation.
 */
void MLV_rotate_image( MLV_Image* image, double rotation );

/**
 * \brief Effectue une rotation suivie d'une homotétie
 *
 * \param image L'image à modifier
 * \param rotation L'angle de la rotation
 * \param scalar Le coéfficient de proportionnalité de l'homotétie.
 */
void MLV_rotate_scale_image( MLV_Image* image, double rotation, double scalar );

/**
 * \brief Étire l'image suivant l'axe X et l'axe Y avec des coéefficients de proportionnalité différents pour les deux axes.
 *
 * \param image L'image à modifier
 * \param scalar_x Le coéfficient de proportionnalité de l'axe X.
 * \param scalar_y Le coéfficient de proportionnalité de l'axe Y.
 */
void MLV_scale_xy_image( MLV_Image* image, double scalar_x, double scalar_y );

/**
 * \brief Effectue une rotation, puis étire l'image suivant l'axe X et l'axe Y avec des coéefficients de proportionnalité différents pour les deux axes.
 *
 * \param image L'image à modifier
 * \param rotation L'angle de la rotaiton
 * \param scalar_x Le coéfficient de proportionnalité de l'axe X.
 * \param scalar_y Le coéfficient de proportionnalité de l'axe Y.

 */
void MLV_rotate_scale_xy_image(
	MLV_Image* image, double rotation, double scalar_x, double scalar_y 
);

/**
 * \brief Dessine une image donnée à une position donnée de la fenêtre.
 *
 * \param image Image à déessiner.
 * \param x Coordonnée en X de la position dans la fenêtre de sommet Nord-Ouest de l'image
 * \param y Coordonnée en Y de la position dans la fenêtre du sommet Nord-Ouest de l'image
 */
void MLV_draw_image( MLV_Image *image, int x, int y );

/**
 * \brief Dessine une partie d'une image donnée à une position donnée sur la fenêtre.
 * \param image L'image à afficher sur la fenetre
 * \param x_source La coordonnée en X du sommet Nord-Ouest de la partie de l'image à afficher dans la fenêtre.
 * \param y_source La coordonnée en Y du sommet Nord-Ouest de la partie de l'image à afficher dans la fenêtre.
 * \param width_source La largeur de la partie de l'image à afficher dans la fenetre.
 * \param height_source La hauteur de la partie de l'image à afficher dans la fenetre.
 * \param x Coordonnée en X de la position dans la fenetre du sommet Nord-Ouest de la partie de l'image à afficher.
 * \param y Coordonnée en Y de la position dans la fenetre du sommet Nord-Ouest de la partie de l'image à afficher.
 */
void MLV_draw_partial_image(
	 MLV_Image *image, int x_source, int y_source, 
	int width_source, int height_source, 
	int x, int y
);


/* TODO  : A Vérifier !
void MLV_draw_rotated_image(  MLV_Image *image, int x, int y, double rotation );
void MLV_draw_scaled_image(  MLV_Image *image, int x, int y, double scalar );
void MLV_draw_scaled_rotated_image(  MLV_Image *image, int x, int y, double roation, double scalar );
*/

#endif
