/*
  upad - A program for debugging, and uploading code to embedded devices.
  Copyright (C) 2016, 2019 John Darrington

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include <config.h>
#include <stdarg.h>

#include "transaction.h"

#include "bdccsr.h"

#include "sep.h"
#include "misc.h"
#include "commands.h"
#include "link.h"
#include "lib/md5.h"
#include "registers.h"

#include "target.h"

static inline uint8_t *
sep_append_byte (uint8_t *data, uint8_t x)
{
  *data++ = x;
  return data;
}

static inline uint8_t *
sep_append_word (uint8_t *data, uint16_t x)
{
  memcpy (data, &x, sizeof (x));
  data += sizeof (x);
  return data;
}

static inline uint8_t *
sep_append_dword (uint8_t *data, uint32_t x)
{
  memcpy (data, &x, sizeof (x));
  data += sizeof (x);
  return data;
}

static uint8_t buffer[MAX_PAYLOAD_LENGTH];

static int execute_write_regs_v (int des, ...);

int
execute_write_memory_full (int des, tgt_addr addr, size_t n_bytes, uint8_t *tx_buf, const struct memsector *ms)
{
  tx_buf[0] = ms ? ms->cmd : CMD_WRITE_MEM;

  uint8_t *data = tx_buf + 1;
  struct write_mem_params wmp;

  wmp.quantity = n_bytes;
  wmp.address = addr;
  memcpy (data, &wmp, sizeof (wmp));

  sep_send (des, tx_buf, n_bytes + 1 + sizeof (wmp));
  int n = sep_recv (des, buffer, MAX_PAYLOAD_LENGTH);
  if (n < 1 || buffer[0] != CMD_ACK)
    return -1;
  return 0;
}

/*
   Write data to memory on the target.
   DES is the descriptor on which the target is connected.
   ADDR is the address to write.
   N_BYTES is the number of bytes to write.
   DATA is a pointer to the data to write.
   Returns zero on success.
*/
int
execute_write_memory (int des, tgt_addr addr, size_t n_bytes, const uint8_t *data)
{
  struct write_mem_params wmp;
  uint8_t *buf = alloca (n_bytes + 1 + sizeof (wmp));
  buf[0] = CMD_WRITE_MEM;

  wmp.quantity = n_bytes;
  wmp.address = addr;
  memcpy (buf + 1, &wmp, sizeof (wmp));

  memcpy (buf + 1 + sizeof (wmp), data, n_bytes);

  sep_send (des, buf, n_bytes + 1 + sizeof (wmp));
  int n = sep_recv (des, buffer, MAX_PAYLOAD_LENGTH);
  if (n < 1 || buffer[0] != CMD_ACK)
    return -1;
  return 0;
}


int
post_final (int des, const struct bounce_code_usage *bcu, uint32_t end)
{
  if (bcu == NULL)
    return 0;

  /* Wait for the end of any previous operation and check result */
  while ((execute_read_bdccsr (des) & BDCCSR_ACTIVE) == 0)
    ;
  const struct memsector *ms = get_memory_class (end - 1);
  if (ms == NULL)
    return -1;

  /* Check that the bounce code ran to the end of all the data.
     That is to say, its destination register should be pointing
     to one place past the last address written. */
  uint32_t y = execute_read_register (des, bcu->dest_register);
  if ( (y & ~(~0 << (registers [bcu->dest_register].size * 8))) !=
       (((unsigned long long int) end + (ms->write_sector - 1)) /
        ms->write_sector) * ms->write_sector)
    return -1;

  /* Check that the bounce code does not indicate any errors */
  uint32_t d0 = execute_read_register (des, bcu->err_register);
  if (d0 != 0)
    {
      return -1;
    }
  return 0;
}


int
write_fragment (int des, tgt_addr addr, uint8_t *tx_buf, size_t n_bytes,
                const struct bounce_code_usage *bcu, bool first)
{
  const struct memsector *ms = get_memory_class (addr);
  if (ms == NULL)
    return -1;

  if (bcu && ms->klass == MEMCLASS_PFLASH)
    {
      /* If we're writing to flash, then it's worthwhile bouncing through RAM */
      uint32_t intermediate_location = bcu->address + bcu->size + ms->write_sector;

      /* So first, write the fragment to RAM */
      int result = write_fragment (des, intermediate_location, tx_buf, n_bytes, NULL, first);
      if (result < 0)
	return result;

      /* Wait for the end of any previous operation and check result */
      while ((execute_read_bdccsr (des) & BDCCSR_ACTIVE) == 0)
	;
      if (!first)
	{
	  /* Read the execution status of the previous fragment.
	   That is to say,  did the previous fragment get bounced successfully? */
          uint32_t d0 = execute_read_register (des, 0);
          if (d0 != 0)
	    {
              printf ("FSTAT is 0x%08X\n", d0);
              return -1;
	    }
	}

      /* Set the necessary registers that the  bounce code needs,
	 including the PC */
      execute_write_regs_v (des,
			    bcu->size_register,   n_bytes,
			    bcu->source_register, intermediate_location,
			    bcu->dest_register,   addr,
			    bcu->pc_register,     bcu->address,
			    -1);

      /* And set it running. */
      execute_go (des);

      return 0;
    }

  if (addr & (ms->write_sector - 1))
    {
      upad_msg (MSG_WARNING, "Warning: Non aligned address: 0x%08x\n", addr);
    }

  if (0 > execute_write_memory_full (des, addr, n_bytes, tx_buf, ms))
    return 0;

  return n_bytes;
}


/*
   Calculate the MD5 sum of the data in the target's memory.
   DES is the descriptor on which the target is connected.
   ADDRESS is the starting address of the data.
   QUANTITY is the size of the data.
   RESULT is where the m5dsum will be placed.
   It is the caller's responsibility to ensure that RESULT points
   to at least  MD5_DIGEST_SIZE bytes of data.
   Returns zero on success.
*/
int
remote_md5_check (int des, tgt_addr address, uint32_t quantity, uint8_t *result)
{
  uint8_t *data = buffer;
  data = sep_append_byte (data, CMD_MD5_CHECK);
  data = sep_append_dword (data, address);
  data = sep_append_dword (data, quantity);

  sep_send (des, buffer, data - buffer);
  int n = sep_recv (des, buffer, MAX_PAYLOAD_LENGTH);
  if (n != 17)
    {
      upad_errno (MSG_ERROR, "Communication error");
      return 1;
    }
  if (buffer[0] != CMD_ACK)
    {
      upad_msg (MSG_ERROR, "command failed");
      return 1;
    }

  memcpy (result, buffer + 1, MD5_DIGEST_SIZE);
  return 0;
}


int
execute_echo (int des, int size, FILE *fpin, FILE *fpout)
{
  uint8_t *data = safe_realloc (NULL, size + 3);
  uint8_t *buf = data;
  data = sep_append_byte (data, CMD_ECHO);
  data = sep_append_word (data, size);

  int x = 0;
  while (!feof (fpin))
    {
      int r = fread (data + x, size - x, 1, fpin);
      if (r < 0)
	break;
      x += r;
    }
  fclose (fpin);

  sep_send (des, buf, size + 3);
  memset (buf, 0, size + 3);
  int n = sep_recv (des, buf, size + 3);

  if (n != size + 1)
    {
      upad_msg (MSG_ERROR, "Expected %d bytes.  Got %d bytes", size + 1, n);
      free (buf);
      return 1;
    }

  if (buf[0] != CMD_ACK)
    {
      upad_msg (MSG_ERROR, "command failed");
      free (buf);
      return 1;
    }

  int i;
  for (i = 0; i < size; ++i)
    {
      fputc (buf[1+i], fpout);
    }
  free (buf);
  return 0;
}


/*
   Write CSR to  the BDCCSR register on the target connected to
   DES.
   Returns zero on success.
*/
int
execute_write_bdccsr (int des, uint16_t csr)
{
  uint8_t *data = buffer;
  data = sep_append_byte (data, CMD_WRITE_BDCCSR);
  data = sep_append_word (data, csr);

  sep_send (des, buffer, data - buffer);
  int n = sep_recv (des, buffer, MAX_PAYLOAD_LENGTH);
  if (n < 1)
    {
      upad_errno (MSG_ERROR, "Communication error");
      return 1;
    }

  if (buffer[0] != CMD_ACK)
    {
      upad_msg (MSG_ERROR, "Bad response from target\n");
      return 1;
    }

  return 0;
}

/* Returns the BDCCSR of the target connected to DES.
   FIXME:  We cannot distinquish between 1 and failure!!
 */
uint16_t
execute_read_bdccsr (int des)
{
  buffer[0] = CMD_READ_BDCCSR;
  sep_send (des, buffer, 1);
  int n = sep_recv (des, buffer, MAX_PAYLOAD_LENGTH);
  if (n < 3)
    {
      upad_errno (MSG_ERROR, "Communication error");
      return 1;
    }

  if (buffer[0] != CMD_ACK)
    {
      upad_msg (MSG_ERROR, "Bad response from target\n");
      return 1;
    }

  return  *((uint16_t *)(buffer + 1));
}


/*
   Perform a hard reset of the target connected to DES.
   Returns zero on success.
 */
int
execute_reset (int des)
{
  buffer[0] = CMD_RESET;
  sep_send (des, buffer, 1);
  int n = sep_recv (des, buffer, MAX_PAYLOAD_LENGTH);
  if (n < 0)
    {
      upad_errno (MSG_ERROR, "Communication error");
      return 1;
    }
  return 0;
}


/*
   Commands the target connected on DES to leave active BDM.
   In other words, it sets the CPU running.
   Returns zero on success.
*/
int
execute_go (int des)
{
  buffer[0] = CMD_GO;
  sep_send (des, buffer, 1);
  int n = sep_recv (des, buffer, MAX_PAYLOAD_LENGTH);
  if (n < 0)
    {
      upad_errno (MSG_ERROR, "Communication error");
      return 1;
    }
  return 0;
}


/*
   Commands the target connected on DES to execute a single
   instruction at whereever the PC is pointing.
   Returns zero on success.
*/
int
execute_step (int des)
{
  buffer[0] = CMD_STEP;
  sep_send (des, buffer, 1);
  int n = sep_recv (des, buffer, MAX_PAYLOAD_LENGTH);
  if (n < 0)
    {
      upad_errno (MSG_ERROR, "Communication error");
      return 1;
    }
  return 0;
}


/*
   Commands the target connected to DES to enter active
   BDM.  In other words, halts the CPU.
   Returns zero on success.
*/
int
execute_background (int des)
{
  buffer[0] = CMD_BACKGROUND;
  sep_send (des, buffer, 1);
  int n = sep_recv (des, buffer, MAX_PAYLOAD_LENGTH);
  if (n < 0)
    {
      upad_errno (MSG_ERROR, "Communication error");
      return 1;
    }
  return 0;
}


/*
   Tests communication between the local machine and the programmer
   connected on DES.  This makes no attempt to communicate with the target.
   Returns zero on success.
*/
int
execute_poll (int des)
{
  buffer[0] = CMD_POLL;
  sep_send (des, buffer, 1);
  int n = sep_recv (des, buffer, MAX_PAYLOAD_LENGTH);
  if (n < 1)
    {
      upad_msg (MSG_ERROR, "No valid response from programmer.  Is it connected?\n");
      return 1;
    }

  if (0 != strncmp ((const char *) buffer + 1, PACKAGE_VERSION,
		    strlen (PACKAGE_VERSION)))
    {
      upad_msg (MSG_WARNING,
		"Warning: Controller and program are of different versions.  This could cause problems.\n");
    }
  return 0;
}

/* Turn on/off the autoconnect feature */
int
execute_autoconnect (int des, bool enabled)
{
  buffer[0] = CMD_AUTOCONNECT;
  buffer[1] = enabled;
  sep_send (des, buffer, 2);
  int n = sep_recv (des, buffer, MAX_PAYLOAD_LENGTH);
  if (n < 1)
    {
      upad_msg (MSG_ERROR, "No valid response from programmer.  Is it connected?\n");
      return 1;
    }
  return 0;
}


/*
   Reset the target into BDM mode.
   Only in BDM mode can we communicate with the target.
   Returns zero on success.
 */
int
execute_reset_bdm (int des)
{
  buffer[0] = CMD_BKDM;
  sep_send (des, buffer, 1);
  int n = sep_recv (des, buffer, MAX_PAYLOAD_LENGTH);
  if (n < 0)
    {
      upad_msg (MSG_FATAL, "No valid response from programmer.  Is it connected?\n");
      return 1;
    }
  return 0;
}


/*
   Perform a sync.
   Generally, this is necessary after a reset_bdm and before any other communication is
   attempted.
   Returns zero if successfull.
*/
int
execute_sync (int des)
{
  buffer[0] = CMD_SYNC;
  sep_send (des, buffer, 1);
  int n = sep_recv (des, buffer, MAX_PAYLOAD_LENGTH);
  if (n >= 5)
    {
      int *sync_time = (int *) (buffer + 1);
      upad_msg (MSG_NOTE, "BDC bit period is %gus\n", *sync_time / 8.0);
    }
  else if (n < 0)
    {
      upad_errno (MSG_ERROR, "Communication error");
      return 1;
    }
  else
    {
      upad_msg (MSG_FATAL, "The target device is not responding to the programmer.  Is the programmer connected to the debug port?\n");
      return 1;
    }
  return 0;
}


/*
   Performs a bulk erase of the memory of the target connected to DES.
   Returns zero on success.
*/
int
execute_bulk_erase (int des)
{
  buffer[0] = CMD_BULK_ERASE;
  sep_send (des, buffer, 1);
  int n = sep_recv (des, buffer, MAX_PAYLOAD_LENGTH);
  if (n < 3)
    {
      upad_errno (MSG_ERROR, "Communication error");
      return 1;
    }
  return 0;
}


/* Read QUANTITY bytes of memory from DES starting at ADDRESS.
   The caller must allocate BUF which must be at least QUANTITY
   bytes long.
   This function returns the number of bytes read.  */
int
execute_read_memory (int des, tgt_addr address, int quantity, uint8_t *buf)
{
  uint8_t *data = buffer;
  data = sep_append_byte (data, CMD_READ_MEM);
  data = sep_append_dword (data, address);
  data = sep_append_word (data, quantity);

  sep_send (des, buffer, data - buffer);
  int n = sep_recv (des, buffer, MAX_PAYLOAD_LENGTH);
  if (n < 1)
    {
      upad_errno (MSG_ERROR, "Communication error");
      return -1;
    }
  if (buffer[0] != CMD_ACK)
    {
      upad_msg (MSG_ERROR, "command failed");
      return -1;
    }

  for (int i = 0; i < n - 1; ++i)
    buf[i] = buffer[i+1];

  return n;
}


/*
   Reads register number REG on the target connected to DES.
 */
uint32_t
execute_read_register (int des, int reg)
{
  buffer[0] = CMD_READ_REG;
  buffer[1] = reg;
  sep_send (des, buffer, 2);
  int n = sep_recv (des, buffer, MAX_PAYLOAD_LENGTH);
  if (n < 5)
    {
      upad_errno (MSG_ERROR, "Communication error");
      return 1;
    }

  uint32_t reg_value = *((uint32_t *)(buffer + 1));

  return reg_value;
}

/*
   Reads all registers from the targeted connected on the descriptor
   DES.
   If this function succeeds, on completion N_REGS will contain the number
   of registers read and REG_VALUES will point to an array  containing
   the values of the registers.

   Returns zero on success.
*/
int
execute_read_all_regs (int des, uint32_t ** reg_values, size_t *n_regs)
{
  buffer[0] = CMD_READ_ALL_REGS;
  sep_send (des, buffer, 1);
  int n = sep_recv (des, buffer, MAX_PAYLOAD_LENGTH);
  if (n < 3)
    {
      upad_errno (MSG_ERROR, "Communication error");
      return 1;
    }

  *reg_values = (uint32_t *)(buffer + 1);

  *n_regs = (n - 1) / sizeof (uint32_t);

  return 0;
}

int
execute_write_regs (int des, int n_regs, const struct register_spec *rs)
{
  buffer[0] = CMD_WRITE_REGS;
  buffer[1] = n_regs;
  int i;
  for (i = 0; i < n_regs; ++i)
    {
      buffer[2 + i * 5] = rs[i].reg;
      sep_append_dword (buffer + 3 + i * 5, rs[i].value);
    }

  sep_send (des, buffer, 2 + n_regs * 5);

  int n = sep_recv (des, buffer, MAX_PAYLOAD_LENGTH);
  if (n < 1)
    {
      upad_errno (MSG_ERROR, "Communication error");
      return 1;
    }

  if (buffer[0] != CMD_ACK)
    {
      upad_msg (MSG_ERROR, "Bad response from target\n");
      return 1;
    }

  return 0;
}

/* Write the CPU registers.   DES is the descriptor of the connection.
   It must be followed by an even number of integers of the form
   [REGNAME0, VALUE0,   REGNAME1, VALUE1, ... REGNAMEn VALUEn]
   and then terminated with -1. */
static int
execute_write_regs_v (int des, ...)
{
  int n_regs = 0;

  struct register_spec *rs = NULL;

  va_list ap;
  va_start (ap, des);
  for (;;)
    {
      int reg = va_arg (ap, int);
      if (reg == -1)
	break;
      rs = safe_realloc (rs, sizeof (*rs) * (n_regs + 1));
      uint32_t val = va_arg (ap, uint32_t);
      rs[n_regs].value = val;
      rs[n_regs].reg = reg;
      n_regs++;
    };
  va_end (ap);


  return execute_write_regs (des, n_regs, rs);
}

/*
   Erase the sector of the target's memory which includes ADDRESS.
   Returns zero on success.
*/
int
execute_erase_sector (int des, tgt_addr address)
{
  const struct memsector *ms = get_memory_class (address);
  if (ms->klass == MEMCLASS_RAM)
    {
      /* Nothing to be done for RAM */
      return 0;
    }

  uint8_t *data = buffer;
  data = sep_append_byte (data, ms->cmd);
  data = sep_append_dword (data, address);

  sep_send (des, buffer, data - buffer);
  int n = sep_recv (des, buffer, MAX_PAYLOAD_LENGTH);
  if (n < 1)
    {
      upad_errno (MSG_ERROR, "Communication error");
      return 1;
    }
  if (buffer[0] != CMD_ACK)
    {
      upad_msg (MSG_ERROR, "command failed");
      return 1;
    }
  return 0;
}

/*
   Returns the sector information of the sector which contains ADDRESS.
*/
const struct memsector *
get_memory_class (tgt_addr address)
{
  int i;
  for (i = 0; i < n_memsectors; ++i)
    {
      const struct memsector *sec = memmap + i;
      if (address >= sec->address && address < sec->address + sec->size)
        {
          return sec;
        }
    }
  return NULL;
}

/* Local Variables:  */
/* mode: c           */
/* c-style: "gnu"    */
/* End:              */
