/*
  upad - A program for debugging, and uploading code to embedded devices.
  Copyright (C) 2016, 2019 John Darrington

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include <config.h>

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include "srecord-uploader.h"
#include <stdarg.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <termios.h>
#include <unistd.h>

#include "bdccsr.h"
#include "binary-uploader.h"
#include "elf-uploader.h"
#include "main-loop.h"
#include "misc.h"
#include "options.h"
#include "sep.h"
#include "transaction.h"
#include "upad.h"
#include "uploader.h"
#include "user-interface.h"

static void
confidence_message (FILE *fp, const char *format, ...)
{
  if (!isatty (fileno (fp)))
    return;

  fputs ("\r", fp);
  va_list ap;
  va_start (ap, format);
  vfprintf (fp, format, ap);
  va_end (ap);
  //  fputs ("\n", fp);
  fflush (fp);
}

int verbosity = MSG_INFORMATION;

void
upad_msg (enum msglevel lvl, const char *fmt, ...)
{
  if (lvl > verbosity)
    return;
  va_list ap;
  va_start (ap, fmt);
  vprintf (fmt, ap);
  va_end (ap);
}

void
upad_errno (enum msglevel lvl, const char *fmt, ...)
{
#define BUFMAX 512
  char buf[BUFMAX];
  char *err = strerror (errno);
  va_list ap;
  va_start (ap, fmt);
  vsnprintf (buf, BUFMAX, fmt, ap);
  va_end (ap);
  upad_msg (lvl, "%s: %s\n", buf, err);
}

/* Search for a device whose name matches TEMPL.
   TEMPL is a printf format string containing a single %d.
   If found, returns a pointer to NAME.
   If no match is found, returns NULL. */
static char *
search_serial_device (const char *templ)
{
  int d;
  static char name[64];
  for (d = 0; d < 8; ++d)
    {
      struct stat sb;
      snprintf (name, 64, templ, d);
      if (0 != stat (name, &sb))
	{
          int err = errno;
          if (err != ENOENT && err != ENXIO)
	    {
              upad_msg (MSG_ERROR,
			"%s exists but cannot be statted: %s\n",
			name, strerror (err));
	    }
	}
      else if (sb.st_rdev != 0)
	{
          return name;
	}
    }
  return NULL;
}

/* Upload the format UF using the interface DES */
int
upload_format (int des, struct upload_format *uf, const struct bounce_code_usage *bcu)
{
  tgt_addr address;
  uint8_t buffer[MAX_PAYLOAD_LENGTH * 2]; // FIXME: Buffer overflow ....
  size_t n_data;
  size_t total_data = 0;
  tgt_addr start_address = -1;
  bool first = true;
  size_t data_offset =  1 + sizeof (struct write_mem_params);
  if (uf == NULL)
    return 1;
  while (uf->get_data_fragment (uf, &address, buffer + data_offset + total_data, MAX_PAYLOAD_LENGTH - 8, &n_data))
    {
      if (first)
	start_address = address;
      if (start_address + total_data != address || total_data + n_data > MAX_PAYLOAD_LENGTH - 8)
	{
	  confidence_message (stdout, "Start address: 0x%08x", start_address);
	  if (0 > write_fragment (des, start_address, buffer, total_data, bcu, first))
	    {
	      confidence_message (stdout, "\n");
	      upad_msg (MSG_ERROR, "Error programming target.\n");
	      return 1;
	    }
	  memcpy (buffer + data_offset,
		  buffer + data_offset + total_data, n_data);
	  start_address = address;
	  total_data = 0;
	}
      total_data += n_data;
      first = false;
    }
  confidence_message (stdout, "Start address: 0x%08x", start_address);
  if (0 > write_fragment (des, start_address, buffer, total_data, bcu, first))
    {
      confidence_message (stdout, "\n");
      upad_msg (MSG_ERROR, "Error programming target.\n");
      return 1;
    }
  confidence_message (stdout, "                                                \r");

  return post_final (des, bcu, start_address + total_data);
}




/* Attempt to open the serial port called PORT, initialise it according
   to our needs and return a file descriptor to the opened port.
   On error -1 is returned. */
static int
initialise_serial_port (const char *port)
{
  int des = open (port, O_RDWR);
  if (des < 0)
    {
      upad_errno (MSG_ERROR, "Cannot open serial port `%s'", port);
      return -1;
    }

  struct termios termios;
  if (0 != cfsetspeed (&termios, B115200))
    {
      upad_msg (MSG_ERROR, "Baud rate of 115200 seems not to be supported by this system\n");
    }
  cfmakeraw (&termios);
  if (0 != tcsetattr (des, TCSANOW, &termios))
    {
      upad_errno (MSG_ERROR, "Cannot set serial port attributes of %s", port);
      return -1;
    }

  /* Discard any rubbish which might be in the buffer */
  tcflush (des, TCIOFLUSH);

  return des;
}


/* Process the command string CMD.  */
bool
process_command (const char *cmd, int des)
{
  bool result = false;
  char *str = strdup (cmd);

  char *semicolon_r;
  char *stmt = strtok_r (str, ";", &semicolon_r);
  do
    {
      char *space_r;
      char *s = strtok_r (stmt, " ", &space_r);
      if (s == NULL)
        {
	  goto done;
        }
      result = false;
      int c;
      for (c = 0; c < N_COMMANDS; ++c)
        {
	  if (0 == strcmp (s, cmds[c].name))
            {
	      size_t arg_capacity = 4;
	      char **args = safe_realloc (NULL, sizeof (*args) * arg_capacity);
	      int arg = 0;
	      while ((args[arg] = strtok_r (NULL, " ", &space_r)))
                {
		  if (args[arg] == NULL)
		    break;

		  arg++;
		  if (arg >= arg_capacity)
                    {
		      arg_capacity *= 2;
		      args = safe_realloc (args, sizeof (*args) * arg_capacity);
                    }
                }
	      result = (0 == cmds[c].call (des, arg, args));
	      free (args);
            }
        }
    }
  while ((stmt = strtok_r (NULL, ";", &semicolon_r)));

 done:
  free (str);
  return result;
}


/* Open a file called SCRIPT, and process it as a list of upad commands. */
static int
run_script (int des, const char *script)
{
  int ret = 0;
  FILE *sfp = fopen (script, "r");
  if (sfp == NULL)
    {
      upad_errno (MSG_FATAL, "Cannot open file \"%s\"", script);
      return 1;
    }
  char *lineptr = NULL;
  size_t line_len = 0;
  while (!feof (sfp))
    {
      int n = getline (&lineptr, &line_len, sfp);
      if (n <= 0)
	continue;
      /* Discard the trailing delimiter */
      lineptr[n - 1] = '\0';
      if (!process_command (lineptr, des))
	{
	  upad_msg (MSG_ERROR, "Command \"%s\"failed\n", lineptr);
          ret = 1;
	  break;
	}
    }
  if (sfp)
    fclose (sfp);
  return ret;
}


enum ftype
  {
    FTYPE_SREC,
    FTYPE_BINARY,
    FTYPE_ELF
  };


/* Return true iff we believe that the file associated with the stream FP
   is an ELF file.  FP must be open for reading and located at the start
   of the file.   When this function returns it will rewind FP to the start
   of the file.  */
static bool
is_elf (FILE *fp)
{
  bool ret = false;
  uint8_t buf[4];
  int n = 0;
  while (!feof (fp))
    {
      n += fread (buf + n, 1, 4, fp);
      if (n >= 4)
	break;
    }

  if (buf[0] == 0x7F &&
      buf[1] == 0x45 &&
      buf[2] == 0x4c &&
      buf[3] == 0x46)
    {
      ret = true;
    }
  fseeko (fp, 0, SEEK_SET);
  return ret;
}


/* Attempts to autoidentify the format of the file associated with FP.
   FP must be a stream opened for reading and positioned at the start
   of the file.
   This function will always succeed.  If the file format cannot be recognised,
   then FTYPE_BINARY will be returned.
*/
enum ftype
identify_format (FILE *fp)
{
  char *lineptr = NULL;
  size_t size = 0;

  /* If we cannot recognise what it is, then default to BINARY */
  enum ftype ret = FTYPE_BINARY;

  /* Is it an elf file? */
  if (is_elf (fp))
    {
      ret = FTYPE_ELF;
      goto end;
    }

  /* A very short file can't be an S-record */
  if (getline (&lineptr, &size, fp) < 6)
    goto end;

  if (lineptr[0] != 'S')
    goto end;

  if (lineptr[1] < '0' || lineptr[1] > '9')
    goto end;

  ret = FTYPE_SREC;

 end:
  free (lineptr);
  fseeko (fp, 0, SEEK_SET);
  return ret;
}


/* Perform a complete bulk erase of the target.  */
static int
bulk_erase (int des)
{
  if (0 != execute_poll (des))
    return 1;

  if (0 != execute_reset_bdm (des))
    return 1;

  if (0 != execute_sync (des))
    return 1;

  if (0 != execute_bulk_erase (des))
    return 1;

  return 0;
}


/* Upload a file to the target.  */
static int
run_upload (int des, const struct user_options *opts)
{
  struct bounce_code_usage bcu;
  int ret = 0;

  FILE *fp = fopen (opts->file, "r");
  if (fp == NULL)
    {
      upad_errno (MSG_ERROR, "Cannot open file `%s'", opts->file);
      return 1;
    }

  execute_poll (des);

  FILE *fpb = NULL;
  if (opts->bounce_code)
    {
      fpb = fopen (opts->bounce_code, "r");
      if (fpb == NULL)
	{
	  upad_errno (MSG_ERROR,
		      "Cannot open bounce code file `%s'. Falling back to direct programming",
		      opts->bounce_code);
	}
      else
	{
	  fseek (fpb, 0, SEEK_END);
	  bcu.address = 0x1000;
	  bcu.size = ftell (fpb);
	  fseek (fpb, 0, SEEK_SET);
	  bcu.size_register = 7;    // D7
	  bcu.source_register = 8;  // X
	  bcu.dest_register = 9;    // Y
	  bcu.pc_register = 11;     // PC
	  bcu.err_register = 0;     // D0
	}
    }

  /* Prepare the microprocessor for upload of a new file.  */
  uint16_t csr = 1;
  do
    {
      execute_reset_bdm (des);
      execute_sync (des);
      if (0 != execute_bulk_erase (des))
	return 1;

      csr = execute_read_bdccsr (des);
    }
  while (csr & BDCCSR_ILLCMD);


  /* If appropriate, upload the bounce code.  */
  if (fpb)
    {
      struct upload_format *uf = create_binary_uploader (fpb, bcu.address);
      int ret = upload_format (des, uf, NULL);
      uf->destroy (uf);
      if (ret != 0)
	upad_msg (MSG_ERROR, "Upload of bounce code failed\n");
    }

  /* For binary uploads, an address must be explicitly specified, since there
     is no other way to know the destination.  */
  enum ftype ft = opts->binary ? FTYPE_BINARY : identify_format (fp);
  if (ft == FTYPE_BINARY && !opts->binary)
    {
      upad_msg (MSG_ERROR, "A binary format was detected, but no address for upload was specified.\n");
      return 1;
    }

  /* Now perform the upload.  */
  struct upload_format *uf = NULL;
  switch (ft)
    {
    default:
      uf = create_binary_uploader (fp, opts->load_address);
      break;
    case FTYPE_ELF:
      uf = create_elf_uploader (fp);
      break;
    case FTYPE_SREC:
      uf = create_srec_uploader (fp);
      break;
    }
  ret = upload_format (des, uf, fpb ? &bcu : NULL);
  uf->destroy (uf);
  fclose (fp);

  if (fpb)
    fclose (fpb);

  if (! opts->no_reset)
    execute_reset (des);

  return ret;
}


/* Stop a backgrounded gdb server */
static void
stop_gdb_server (const struct user_options *opts)
{
  FILE *fp = fopen (opts->control_file, "r");
  if (fp == NULL)
    {
      upad_msg (MSG_ERROR, "Cannot open file %s",
		opts->control_file);
      exit (1);
    }
  pid_t pid;
  if (1 != fscanf (fp, "pid: %d", &pid))
    {
      upad_msg (MSG_ERROR, "Control file %s corrupt",
		opts->control_file);
      exit (1);
    }
  fclose (fp);
  if (0 != kill (pid, SIGTERM))
    {
      upad_errno (MSG_ERROR, "Terminate failed");
      exit (1);
    }
}

void
canonicalise_control_file (struct user_options *opts)
{
  /* If the control file is the name of an existant directory,
     then append a terminating "/ctrl" to it.  */
  struct stat statbuf;
  if (0 == stat (opts->control_file, &statbuf))
    {
      if (statbuf.st_mode & S_IFDIR)
        {
          char *port = safe_realloc (0, strlen (opts->control_file) + 6);
          strcpy (port, opts->control_file);
          strcat (port, "/port");

          char *cf = safe_realloc (0, strlen (opts->control_file) + 6);
          strcpy (cf, opts->control_file);
          strcat (cf, "/ctrl");
          free (opts->control_file);
          opts->control_file = cf;

          /* If no port has been specified then use a local socket
             in the directory. */
          if (opts->gdb_port == NULL)
            {
              opts->gdb_port = port;
            }
          else
            free (port);
        }
    }
}


int
main (int argc, char **argv)
{
  struct user_options opts = {0};
  parse_options (argc, argv, &opts);

  if (opts.stop_server)
    {
      canonicalise_control_file (&opts);
      stop_gdb_server (&opts);
      return 0;
    }

  /* If a port wasn't specified in the command line, then
     search for a suitable default. */
  if (opts.port == NULL)
    {
      opts.port = search_serial_device ("/dev/ttyACM%d");
      if (opts.port == NULL)
        opts.port = search_serial_device ("/dev/ttyS%d");
      if (opts.port == NULL)
	{
          fprintf (stderr,
                   "No serial port was specified and no suitable candidate could be found\n");
          return 1;
	}
    }

  int des = initialise_serial_port (opts.port);
  if (des < 0)
    return 1;

  int ret = 0;
  if (opts.script)
    ret = run_script (des, opts.script);
  else if (opts.file)
    ret = run_upload (des, &opts);
  else if (opts.erase_only)
    ret = bulk_erase (des);
  else
    ret = main_loop (des, &opts);

  if (des >= 0)
    close (des);

  return ret;
}

/* Local Variables:  */
/* mode: c           */
/* c-style: "gnu"    */
/* End:              */
