/*
  upad - A program for debugging, and uploading code to embedded devices.
  Copyright (C) 2016, 2019 John Darrington

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* This module implements a GDB server. */

#include <config.h>

#include "gdb-interface.h"
#include "gdb-cmds.h"

#include <assert.h>
#include <stdbool.h>
#include <stdint.h>
#include <string.h>
#include <stddef.h>
#include <stdio.h>
#include <unistd.h>

#include <sys/socket.h>
#ifdef HAVE_SYS_UN_H
#include <sys/un.h>
#endif
#include <netinet/in.h>
#include <arpa/inet.h>

#include "cfm.h"
#include "misc.h"

#include "registers.h"
#include "target.h"
#include "transaction.h"

#include "bdccsr.h"
#include "target/s12z/cpmu.h"
#include "breakpoints.h"

struct gdb_svr
{
  /* The descriptor of the socket on which the server listens.  */
  int sock_fd;

  /* The name of the socket.  Only relevant for Unix domain sockets.  */
  char *sock_name;

  /* The port on which the server listens.  Not relevant for Unix domain sockets.  */
  int portno;
};

int
gdb_svr_conn_fd (const struct gdb_svr *svr)
{
  return svr->sock_fd;
}

/* This flag is set after the QStartNoAckMode packet is received.
   It means that we no longer need to respond with + to every packet.  */
static bool no_ack = false;

/* Convert the 4 lower bits of C to a ascii encoded hexadecimal number.  */
static char
char_to_hex (char c)
{
  assert (0 == (c & 0xF0));
  if (c > 9)
    return c - 10 + 'A';
  return c + '0';
}



static bool send_reply (int fd, const char *cmd);


void
gdb_send_interrupt (int conn, int dev)
{
  disarm_breakpoints (dev);
  send_reply (conn, "T05hwbreak:;");
}


/* Send an empty message GDB */
bool
gdb_send_ping (int conn)
{
  return send_reply (conn, "O00");  /* Note this string is [upper-case-O zero zero] */
}


void
gdb_send_broken (int conn, int dev)
{
  send_reply (conn, "X00");
}


void
gdb_send_stopped (int conn, int dev)
{
  send_reply (conn, "S00");
}

#ifdef HAVE_SYS_UN_H
#ifndef UNIX_PATH_MAX
#define UNIX_PATH_MAX sizeof(((struct sockaddr_un *) NULL)->sun_path)
#endif
#endif

/* Convert S which must be a null terminated string, to an ascii encoded
   hexidecimal representation.
   Store the result in HEX_DATA.
   It is the caller's responsibility to ensure that HEX_DATA is large enough
   to hold the result.  That is, it should be at least 2 * strlen(S)
*/
static void
string_to_hex (char *hex_data, const char *s)
{
  while (*s)
    {
      hex_data[0] = char_to_hex (*s >> 4);
      hex_data[1] = char_to_hex (*s & 0x0F);
      hex_data += 2;
      s++;
    }
}

/* Convert the character C, which must be in the set [0-9a-fA-F]
   to the datum which the character represents.  */
static uint8_t
hex_to_datum (char c)
{
  if (c >= '0' && c <= '9')
    return c - '0';

  if (c >= 'A' && c <= 'F')
    return c - 'A';

  if (c >= 'a' && c <= 'f')
    return c - 'a';

  assert (false);
}

/*  Convert the string S which should be a hexadecimal ascii encoded
    string into the data it represents.
    It is the caller's responsibility ensure that DATA is large enough
    to hold the result.
    That is to say it must be at least strlen (S) / 2 .
*/
static void
hexstring_to_data (uint8_t *data, const char *s)
{
  int x = 0;
  while (*s)
    {
      data[x]  = hex_to_datum (s[0]);
      data[x] <<= 4;
      data[x] |= hex_to_datum (s[1]);
      s += 2;
      x++;
    }
}


#define TX_BUF_SIZE 1024

/* Send CMD to the GDB client connected to FD.
   Returns TRUE if successful.
*/
static bool
send_reply (int fd, const char *cmd)
{
  uint8_t checksum = 0;
  char buffer[TX_BUF_SIZE];
  memset (buffer, 0, TX_BUF_SIZE);
  buffer[0] = '$';
  int i;
  size_t cmd_len = strlen (cmd);
  for (i = 0; i < cmd_len; ++i)
    {
      buffer[i+1] = cmd[i];
      checksum += cmd[i];
    }
  buffer[i + 1] = '#';
  snprintf (buffer + i + 2, 3, "%02x", checksum);

  int n = write (fd, buffer, cmd_len + 4);
  if  (n < 0)
    {
      upad_errno (MSG_ERROR, "ERROR writing to socket");
      return false;
    }

  return true;
}

#define RX_BUF_SIZE 1024

/* Read a command from the GDB client connected to FD.
   Store the command in CMD.  CMD_LEN gets the length of
   CMD  */
static bool
receive_command (int fd, char **cmd, size_t *cmd_len)
{
  static char buffer[RX_BUF_SIZE];

  bzero (buffer, RX_BUF_SIZE);
  int idx = 0;

  uint8_t checksum = 0;
  /* Read the packet's payload, updating the checksum as we go. */
  while (idx < RX_BUF_SIZE)
    {
      int n = read (fd, buffer + idx, 1);
      if  (n < 0)
        {
	  upad_errno (MSG_ERROR, "ERROR reading from socket");
	  exit (1);
        }
      if (buffer[idx] == '#')
	break;
      checksum += buffer[idx];
      idx++;
    }

  /* Now read the provided checksum */
  char check_digits[3];
  bzero (check_digits, 3);
  int n = read (fd, check_digits, 2);
  if  (n < 2)
    {
      upad_errno (MSG_ERROR, "ERROR reading from socket");
      exit (1);
    }

  long xx = strtol (check_digits, NULL, 16);

  /* ... and make sure it's correct.  */
  if (xx != checksum)
    {
      int n = write (fd, "-", 1);
      if  (n < 1)
        {
	  upad_errno (MSG_ERROR, "ERROR writing to socket");
	  exit (1);
        }
      return false;
    }

  if (! no_ack )
    {
      n = write (fd, "+", 1);
      if  (n < 1)
        {
	  upad_errno (MSG_ERROR, "ERROR writing to socket");
	  exit (1);
        }
    }

  *cmd = buffer;
  *cmd_len = idx;

  return true;
}

/*
  Returns the number of characters in the string at X describing
  the name of a general query feature.
*/
static int
feature_span (const char *x)
{
  const char *s = 0;
  for (s = x; *s != 0; ++s)
    {
      if (*s == '+' || *s == '=')
	break;
      if (*s == '-')
	if (s[1] == ';' || s[1] == '\0')
	  break;
    }
  return s - x;
}

/* Process a qSupported command sent by the GDB client
   on FD.  */
void
gdb_supported (const char *ccmd, int fd, int dev)
{
  char *cmd = strdup (ccmd);
  char send_buffer[512];
  /* These features we support.  */
  snprintf (send_buffer, 512,
	    "PacketSize=0x%x;QStartNoAckMode+;QNonStop+;qXfer:memory-map:read+;",
            RX_BUF_SIZE);

  /* Any others which the client asks about, we reply in the negative.   */
  strtok (cmd, ":"); /* This is a colon.  NOT a semicolon!  */
  char *x = NULL;
  while ((x = strtok (NULL, ";")))
    {
      char name[64];
      size_t len = feature_span (x);
      bzero (name, 64);
      strncpy (name, x, len);
      strcat (send_buffer, name);
      if (0 == strcmp (name, "hwbreak"))
        strcat (send_buffer, "+;");
      else
        strcat (send_buffer, "-;");
    }

  send_reply (fd, send_buffer);
  free (cmd);
}

/* Shutdown a server and free its resources.  */
void
destroy_gdb_server (struct gdb_svr *svr)
{
  if (NULL == svr)
    return;
  close (svr->sock_fd);
  shutdown (svr->sock_fd, SHUT_RDWR);
  if (svr->portno == -1)
      unlink (svr->sock_name);
  free (svr->sock_name);
  free (svr);
}


/* Create an instance of a GBD server and start it listening on
   CONNECTION_POINT.
   Returns the descriptor of the listening socket.  */
struct gdb_svr *
create_gdb_server (const char *connection_point)
{
  struct sockaddr_in inet_addr;
  struct sockaddr *addr;
  size_t addr_size;
  int portno = -1;

  assert (connection_point != NULL);

  if (connection_point[0] >= '0' && connection_point[0] <= '9')
      portno = strtol (connection_point, 0, 10);

  int domain = AF_INET;
  if (portno == -1)
    {
#ifdef HAVE_SYS_UN_H
      struct sockaddr_un unix_addr;
      /* Unix domain socket */
      if (strlen (connection_point) > UNIX_PATH_MAX - 1)
        {
          upad_msg (MSG_ERROR,
                      "Socket name is too long.  It may be no longer than %zu bytes.\n",
                      UNIX_PATH_MAX - 1L);
          return NULL;
        }
      memset (&unix_addr, 0, sizeof (unix_addr));
      unix_addr.sun_family = AF_UNIX;
      strncpy (unix_addr.sun_path, connection_point, UNIX_PATH_MAX);

      domain = AF_UNIX;
      addr = (struct sockaddr *) &unix_addr;
      addr_size = sizeof (unix_addr);
#else
      upad_msg (MSG_ERROR,
                "Unix domain sockets are not built into this version of uPad.\n");
      return NULL;
#endif
    }
  else
    {
      /* Internet domain socket */
      memset (&inet_addr, 0, sizeof (inet_addr));
      inet_addr.sin_family = AF_INET;
      inet_addr.sin_addr.s_addr = INADDR_ANY;
      inet_addr.sin_port = htons (portno);

      domain = AF_INET;
      addr = (struct sockaddr *) &inet_addr;
      addr_size = sizeof (inet_addr);
    }

  int sockfd = socket (domain,  SOCK_STREAM,  0);
  if  (sockfd < 0)
    {
      upad_errno (MSG_ERROR, "ERROR opening socket");
      return NULL;
    }

  if (bind (sockfd, addr, addr_size) < 0)
    {
      upad_errno (MSG_ERROR, "ERROR on binding");
      return NULL;
    }

  if (0 != listen (sockfd, 5))
    {
      upad_errno (MSG_ERROR, "ERROR on listen");
      return NULL;
    }

  if (portno == -1)
    upad_msg (MSG_DIAGNOSTIC, "Listening on socket %s.\n", connection_point);
  else
    upad_msg (MSG_DIAGNOSTIC, "Listening on port %d.\n", portno);


  /* Create a structure holding the particulars of this
     server.  */
  struct gdb_svr *srv = safe_realloc (0, sizeof *srv);
  srv->sock_fd = sockfd;
  srv->sock_name = connection_point ? strdup (connection_point) : 0;
  srv->portno = portno;

  return srv;
}


/*  Look up the target's restart vector, point the program counter
    to that address and set it running.
    Note, that this means we remain in Active BDM mode.
*/
void
gdb_restart (const char *cmd, int conn, int dev)
{
  uint8_t rvec[4];
  cfm_read_memory (dev, RESET_VECTOR, 4, rvec);

  struct register_spec pc;
  pc.value = rvec[0] << 24;
  pc.value |= rvec[1] << 16;
  pc.value |= rvec[2] << 8;
  pc.value |= rvec[3];
  pc.reg = 11;

  execute_write_regs (dev, 1, &pc);
  /* R should have no reply */
}

/*
  A flag indicating whether the code is running.
  This flag is true IFF the CPU is NOT in *active* background debug mode.
*/
bool running = false;

/* Append an XML fragment to B.
   The fragment represents a memory region of TYPE, located in the range
   [START, START+LENGTH] and with ERASE_SIZE.
   REST is the maximum size of B.
   Returns the number of characters written to B.  */
static int
xml_append_region (char *b, int rest, const char *type, tgt_addr start, uint32_t length, uint32_t erase_size)
{
  char *a = b;
  int n = snprintf (b, rest, "  <memory type=\"%s\" start=\"0x%08x\" length=\"0x%08x\">\n",
                    type, start, length);
  rest -= n;
  b += n;
  n = snprintf (b, rest, "  <property name=\"blocksize\">0x%08x</property>\n", erase_size);
  rest -= n;
  b += n;
  n = snprintf (b, rest, "  </memory>\n");
  rest -= n;
  b += n;

  return b - a;
}

/* Generate a XML for use as a reply to a qXfer:memory-map:read packet.  */
static const char *
gen_memmap_xml (void)
{
#define XMLSIZE 1024
  static char xml[XMLSIZE];
  const char top[]=
    /* There is an 'l' prefixed to this XML to satisfy the GDB protocol.
       v  */
    "l<?xml version=\"1.0\"?>"
    "<!DOCTYPE memory-map"
    " PUBLIC \"+//IDN gnu.org//DTD GDB Memory Map V1.0//EN\""
    " \"http://sourceware.org/gdb/gdb-memory-map.dtd\">\n"
    "<memory-map>\n";

  const char bottom[]="</memory-map>\n";

  char *b = xml;
  int rest = XMLSIZE;
  int n = snprintf (b, rest, "%s", top);
  rest -= n;
  b += n;

  for (int s = 0; s < n_memsectors; ++s)
    {
      const char *type = NULL;
      const struct memsector *ms = memmap + s;
      switch (ms->klass)
        {
        case MEMCLASS_RAM:
        case MEMCLASS_OTHER:
          type = "ram";
          break;
        case MEMCLASS_PFLASH:
          type = "rom";
          break;
        case MEMCLASS_EEPROM:
          type = "flash";
          break;
        }
      n = xml_append_region (b, rest, type, ms->address, ms->size, ms->erase_sector);
      rest -= n;
      b += n;
    }

  n = snprintf (b, rest, "%s", bottom);

  assert (rest >= 0);
  rest -= n;
  b += n;

  return xml;
}

/* This flag is FALSE if we are continuing with an already running program.
   If we have restarted the target, then it will be TRUE.  */
static bool restarted = false;

/*  Called immediately after successfull connection of a GDB client.  */
void
gdb_connection_initialize (int conn, int dev)
{
  restarted = false;
  /*   Here, we try to avoid resetting the target.
       This is because sometimes the user wants to start a debug session from
       an already running program.
       However, if we're not in BDM mode, we have no choice.  We must reset the
       system into BDM mode before we can do anything.
  */
  int res = execute_sync (dev);
  if (res != 0)
    {
      execute_reset_bdm (dev);
      restarted = true;
    }

  uint16_t csr = execute_read_bdccsr (dev);
  upad_msg (MSG_DIAGNOSTIC,
	    "Initialising communication with target.  BDCCSR: %04X\n", csr);
  if (!(csr & BDCCSR_ACTIVE))
    {
      execute_reset_bdm (dev);
      execute_sync (dev);
      gdb_restart (0, conn, dev);
      restarted = true;
    }

  /* Disable the watchdog whilst in BDM mode */
  uint8_t cop;
  execute_read_memory  (dev, CPMU_BASE + offsetof (CPMU_TypeDef, COP), 1, &cop);
  cop |= 0x40;
  execute_write_memory (dev, CPMU_BASE + offsetof (CPMU_TypeDef, COP), 1, &cop);

  arm_breakpoints (dev);
  execute_go (dev);

  running = true;
}

/*  Called when there is any activity on the socket.   It
    accepts a connection and returns the descriptor to the open port.  */
int
accept_gdb_connection (struct gdb_svr *svr)
{
  struct sockaddr_in cli_addr;
  socklen_t clilen = sizeof (cli_addr);

  /* Accept connection from the client */
  int conn = accept (svr->sock_fd, (struct sockaddr *) &cli_addr, &clilen);
  if  (conn < 0)
    {
      upad_errno (MSG_ERROR, "ERROR on accept");
      exit (1);
    }

  upad_msg (MSG_DIAGNOSTIC, "Remote debugging from host %s\n",
	    inet_ntoa (cli_addr.sin_addr));

  no_ack = false;

  return conn;
}

/* Undoes the effect of accepting a connection.   */
void
close_gdb_connection (int conn)
{
  close (conn);
  upad_msg (MSG_DIAGNOSTIC, "Closed remote debug\n");
}



/* Sent by GDB shortly before it detaches.  */
void
gdb_detach (const char *cmd, int conn, int dev)
{
  /* Client is detaching */
  send_reply (conn, "OK");
  upad_msg (MSG_DIAGNOSTIC, "GDB client announced intention to detach\n");
}


/* Writes all registers in one operation.  */
void
gdb_write_registers (const char *cmd, int conn, int dev)
{
  int offs = 0;
  for (int reg = 0; reg < n_registers; ++reg)
    {
      char thisreg[9];
      int size = registers[reg].size;
      const char *start = cmd + 1 + offs;
      memset (&thisreg, 0, 9);
      strncpy (thisreg, start, size * 2);
      uint32_t val = strtol (thisreg, 0, 16);
      offs += size * 2;
      struct register_spec r;
      r.reg = reg;
      r.value = val;
      execute_write_regs (dev, 1, &r);
    }
  send_reply (conn, "OK");
}

void
gdb_kill (const char *cmd, int conn, int dev)
{
  execute_reset_bdm (dev);
  execute_sync (dev);
  gdb_restart (cmd, conn, dev);
  /* k should have no reply */
}

/* Writes a location in memory on the target.  */
void
gdb_write_memory (const char *cmd, int conn, int dev)
{
  /* Write data to a memory address */
  tgt_addr address = 0;
  unsigned int length = 0;
  char *str = NULL;
  int n = sscanf (cmd, "M" UTGT_ADDR ",%x", &address, &length);
  if (n != 2)
    {
      send_reply (conn, "E00");
      return;
    }
  str = alloca (length * 2 + 1);
  n = sscanf (cmd, "M%*x,%*u:%[0-9A-Za-z]", str);
  if (n != 1)
    {
      send_reply (conn, "E01");
      return;
    }
  uint8_t *data = alloca (length);
  hexstring_to_data (data, str);
  if (0 != execute_write_memory (dev, address, length, data))
    {
      send_reply (conn, "E02");
      return;
    }
  send_reply (conn, "OK");
}



/* Deal with the request to read the BDCCSR.  */
void
gdb_rcmd (const char *cmd, int conn, int dev)
{
  char *x = strchr (cmd, ',');
  if (x == NULL)
      goto err;

  if (strlen (x) > strlen ("bdccsr") * 2 + 1)
    goto err;

  x++;  /* Skip the comma */

  /* The command is hex encoded, so decode it.  */
  uint8_t operation[64] = {0};
  hexstring_to_data (operation, x);

  if (strcmp ("bdccsr", (char *) operation) != 0)
    goto err;

  uint16_t csr = execute_read_bdccsr (dev);

  /* Print the result as a string.  */
  char string[5];
  snprintf (string, 5, "%04x", csr);
  string[4] = '\0';

  /* Convert the result back into hex.  */
  char buf[9];
  memset (buf, 0, 9);
  string_to_hex (buf, string);

  send_reply (conn, buf);

err:

  send_reply (conn, "E00");
}


/*  Reads all the registers in one operation.  */
void
gdb_read_registers (const char *cmd, int conn, int dev)
{
  char reply[13*8];
  reply[0] = '\0';
  uint16_t csr = execute_read_bdccsr (dev);
  if (0 == (csr & BDCCSR_ENABLE))
    send_reply (conn, "E00");
  else if (0 == (csr & BDCCSR_ACTIVE))
    send_reply (conn, "E01");
  else
    {
      for (int reg = 0; reg < n_registers; ++reg)
	{
	  char thisreg[9];
	  int size = registers[reg].size;
	  uint32_t value = execute_read_register (dev, reg);
	  snprintf (thisreg, 9, "%0*X", size * 2, value);
	  strcat (reply, thisreg);
	}
      send_reply (conn, reply);
    }
}


/* Reads data from memory on the target.  */
void
gdb_read_memory (const char *cmd, int conn, int dev)
{
  tgt_addr address = 0;
  unsigned int length = 0;
  int n = sscanf (cmd, "m" UTGT_ADDR ",%x", &address, &length);
  if (n != 2)
    {
      send_reply (conn, "E00");
      return;
    }
  char *reply = alloca (length * 2 + 1);
  reply[0] = '\0';
  uint8_t *x = alloca (length);
  if (! cfm_read_memory (dev, address, length, x))
    {
      send_reply (conn, "E01");
      return;
    }
  for (int i = 0; i < length; ++i)
    {
      snprintf (reply + i*2, 3, "%02x", x[i]);
    }
  send_reply (conn, reply);
}


/* Reads a single register from the target.  */
void
gdb_read_register (const char *cmd, int conn, int dev)
{
  int reg = -1;
  int n = sscanf (cmd, "p%02x", &reg);
  if (n != 1)
    {
      send_reply (conn, "E00");
      return;
    }
  if (reg >= n_registers)
    {
      send_reply (conn, "E01");
      return;
    }
  char reply[9];
  uint32_t value = execute_read_register (dev, reg);
  int size = registers[reg].size;

  snprintf (reply, 9, "%0*X", size * 2, value);
  send_reply (conn, reply);
}


/* Writes a single register to the target.  */
void
gdb_write_register (const char *cmd, int conn, int dev)
{
  struct register_spec rs = {0, 0};
  int n = sscanf (cmd, "P%x=%0x", &rs.reg, &rs.value);
  if (n != 2)
    {
      send_reply (conn, "E00");
      return;
    }
  if (rs.reg >= n_registers)
    {
      send_reply (conn, "E01");
      return;
    }

  if (0 != execute_write_regs (dev, 1, &rs))
    {
      send_reply (conn, "E02");
      return;
    }
  send_reply (conn, "OK");
}

/* Sets a breakpoint on the target.  */
void
gdb_insert_breakpoint (const char *cmd, int conn, int dev)
{
  /* Insert breakpoint */
  int type = -1;
  tgt_addr address = 0;
  int kind = -1;
  enum bp_type break_type = BP_NONE;
  sscanf (cmd, "Z%d," UTGT_ADDR ",%x", &type, &address, &kind);
  switch (type)
    {
    case 1:
      break_type = BP_BREAK;
      break;
    case 2:
      break_type = BP_WATCH;
      break;
    default:
      send_reply (conn, "");
      return;
    }

  if (bp_insert (dev, address, break_type))
    send_reply (conn, "OK");
  else
    send_reply (conn, "E00");
}

/* Removes a breakpoint from the target.  */
void
gdb_remove_breakpoint (const char *cmd, int conn, int dev)
{
  /* Remove breakpoint */
  int type = -1;
  tgt_addr address = 0;
  int kind = -1;
  sscanf (cmd, "z%d," UTGT_ADDR ",%x", &type, &address, &kind);
  switch (type)
    {
    case 1:
    case 2:
      if (bp_remove (dev, address))
	send_reply (conn, "OK");
      else
	send_reply (conn, "E00");
      break;
    default:
      send_reply (conn, "");
    }
}

/* Single instruction step.  */
void
gdb_step (const char *cmd, int conn, int dev)
{
  tgt_addr address;
  int n = sscanf (cmd, "s" UTGT_ADDR, &address);
  if (n == 1)
    {
      struct register_spec pc;
      pc.value = address;
      pc.reg = 11;
      execute_write_regs (dev, 1, &pc);
    }
  execute_step (dev);
  send_reply (conn, "S05");
}

/* Set the target running.  */
void
gdb_vrun (const char *ccmd, int conn, int dev)
{
  char *cmd = strdup (ccmd);
  strtok (cmd, ";");
  char *prog = strtok (0, ";");
  if (prog != 0)
    {
      /* We don't currently support a program name argument */
      send_reply (conn, "E01");
    }
  else
    {
      execute_reset_bdm (dev);
      execute_sync (dev);
      gdb_restart (ccmd, conn, dev);
      send_reply (conn, "S00");
    }
  free (cmd);
}

/* Enter or leave non-stop mode.  */
void
gdb_nonstop (const char *cmd, int conn, int dev)
{
  int stat = -1;
  int n = sscanf (cmd, "QNonStop: %d", &stat);
  if (n != 1)
    {
      send_reply (conn, "E00");
    }
  else
    {
      if (stat == 0)
	{
	  /* All stop mode */
	  execute_background (dev);
	  disarm_breakpoints (dev);
	  running = false;
	  send_reply (conn, "OK");
	}
      else
	{
	  send_reply (conn, "E01");
	}
    }
}


/* Indicate to GDB whether we have restarted the target since
   we attached.  */
void
gdb_attached (const char *cmd, int conn, int dev)
{
  if (restarted)
    send_reply (conn, "0");
  else
    send_reply (conn, "1");
}

/* Set the CPU running starting at the current PC.  */
void
gdb_continue (const char *cmd, int conn, int dev)
{
  arm_breakpoints (dev);
  execute_go (dev);

  running = true;
}


void
gdb_positive_ack (const char *cmd, int conn, int dev)
{
  send_reply (conn, "S00");
}


/* Transfer the target's memory map.  */
void
gdb_memory_map (const char *cmd, int conn, int dev)
{
  if (0 != strncmp (cmd, "qXfer:memory-map:read", strlen ("qXfer:memory-map:read")))
    return;
  const char *x = gen_memmap_xml ();
  send_reply (conn, x);
}

/* Starts no ack mode.  */
void
gdb_start_no_ack (const char *cmd, int conn, int dev)
{
  send_reply (conn, "OK");
  no_ack = true;
}

static const char *
strchrs (const char *in, const char *what)
{
    while (*in)
    {
        for (const char *s = what; *s; s++)
        {
            if (*s == *in)
                return in;
        }
        in++;
    }
    return in;
}


/*  FIXME: This prototype is a lie!!  The arguments are not undefined.
    However they vary between platforms.
    We need somehow to get the correct prototype out of the generated code.
    This however suffices in that the return type is correct.  */
struct gdb_cmd *in_word_set ();


/* Receives a command from GDB over CONN, processes it as appropriate
   sending any necessary commands to the target using DEV.  */
static void
process_packet (int conn, int dev)
{
  char *cmd = NULL;
  size_t cmd_len = 0;
  if (receive_command (conn, &cmd, &cmd_len))
    {
      cmd[cmd_len] = '\0';

      /* Except for a few specials, command names are a single byte long.  */
      int length = 0;
      switch (cmd[0])
        {
        case 'q':
        case 'Q':
        case 'v':
          {
            const char *end = strchrs (cmd, "0123456789:,");
            length = end - cmd;
            break;
          }
        default:
          length = 1;
          break;
        }

      const struct gdb_cmd *gc = in_word_set (cmd, length);
      if (gc == NULL)
        {
	  send_reply (conn, "");
        }
      else
        {
          gc->funx (cmd, conn, dev);
        }
    }
}

/*  Called by the body of the main loop.  That is to say,
    when there are received data or a timeout.  */
void
process_gdb_message (int conn, const char bcmd, int dev)
{
  switch (bcmd)
    {
    case '+':
      /* Positive ACK.  Do nothing */
      break;
    case '-':
      /* Negative ACK.
	 FIXME: we should resend the last packet.  */
      upad_msg (MSG_ERROR, "Negative ack from gdb");
      break;
    case '$':
      process_packet (conn, dev);
      break;
    case 0x03: /* Interrupt */
      execute_background (dev);
      disarm_breakpoints (dev);
      running = false;
      send_reply (conn, "S02");
      break;
    default:
      break;
    }
}

/* Local Variables:  */
/* mode: c           */
/* c-style: "gnu"    */
/* End:              */
