/*
  upad - A program for debugging, and uploading code to embedded devices.
  Copyright (C) 2016, 2019 John Darrington

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include <config.h>
#include <assert.h>
#include "elf-uploader.h"
#include "misc.h"
#include <string.h>

static uint16_t
be_to_native_u16 (uint16_t x)
{
#if WORDS_BIGENDIAN
  return x;
#else
  return ((x & 0x00FF) << 8) | x >> 8;
#endif
}

static uint32_t
be_to_native_u32 (uint32_t x)
{
#if WORDS_BIGENDIAN
  return x;
#else
  return
    ((x & 0x000000FF) << 24) |
    ((x & 0x0000FF00) <<  8) |
    ((x & 0x00FF0000) >>  8) |
    ((x & 0xFF000000) >> 24) ;
#endif
}


/* Shamelessly lifted from that veritable source of
   information: Wikipedia */
struct elf_header
{
  uint8_t  e_ident[0x10];
  uint16_t e_type;
  uint16_t e_machine;
  uint32_t e_version;

  uint32_t e_entry;
  uint32_t e_phoff;
  uint32_t e_shoff;

  uint32_t e_flags;
  uint16_t e_ehsize;
  uint16_t e_phentsize;
  uint16_t e_phnum;
  uint16_t e_shentsize;
  uint16_t e_shnum;
  uint16_t e_shstrndx;
};

struct program_header_32
{
  uint32_t p_type;
  uint32_t p_offset;
  uint32_t p_vaddr;
  uint32_t p_paddr;
  uint32_t p_filesz;
  uint32_t p_memsz;
  uint32_t p_flags;
  uint32_t p_align;
};


#define PT_LOAD 0x00000001



#define MIN(A,B) ((A) > (B)) ? (B) : (A)

struct elf_upload
{
  struct upload_format parent;
  FILE *fp;
  struct elf_header eh;

  size_t ph_idx;
  size_t posn;
};

static bool
load_predicate (const struct program_header_32 *h)
{

  if (0 == be_to_native_u32 (h->p_filesz))
    return false;

  if (PT_LOAD != be_to_native_u32 (h->p_type))
    return false;

  return true;
}

static bool
read_structure (void *x, size_t bytes, FILE *fp)
{
  size_t n = 0;
  while (n < bytes)
    {
      size_t r = fread (x, 1, bytes - n, fp);
      if (r == 0)
	return false;
      n += r;
    }
  return true;
}

#define READ_STRUCTURE(S, FP) \
  read_structure (&(S), sizeof (S), (FP))


static bool
elf_get_data_fragment (struct upload_format *uf, tgt_addr *address,
		       uint8_t *data, size_t bufsize, size_t *n_data)
{
  struct elf_upload *sru = (struct elf_upload *) uf;

  *n_data = 0;

  struct program_header_32 ph;
  do
    {
      if (sru->ph_idx >= be_to_native_u16 (sru->eh.e_phnum))
	{
	  return false;
	}

      if (0 != fseek (sru->fp,
		      be_to_native_u32 (sru->eh.e_phoff) +
		      sru->ph_idx * sizeof (ph),
		      SEEK_SET))
	{
	  upad_errno (MSG_ERROR, "Cannot seek to elf program header");
	  return false;
	}

      if (! READ_STRUCTURE (ph, sru->fp))
	{
	  upad_errno (MSG_ERROR, "Cannot read elf program header");
	  return false;
	}
      if (! load_predicate (&ph))
	sru->ph_idx++;
    }
  while (! load_predicate (&ph));

  *address = be_to_native_u32 (ph.p_paddr) + sru->posn;
  int n = be_to_native_u32 (ph.p_memsz);

  if (0 != fseek (sru->fp, sru->posn + be_to_native_u32 (ph.p_offset), SEEK_SET))
    {
      upad_errno (MSG_ERROR, "Cannot seek.");
      return false;
    }
  assert (sru->posn <= n);
  size_t n_bytes = MIN (bufsize, n - sru->posn);
  size_t r = fread (data, 1, n_bytes, sru->fp);
  if (r == 0)
    {
      upad_errno (MSG_ERROR, "Cannot read from elf file offset 0x%08zx.",
                  sru->posn + be_to_native_u32 (ph.p_offset));
      return false;
    }
  *n_data = r;
  sru->posn += r;

  if (sru->posn >= n)
    {
      sru->ph_idx++;
      sru->posn = 0;
    }

  return true;
}


static void
destroy_elf_uploader (struct upload_format *u)
{
  free (u);
}

struct upload_format *
create_elf_uploader (FILE *fp)
{
  struct elf_upload *sru = safe_realloc (NULL, sizeof (*sru));
  memset (sru, 0, sizeof (*sru));
  sru->fp = fp;
  sru->ph_idx = 0;
  sru->posn = 0;

  if (! READ_STRUCTURE (sru->eh, fp))
    {
      upad_errno (MSG_ERROR, "Cannot read elf header");
      goto fail;
    }

  if (sru->eh.e_ident[4] != 1)
    {
      upad_msg (MSG_FATAL, "Only 32 bit elf files are currently supported.\n");
      goto fail;
    }

  if (be_to_native_u16 (sru->eh.e_type) != 2)
    {
      upad_msg (MSG_WARNING, "Elf file is not an executable file.\n");
    }

  ((struct upload_format *)sru)->get_data_fragment = elf_get_data_fragment;
  ((struct upload_format *)sru)->destroy = destroy_elf_uploader;
  return (struct upload_format *) sru;

 fail:
  free (sru);
  return NULL;
}

/* Local Variables:  */
/* mode: c           */
/* c-style: "gnu"    */
/* End:              */
