/*
  upad - A program for debugging, and uploading code to embedded devices.
  Copyright (C) 2016, 2019 John Darrington

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* This module implements hardware breakpoints on the S12Z */

#include <config.h>
#include "breakpoints.h"

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>

#include <netdb.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <arpa/inet.h>

#include <string.h>
#include <unistd.h>

#include <stdint.h>
#include <stdbool.h>

#include "gdb-interface.h"

#include "transaction.h"
#include "misc.h"
#include "registers.h"

#include "target/s12z/dbg.h"

/* Arm all breakpoints */
void
arm_breakpoints (int dev)
{
  uint8_t buf[4];
  /* Arm the breakpoint module, and configure it to go into background mode
     when a breakpoint occurs. */
  buf[0] = 0x98;
  execute_write_memory (dev,
                        DBG_BASE + offsetof (DBG_TypeDef, DBGC1),
                        sizeof (DBG->DBGC1),
                        buf);
}

/* Disarm all breakpoints */
void
disarm_breakpoints (int dev)
{
  uint8_t buf[4];
  buf[0] = 0x18;
  execute_write_memory (dev,
                        DBG_BASE + offsetof (DBG_TypeDef, DBGC1),
                        sizeof (DBG->DBGC1),
                        buf);
}

static int n_breakpoints = 0;


/* Return true if at least one breakpoint is set. */
bool
breakpoints_active (void)
{
  return n_breakpoints > 0;
}


/* Return non-zero if at least one breakpoint event occured since the
   module was armed.  */
uint8_t
breakpoint_fired (int dev)
{
  uint8_t buf[1];

  execute_read_memory (dev,
                       DBG_BASE + offsetof (DBG_TypeDef, DBGEFR),
                       1,
                       buf);

  return buf[0] & 0x0B;
}

/* Return TRUE iff the breakpoint comparator at CTL_OFFS is currently disabled. */
static bool
isclear_breakpoint (int dev, size_t ctl_offs, size_t ctl_size)
{
  uint8_t buf[4];
  /* Check to see if this comparator is enabled */
  execute_read_memory (dev,
                       DBG_BASE + ctl_offs,
                       ctl_size,
                       buf);

  return !(buf[0] & 0x01);
}


/* Return the type of the breakpoint whose control register is located at CTL_OFFS */
static enum bp_type
typeof_breakpoint (int dev, size_t ctl_offs, size_t ctl_size)
{
  uint8_t buf[4];
  /* Check to see if this comparator is enabled */
  execute_read_memory (dev,
                       DBG_BASE + ctl_offs,
                       ctl_size,
                       buf);


  /* IF the COMPE bit is unset, then there is no breakpoint here.  */
  if (0 == (buf[0] & 0x01))
    return BP_NONE;

  /* Check the INST bit and return the appropriate type.  */
  return (buf[0] & 0x20) ? BP_BREAK : BP_WATCH;
}


/*
  Remove a breakpoint from CTL if any is set there.
  Normally this should be called from the REMOVE_BP macro.
*/
static bool
remove_breakpoint (int dev, size_t ctl_offs, size_t ctl_size)
{
  uint8_t buf[4];

  /* Disable the comparator. */
  buf[0] = 0x00;
  execute_write_memory (dev,
                        DBG_BASE + ctl_offs,
                        ctl_size,
                        buf);

  /* Check that it worked */
  isclear_breakpoint (dev, ctl_offs, ctl_size);

  return true;
}

/*
  Return true if comparator CTL is set and its value (CMP) is ADDRESS.
  Normally this should be called using the INSPECT_BP macro.
*/
static bool
inspect_breakpoint (int dev, tgt_addr address, size_t ctl_offs, size_t ctl_size,
                    size_t cmp_offs, size_t cmp_size)
{
  uint8_t buf[4];
  /* Check to see if this comparator is enabled */
  execute_read_memory (dev,
                       DBG_BASE + ctl_offs,
                       ctl_size,
                       buf);

  if (buf[0] & 0x21)
    {
      /* If it is, then see if the compare address is what we're looking for. */
      execute_read_memory (dev,
                           DBG_BASE + cmp_offs,
                           cmp_size,
                           buf);
      uint32_t addr ;
      addr = buf[3];
      addr |= buf[2] << 8;
      addr |= buf[1] << 16;
      addr |= buf[0] << 24;
      if (addr == address)
        return true;
    }

  return false;
}

/* Insert a hardware breakpoint at ADDRESS.
   Normally this function should be called using the INSERT_BP macro.
*/
static bool
insert_breakpoint (int dev, tgt_addr address,
                   size_t ctl_offs, size_t ctl_size,
                   size_t cmp_offs, size_t cmp_size,
                   enum bp_type type)
{
  uint8_t buf[4];

  /* Set the address to match */
  buf[0] = address >> 24;
  buf[1] = address >> 16;
  buf[2] = address >> 8;
  buf[3] = address ;
  execute_write_memory (dev,
                        DBG_BASE + cmp_offs,
                        cmp_size,
                        buf);

  /* Enable the comparator for exact matches of the program counter. */
  buf[0] = (type == BP_WATCH) ? 0x05 : 0x21;
  execute_write_memory (dev,
                        DBG_BASE + ctl_offs,
                        ctl_size,
                        buf);

  /* On any match, go to the "Final State" (which emits the breakpoint). */
  buf[0] = 0xCF;
  execute_write_memory (dev,
                        DBG_BASE + offsetof (DBG_TypeDef, DBGSCR1),
                        sizeof (DBG->DBGSCR1),
                        buf);

  return true;
}



/* Return true if comparator WHICH has a breakpoint set and that breakpoint
   is set at ADDR.  */
#define INSPECT_BP(DEV, ADDR, WHICH)					\
  inspect_breakpoint ((DEV), (ADDR),					\
                      offsetof (DBG_TypeDef, WHICH##CTL), sizeof (DBG->WHICH##CTL), \
                      offsetof (DBG_TypeDef, WHICH##A), sizeof (DBG->WHICH##A))

/* Return true if comparator WHICH is NOT set.  */
#define ISCLEAR_BP(DEV, WHICH)                                          \
  isclear_breakpoint ((DEV),                                            \
                      offsetof (DBG_TypeDef, WHICH##CTL), sizeof (DBG->WHICH##CTL))


/* Return the breakpoint type corresponding to WHICH.  */
#define TYPEOF_BP(DEV, WHICH)                                          \
  typeof_breakpoint ((DEV),                                            \
                      offsetof (DBG_TypeDef, WHICH##CTL), sizeof (DBG->WHICH##CTL))


/* Insert a breakpoint at ADDR, using the comparator WHICH.  TYPE indicates a
   breakpoint or a watchpoint */
#define INSERT_BP(DEV, ADDR, WHICH, TYPE)                               \
  insert_breakpoint ((DEV), (ADDR),                                     \
                     offsetof (DBG_TypeDef, WHICH##CTL), sizeof (DBG->WHICH##CTL), \
                     offsetof (DBG_TypeDef, WHICH##A), sizeof (DBG->WHICH##A), TYPE)

/* Remove a breakpoint (if any is set) from comparator WHICH */
#define REMOVE_BP(DEV, WHICH)                                           \
  remove_breakpoint ((DEV),                                             \
                     offsetof (DBG_TypeDef, WHICH##CTL), sizeof (DBG->WHICH##CTL))


/* Remove a hardware breakpoint from ADDRESS */
bool
bp_remove (int dev, tgt_addr address)
{
  if (INSPECT_BP (dev, address, DBGD))
    {
      REMOVE_BP (dev, DBGD);
      goto success;
    }

  if (INSPECT_BP (dev, address, DBGB))
    {
      REMOVE_BP (dev, DBGB);
      goto success;
    }

  if (INSPECT_BP (dev, address, DBGA))
    {
      REMOVE_BP (dev, DBGA);
      goto success;
    }

  return false;

 success:

  n_breakpoints--;
  return true;
}


/* SET is a bitfield containing the set of breakpoints which have fired.
   This function returns the type of the breakpoint which is a member
   of SET, which has the highest priority.  */
enum bp_type
breakpoint_type_in_set (int dev, uint8_t set)
{
  /* First check if any comparators are configured as breakpoints.  */
  if (set & 0x01)
    {
      if (BP_BREAK ==  TYPEOF_BP (dev, DBGA))
        return BP_BREAK;
    }

  if (set & 0x02)
    {
      if (BP_BREAK ==  TYPEOF_BP (dev, DBGB))
        return BP_BREAK;
    }

  if (set & 0x08)
    {
      if (BP_BREAK ==  TYPEOF_BP (dev, DBGD))
        return BP_BREAK;
    }

  /* Then check if any are configured as watchpoints.   */
  if (set & 0x01)
    {
      if (BP_WATCH ==  TYPEOF_BP (dev, DBGA))
        return BP_WATCH;
    }

  if (set & 0x02)
    {
      if (BP_WATCH ==  TYPEOF_BP (dev, DBGB))
        return BP_WATCH;
    }

  if (set & 0x08)
    {
      if (BP_WATCH ==  TYPEOF_BP (dev, DBGD))
        return BP_WATCH;
    }

  return BP_NONE;
}

/* Insert a hardware breakpoint at ADDRESS */
bool
bp_insert (int dev, tgt_addr address, enum bp_type type)
{
  bool success = false;
  if (ISCLEAR_BP (dev, DBGA))
    {
      success = INSERT_BP (dev, address, DBGA, type);
    }

  else if (ISCLEAR_BP (dev, DBGB))
    {
      success = INSERT_BP (dev, address, DBGB, type);
    }

  else if (ISCLEAR_BP (dev, DBGD))
    {
      success = INSERT_BP (dev, address, DBGD, type);
    }

  if (success)
    n_breakpoints++;

  return success;
}

/* Local Variables:  */
/* mode: c           */
/* c-style: "gnu"    */
/* End:              */
