/*
  upad - A program for debugging, and uploading code to embedded devices.
  Copyright (C) 2016 John Darrington

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include <config.h>

#include "user-interface.h"

#include <stdio.h>
#include <sys/stat.h>

#include <readline/readline.h>
#include <readline/history.h>

#include <errno.h>
#include <unistd.h>
#include <signal.h>
#include "lib/md5.h"
#include "transaction.h"
#include "misc.h"
#include "registers.h"
#include "srecord-uploader.h"
#include "binary-uploader.h"

#if HAVE_WORDEXP_H
# include <wordexp.h>
#endif

static const enum ARG_TYPE echo_args[] = {ARG_FILENAME, ARG_FILENAME};
static const enum ARG_TYPE write_bdccsr_args[] = {ARG_DATUM};
static const enum ARG_TYPE read_memory_args[] = {ARG_ADDRESS, ARG_QUANTITY};
static const enum ARG_TYPE upload_binary_args[] = {ARG_FILENAME, ARG_QUANTITY};

static int
start_upload_binary (int des, int argc, char **argv)
{
  if (argc < 2)
    {
      fprintf (stderr, "Usage: upload-binary <file-name> <address>\n");
      return 1;
    }

  char *file = argv[0];
  uint32_t address = strtol (argv[1], NULL, 0);

#if HAVE_WORDEXP_H
  wordexp_t exp;
  if (0 != wordexp (file, &exp, WRDE_NOCMD))
    {
      upad_msg (MSG_ERROR, "Cannot expand string `%s'\n", file);
      wordfree (&exp);
      return 1;
    }
  file = exp.we_wordv[0];
#endif
  FILE *fp = fopen (file, "r");
  if (fp == NULL)
    {
      char *emsg = strerror (errno);
      upad_msg (MSG_ERROR, "Cannot open file `%s': %s\n", file, emsg);
      return 1;
    }
#if HAVE_WORDEXP_H
  wordfree (&exp);
#endif

  struct upload_format *uf = create_binary_uploader (fp, address);
  int ret = upload_format (des, uf, 0);
  free (uf);

  if (fp)
    fclose (fp);
  return ret;
}


static int
start_upload_srec (int des, int argc, char **argv)
{
  if (argc < 1)
    {
      fprintf (stderr, "Usage: upload-srecord <file-name>\n");
      return 1;
    }

  char *file = argv[0];
#if HAVE_WORDEXP_H
  wordexp_t exp;
  if (0 != wordexp (file, &exp, WRDE_NOCMD))
    {
      upad_msg (MSG_ERROR, "Cannot expand string `%s'\n", file);
      wordfree (&exp);
      return 1;
    }
  file = exp.we_wordv[0];
#endif
  FILE *fp = fopen (file, "r");
  if (fp == NULL)
    {
      char *emsg = strerror (errno);
      upad_msg (MSG_ERROR, "Cannot open file `%s': %s\n", file, emsg);
      return 1;
    }
#if HAVE_WORDEXP_H
  wordfree (&exp);
#endif
  struct upload_format *uf = create_srec_uploader (fp);
  int ret = upload_format (des, uf, 0);
  free (uf);
  if (fp)
    fclose (fp);
  return ret;
}

static int
send_write_regs (int des, int argc, char **argv)
{
  int n_regs = 0;

  if (argc < 1 || argc % 2 == 1)
    {
      upad_msg (MSG_INFORMATION,
		   "Usage: write-regs <reg0-name> <reg0-value>  "
		   "<reg1-name> <reg1-value>  ... <regN-name> <regN-value>\n");
      return 1;
    }

  const struct cpu_reg *reg = NULL;
  struct register_spec *rs = NULL;
  int r;
  for (r = 0; r < argc; ++r)
    {
      if (r % 2 == 0)
        {
          int c = -1;
          for (c = 0; c < n_registers; ++c)
            {
              reg = registers + c;
              if (0 == strcasecmp (reg->name, argv[r]))
                break;
              reg = NULL;
            }
          if (!reg)
            {
              upad_msg (MSG_ERROR, "Invalid register name: %s\n", argv[r]);
              return 0;
            }
          rs = safe_realloc (rs, sizeof (*rs) * (r + 1));
          rs[r/2].reg = c;
          /* Set the register to be written */
        }
      else
        {
          int32_t val = strtol (argv[r], 0, 0);
          rs[r/2].value = val;
          n_regs++;
        }
    }

  return execute_write_regs (des, n_regs, rs);
}



static int
send_write_memory (int des, int argc, char **argv)
{
  if (argc < 3)
    {
      upad_msg (MSG_INFORMATION, "Usage: write-mem <address> <N> <byte>\n");
      return 1;
    }
  uint32_t address = strtol (argv[0], NULL, 0);
  size_t quantity = strtol (argv[1], NULL, 0);
  uint8_t fill = strtol (argv[2], NULL, 0);

  uint8_t *data = safe_realloc (NULL, quantity + 1 + 4 + 4);
  memset (data, fill, quantity + 1 + 4 + 4);


  const struct memsector *ms = get_memory_class (address);

  if (0 > execute_write_memory (des, address, quantity, data + 1, ms))
      return -1;

  return 0;
}

static int
disassemble (int des, int argc, char **argv)
{
  if (argc != 2)
    {
      upad_msg (MSG_INFORMATION, "Usage: disassemble <address> <n-bytes>\n");
      return 1;
    }
  uint32_t address = strtol (argv[0], NULL, 0);
  int quantity = strtol (argv[1], NULL, 0);

  const uint8_t *rx_data = execute_read_memory (des, address, quantity);

  char cmd[256];
  char filename[30] = "/tmp/dis.XXXXXX";
  const char objdump[] = "s12z-objdump";
  int fd = mkstemp (filename);
  if (fd == -1)
    {
      upad_errno (MSG_ERROR, "Cannot create temporary file");
      return 1;
    }
  while (quantity > 0)
    {
      int n = write (fd, rx_data, quantity);
      quantity -= n;
      rx_data += n;
    }

  if (0 > snprintf (cmd, 256,
                    "%s --adjust-vma=%d -b binary -m s12z -D -z %s",
                    objdump,
                    address,
                    filename))
    {
      upad_msg (MSG_ERROR, "Cannot create disassembly command\n");
      return 1;
    }

  if (0 != system (cmd))
    {
      upad_msg (MSG_ERROR, "Error running disassembly\n");
      return 1;
    }

  close (fd);

  return 0;
}

static int
send_read_all_regs (int des, int argc, char **argv)
{
  size_t n_regs;
  uint32_t *reg_values;
  execute_read_all_regs (des, &reg_values, &n_regs);

  int i;
  for (i = 0; i < n_regs; ++i)
    {
      uint32_t reg_value = reg_values[i];

      fprintf (stdout, "%3s: ", registers[i].name);
      if (i == 12)
        ccw_print (stdout, reg_value);
      else
        fprintf (stdout, "%0*X", registers[i].size * 2, reg_value);
      fputc ('\n', stdout);
    }
  return 0;
}

static int
send_read_bdccsr (int des, int argc, char **argv)
{
  printf ("BDCCSR: %04X\n", execute_read_bdccsr (des));
  return 0;
}

static int
send_write_bdccsr (int des, int argc, char **argv)
{
  if (argc < 1)
    {
      upad_msg (MSG_INFORMATION, "Usage: write-bdccsr <word>\n");
      return 1;
    }

  uint16_t csr = strtol (argv[0], NULL, 0);

  return execute_write_bdccsr (des, csr);
}

static int
send_read_register (int des, int argc, char **argv)
{
  int reg = strtol (argv[0], NULL, 0);

  uint32_t reg_value = execute_read_register (des, reg);
  fprintf (stdout, "Register is 0x%08x\n", reg_value);

  return 0;
}


static int
send_step (int des, int argc, char **argv)
{
  return execute_step (des);
}


static int
send_background (int des, int argc, char **argv)
{
  return execute_background (des);
}

static int
send_reset (int des, int argc, char **argv)
{
  return execute_reset (des);
}

static int
send_go (int des, int argc, char **argv)
{
  return execute_go (des);
}

static int
send_poll (int des, int argc, char **argv)
{
  return execute_poll (des);
}


static int
send_bulk_erase (int des, int argc, char **argv)
{
  return execute_bulk_erase (des);
}

static int
send_sync (int des, int argc, char **argv)
{
  return execute_sync (des);
}

static int
send_reset_bdm (int des, int argc, char **argv)
{
  return execute_reset_bdm (des);
}

static int
send_erase_sector (int des, int argc, char **argv)
{
  if (argc != 1)
    {
      upad_msg (MSG_INFORMATION, "Usage: erase-sector <address>\n");
      return 1;
    }
  uint32_t address = strtol (argv[0], NULL, 0);

  return execute_erase_sector (des, address);
}

static int
send_md5_check (int des, int argc, char **argv)
{
  if (argc != 2)
    {
      upad_msg (MSG_INFORMATION, "Usage: md5-check <address> <n-bytes>\n");
      return 1;
    }
  uint32_t address = strtol (argv[0], NULL, 0);
  int quantity = strtol (argv[1], NULL, 0);

  uint8_t result[MD5_DIGEST_SIZE];
  int res = remote_md5_check (des, address, quantity, result);

  int i;
  for (i = 0; i < 16; ++i)
    {
      printf ("%02x", result[i]);
    }
  putchar ('\n');

  return res;
}

static int
send_echo (int des, int argc, char **argv)
{
  if (argc < 2)
    {
      upad_msg (MSG_INFORMATION,
		   "Usage: echo <infile> <outfile>\n");
      return 1;
    }

  FILE *fpin = fopen (argv[0], "r");
  if (fpin == NULL)
    {
      upad_errno (MSG_ERROR, "Cannot open file %s", argv[0]);
      return 1;
    }

  struct stat statbuf;
  if (0 != fstat (fileno (fpin), &statbuf))
    {
      upad_errno (MSG_ERROR, "Stat failed on %s", argv[0]);
      fclose (fpin);
      return 1;
    }

  FILE * fpout = fopen (argv[1], "w");
  if (fpout == NULL)
    {
      upad_errno (MSG_ERROR, "Cannot open file %s for writing",
		     argv[1]);
      return 1;
    }

  if (0 != execute_echo (des, statbuf.st_size, fpin, fpout))
      goto fail;

  fclose (fpout);
  return 0;

 fail:
  fclose (fpout);
  return 1;
}


static void
print_data_as_numeric (uint32_t base, uint32_t offset, int quantity,
                       const uint8_t *data, FILE *stream)
{
  uint32_t a;
  for (a = base ; a < base + 16; ++a)
    {
      if (a < base + offset)
        fputs ("   ", stream);
      else if (a >= base + offset + quantity)
        fputs ("   ", stream);
      else
        fprintf (stream, " %02X", data[a - base - offset]);
      if (a % 8 == 7 && a % 16 < 15)
        fputs ("  ", stream);
    }
}

static void
print_data_as_char (uint32_t base, uint32_t offset, int quantity,
                    const uint8_t *data, FILE *stream)
{
  uint32_t a;
  for (a = base ; a < base + 16; ++a)
    {
      if (a < base + offset)
        fputs (" ", stream);
      else if (a >= base + offset + quantity)
        fputs (" ", stream);
      else
	{
          const char c = data [a - base - offset];
          fprintf (stream, "%c",
                   isprint (c) ? c : '.');
	}
      if (a % 8 == 7)
        fputs (" ", stream);
    }
}

static int
send_read_memory (int des, int argc, char **argv)
{
  if (argc != 2)
    {
      upad_msg (MSG_INFORMATION, "Usage: read-mem <address> <n-bytes>\n");
      return 1;
    }
  uint32_t address = strtol (argv[0], NULL, 0);
  int quantity = strtol (argv[1], NULL, 0);

  const uint8_t *rx_data = execute_read_memory (des, address, quantity);

  uint32_t offset = address % 16;
  uint32_t base = address - offset;

  while (quantity > 0)
    {
      fprintf (stdout, "%08X:  ", base);
      print_data_as_char (base, offset, quantity, rx_data, stdout);
      fputs (" ", stdout);
      print_data_as_numeric (base, offset, quantity, rx_data, stdout);
      fputs ("\n", stdout);
      rx_data += 16 - offset;
      quantity -= 16 - offset;
      offset = 0;
      base += 16;
    }
  return 0;
}



static const char copying[] =
  "This program is free software: you can redistribute it and/or modify\n"
  "it under the terms of the GNU General Public License as published by\n"
  "the Free Software Foundation, either version 3 of the License, or\n"
  "(at your option) any later version.\n"
  "\n"
  "This program is distributed in the hope that it will be useful,\n"
  "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
  "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
  "GNU General Public License for more details.\n"
  "\n"
  "You should have received a copy of the GNU General Public License\n"
  "along with this program.  If not, see <http://www.gnu.org/licenses/>.\n";


static const char lack_of_warranty[] =
  "THERE IS NO WARRANTY FOR THE PROGRAM, TO THE EXTENT PERMITTED BY\n"
  "APPLICABLE LAW.  EXCEPT WHEN OTHERWISE STATED IN WRITING THE COPYRIGHT\n"
  "HOLDERS AND/OR OTHER PARTIES PROVIDE THE PROGRAM \"AS IS\" WITHOUT WARRANTY\n"
  "OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO,\n"
  "THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR\n"
  "PURPOSE.  THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE PROGRAM\n"
  "IS WITH YOU.  SHOULD THE PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF\n"
  "ALL NECESSARY SERVICING, REPAIR OR CORRECTION.\n";


static int
show (int des, int argc, char **argv)
{
  if (argc < 1)
    return 1;
  if (0 == strcmp ("warranty", argv[0]))
    fputs (lack_of_warranty, stdout);
  else if (0 == strcmp ("copying", argv[0]))
    fputs (copying, stdout);
  return 0;
}

const struct xcmd cmds[] =
  {
    /*-
      @node show
      @section show

      @cindex warranty
      @cindex copying
      The @command{show} command displays auxiliary information to the user.  It
      takes a single argument which should be a string describing the information to show.
      Currently there are two possible values for this argument, @i{viz}: @samp{copying} ---
      which shows information about the terms and conditions for copying the program, and
      @samp{warranty} --- which shows details about the warranty (or lack thereof) which
      comes with @upad{}.
    */
    {"show",         show,                  0, NULL},
    /*-
      @node reset
      @section reset

      @cindex reset
      The @command{reset} command sends a reset signal to the target.  This will mean
      that the target will exit BDM mode and start running the program it contains.
      The command takes no arguments.
    */
    {"reset",        send_reset,            0, NULL},
    /*-
      @node poll
      @section poll

      The @command{poll} command checks that the controller is connected and active, and
      that it is ready to communicate with the computer.  It takes no arguments.
    */
    {"poll",         send_poll,             0, NULL},
    /*-
      @node echo
      @section echo

      The @command{echo} command takes two arguments.  Both arguments are filenames.
      The first argument must be the name of an existing file.  The second argument
      is the name of a file to be created.  The command reads the contents of the first
      file, transmits the contents to the controller, which then sends the contents back
      to the computer where it is written to the second file.  For example:

      @samp{echo @file{source} @file{destination}}

      @noindent This command will create a file called @file{destination} which
      upon command completion should have identical contents to @file{source}.
      This command is not very useful.  It is intended for testing the integrity of
      the controller--computer link.
    */
    {"echo",         send_echo,             2, echo_args},
    /*-
      @node reset-bdm
      @section reset-bdm

      The @command{reset-bdm} command sends a reset signal to the target and places it into
      BDM mode.  Usually you should immediately issue a @command{sync} command after
      sending a @command{reset-bdm} command.
    */
    {"reset-bdm",    send_reset_bdm,        0, NULL},
    /*-
      @node sync
      @section sync

      The @command{sync} command requests the target to resynchronise its communication
      clock.  It is required before any other communication between the controller and
      target can take place.  The command takes no arguments.
    */
    {"sync",         send_sync,             0, NULL},

    /*-
      @node bulk-erase
      @section bulk-erase

      @cindex erasing, bulk erasure

      The @command{bulk-erase} command causes the entire non-volatile memory of the target device
      to be erased.  The command takes no arguments.
    */
    {"bulk-erase",   send_bulk_erase,       0, NULL},

    /*-
      @node read-bdccsr
      @section read-bdccsr

      @cindex BDCCSR
      The @command{read-bdccsr} reads the BDCCSR register from the target device and
      displays its value.  The command takes no arguments.
    */
    {"read-bdccsr",  send_read_bdccsr,      0, NULL},
    /*-
      @node write-bdccsr
      @section write-bdccsr

      @cindex BDCCSR
      The @command{write-bdccsr} command writes the BDCCSR register to the target device.
      It takes exactly one argument which is the value to be written.
    */
    {"write-bdccsr", send_write_bdccsr,     1, write_bdccsr_args},

    /*-
      @node read-all-regs
      @section read-all-regs

      The @command{read-all-regs} command reads all the CPU registers from the
      target device and displays their values.  The command takes no arguments.
    */
    {"read-all-regs",   send_read_all_regs,        0, NULL},
    /*-
      @node read-register
      @section read-register

      The @command{read-register} command reads a specific CPU register from the
      target device and displays its value.  The command takes a single argument
      which is the name of the register to read.
    */
    {"read-register",   send_read_register,        1, NULL},
    /*-
      @node write-regs
      @section write-regs

      The @command{write-regs} command can be used to write values into CPU registers
      on the target device.    The command takes a variable number of arguments (always
      an even number).  Each pair of arguments comprises the name of the register followed
      by the desired value which should be written, thus:

      @samp{write-regs @var{reg0} @var{value0} [@var{reg1} @var{value1} [@dots{}]]}
    */
    {"write-regs",   send_write_regs,       -1, NULL},
    /*-
      @node go
      @section go

      The @command{go} command causes the target to fetch the instruction at the current
      value of the program counter register and to leave background debug mode.
      Once this command has been issued, the device will run until the program
      instructs it to return to background debug mode --- which depending on the program
      may never happen.
      The @command{background} command can be used to manually return to background debug
      mode.
      The command takes no arguments.
    */
    {"go",           send_go,               0, NULL},
    /*-
      @node step
      @section step

      The @command{step} command causes the target to fetch the instruction at the current
      value of the program counter register process this instruction and then return to
      background debug mode.
      The command takes no arguments.
    */
    {"step",         send_step,             0, NULL},
    /*-
      @node background
      @section background

      The @command{background} command causes the target to stop processing instructions
      and return to background debug mode.
      The command takes no arguments.
    */
    {"background",   send_background,       0, NULL},
    /*-
      @node read-mem
      @section read-mem

      The @command{read-mem} command reads data from the target's memory.
      The command takes two arguments.  The first argument is the address of the
      first byte to read.  The second argument is the number of bytes to read,
      starting at that address.
    */
    {"read-mem",     send_read_memory,      2, read_memory_args},
    /*-
      @node write-mem
      @section write-mem

      The @command{write-mem} command writes data to the target's memory.
      The command takes three arguments.  The first argument is the address of the
      where the first datum should be written.  The second argument is the number
      of bytes to write.  The third argument is the value of the byte to be
      written.

      Note that this command attempt to write directly into the memory space indicated
      by the specified address.  In general, you cannot use this command to write
      to non-volatile memory such as flash memory or EEPROM.
    */
    {"write-mem",    send_write_memory,     3, NULL},
    /*-
      @node disassemble
      @section disassemble

      @cindex disassembling
      The @command{disassemble} command reads data from the target's memory, and
      displays the assembler mnemonics corresponding to the data.
      The command takes two arguments.  The first argument is the address of the
      first byte to read.  The second argument is the number of bytes to read,
      starting at that address.
    */
    {"disassemble",  disassemble,           2, read_memory_args},
    /*-
      @node upload-srecord
      @section upload-srecord

      @cindex S-Record, file format
      The @command{upload-srecord} command reads a file from the local computer
      which is expected to contain S-Record formatted data.  The data are uploaded
      to the addresses indicated in the S-Record.  The command takes a single argument
      which is the name of the file to upload.

      This command will not erase any memory before uploading.   So it will
      probably fail if you do not issue a @command{bulk-erase} command (@pxref{bulk-erase})
      prior to calling this command.
    */
    {"upload-srecord", start_upload_srec,   1, NULL},
    /*-
      @node upload-binary
      @section upload-binary

      @cindex binary, file format
      The @command{upload-binary} command reads a file from the local computer
      and uploads its contents as raw data to the target device.  The command
      takes two arguments.  The first argument is the name of the file, and the
      second argument is the address where the data should be written.

      This command will not erase any memory before uploading.   So it will
      probably fail if you do not issue a @command{bulk-erase} command (@pxref{bulk-erase})
      prior to calling this command.
    */
    {"upload-binary",  start_upload_binary, 2, upload_binary_args},
    /*-
      @node md5-check
      @section md5-check

      The @command{md5-check} command reads data from the target's memory, and displays
      the data's md5sum.
      The command takes two arguments.  The first argument is the address of the
      first byte to read.  The second argument is the number of bytes to read,
      starting at that address.
    */
    {"md5-check",    send_md5_check,        2, read_memory_args},
    /*-
      @node erase-sector
      @section erase-sector

      @cindex erasing, sector erasure
      The @command{erase-sector} command erases a single sector of non-volatile
      memory on the target device.  On most devices, this is necessary before
      non-volatile memory can be written.   The command takes a single argument
      which is the address of the sector to erase.  The address need not
      be aligned to a sector boundary; anywhere within the sector will suffice.
    */
    {"erase-sector", send_erase_sector,     1, NULL},
  };


static char *
command_generator (const char *text, int state)
{
  char *delim = strchr (rl_line_buffer, ' ');
  if (delim &&
      (0 == strncmp (rl_line_buffer, "upload-srecord", delim - rl_line_buffer)
       ||
       (0 == strncmp (rl_line_buffer, "upload-binary", delim - rl_line_buffer))))
    {
      return rl_filename_completion_function (text, state);
    }
  else
    {
      int len = strlen (text);
      int c;
      for (c = state; c < sizeof (cmds) / sizeof (cmds[0]); ++c)
        {
          if (0 == strncmp (text, cmds[c].name, len))
            return strdup (cmds[c].name);
        }

      return 0;
    }
}

static void
int_handler (int signum)
{
  rl_crlf ();
  rl_replace_line ("", 1);
  rl_forced_update_display ();
}


static char *
initialise_history (void)
{
  char *home = getenv ("HOME");
  char *histfile = NULL;
  char histfilename[] = ".upad_history";
  if (home)
    {
      histfile = safe_realloc (NULL, strlen (home) + strlen (histfilename) + 2);

      if (histfile)
	{
	  char *h = histfile;
	  h = stpcpy (h, home);
	  h = stpcpy (h, "/");
	  h = stpcpy (h, histfilename);
	  using_history ();

	  int rh = read_history (histfile);
	  if (rh != 0 && rh != ENOENT)
	    perror ("Cannot read history");
	}
    }
  return histfile;
}


static const char splash[] =
  "upad  Copyright © 2016  John Darrington\n"
  "This program comes with ABSOLUTELY NO WARRANTY; for details type\n"
  "`show warranty'.\n"
  "This is free software, and you are welcome to redistribute it\n"
  "under certain conditions; type `show copying' for details.\n";


int
run_interactive (int des)
{
  if (! isatty (fileno (stdin)))
    {
      fprintf (stderr, "Interactive mode is only possible from a terminal.\n");
      return 1;
    }

  /* Interactive mode */
  fputs (splash, stdout);

  rl_prep_terminal (0);

  struct sigaction sa;

  sa.sa_handler = int_handler;
  sigemptyset (&sa.sa_mask);
  sa.sa_flags = SA_RESTART; /* Restart functions if interrupted by handler */
  if (sigaction (SIGINT, &sa, NULL) == -1)
    perror ("Cannot set handler for SIGINT");

  rl_completion_entry_function = command_generator;

  char *histfile = initialise_history ();

  int cmd_length = 0;
  char *cmd = NULL;
  for (;;)
    {
      char *s = readline ("> ");
      if (!s)
	break;

      if (strlen (s) == 0)
	goto done;

      char *expansion = NULL;
      int result = history_expand (s, &expansion);
      if (result < 0)
	{
	  free (expansion);
	  upad_msg (MSG_ERROR, "Event not found\n");
	  goto done;
	}
      if (result > 0)
	upad_msg (MSG_ERROR, "%s\n", expansion);

      if (cmd_length < strlen (expansion))
	{
          cmd_length = strlen (expansion);
          cmd = safe_realloc (cmd, cmd_length + 1);
	}
      strcpy (cmd, expansion);
      free (expansion);

      if (!process_command (cmd, cmd_length, des))
	{
	  upad_msg (MSG_ERROR, "Bad command\n");
	}

      {
	int where = where_history ();
	HIST_ENTRY *he = history_get (where);
	if (he && (0 == strcmp (he->line, cmd)))
	  goto done;
	add_history (cmd);
      }
    done:
      free (s);
    }
  free (cmd);

  if (histfile)
    {
      int rh = write_history (histfile);
      if (rh != 0)
	{
	  perror ("Cannot write history");
	}
    }

  free (histfile);

  return 0;
}
