/*
    upad - A program for debugging, and uploading code to embedded devices.
    Copyright (C) 2016 John Darrington

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include <config.h>
#include <stdio.h>
#include <stdint.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>

#include "sep.h"

#include "link.h"

static int
recv_byte (int fp, uint8_t *b)
{
  int count = 0;
  int n;
  while ((n = read (fp, b, 1)) < 1)
    {
      /* Timeout: FIXME  Do this properly! */
      //      if (count > 10000000)
      //	return -2;
      if (n < 0 && errno != EWOULDBLOCK)
	{
	  return -1;
	}
      errno = 0;

      if (n == 1)
	{
	  break;
	}
      ++count;
    }

  return 0;
}

static int
recv_data_byte (int fp, uint8_t *byte)
{
  if (0 != recv_byte (fp, byte))
    return -1;

  if (*byte == SEP_START)
    {
      if (0 != recv_byte (fp, byte))
	return -1;
    }

  return 0;
}

int
sep_recv (int fp, uint8_t *buf, int buflen)
{
  uint8_t s;
  if (0 != recv_byte (fp, &s))
    {
      return -1;
    }
  if (s != SEP_START)
    return -1;

  uint16_t crc = 0;
  int length = 0;
  uint8_t byte;
  /* Length high byte */
  if (0 != recv_data_byte (fp, &byte))
    return -1;
  length = byte;
  length <<= 8;
  crc = crc16_update (crc, byte);

  /* Length low byte */
  if (0 != recv_data_byte (fp, &byte))
    return -1;
  crc = crc16_update (crc, byte);
  length |= byte;

  int rxc = 0;
  int i;
  for (i = 0; rxc < length; ++i)
    {
      if (0 != recv_data_byte (fp, &byte))
	return -1;

      crc = crc16_update (crc, byte);
      if (rxc < buflen)
	buf[rxc] = byte;
      rxc++;
    }

  /* CRC high byte */
  if (0 != recv_data_byte (fp, &byte))
    return -1;
  uint16_t rx_crc = byte;
  rx_crc <<= 8;

  /* CRC low byte */
  if (0 != recv_data_byte (fp, &byte))
    return -1;
  rx_crc |= byte;

  if (crc != rx_crc)
    {
      errno = 0;
      return -1;
    }

  return rxc;
}

int
sep_send (int fp, const uint8_t *payload, size_t n)
{
  uint8_t buf[MAX_PACKET_LENGTH] ;

  if (n > MAX_PAYLOAD_LENGTH)
    return -1;

  buf[0] = SEP_START;

  uint16_t crc = 0;
  int x = 1;

  /* Length high byte */
  buf[x++] = n >> 8;
  crc = crc16_update (crc, buf[x - 1]);
  if (buf[x - 1] == SEP_START)
    buf[x++] = SEP_START;

  /* Length low byte */
  buf[x++] = n;
  crc = crc16_update (crc, buf[x - 1]);
  if (buf[x - 1] == SEP_START)
    buf[x++] = SEP_START;

  int i;
  for (i = 0; i < n; ++i)
    {
      buf[x++] = payload[i];
      crc = crc16_update (crc, buf[x - 1]);
      /* If a data byte is identical to SEP_START, then pad it */
      if (buf[x - 1] == SEP_START)
	buf[x++] = SEP_START;
    }
  buf[x++] = crc >> 8;
  if (buf[x - 1] == SEP_START)
    buf[x++] = SEP_START;
  buf[x++] = crc;
  if (buf[x - 1] == SEP_START)
    buf[x++] = SEP_START;

  while (x > 0)
    {
      int n = write (fp, buf, x);
      if (n < 0 && errno != EWOULDBLOCK)
	return -1;
      errno = 0;
      x -= n;
    }

  return 0;
}
