/*
    upad - A program for debugging, and uploading code to embedded devices.
    Copyright (C) 2016 John Darrington

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/
#include <config.h>
#include <stdint.h>
#include <stdbool.h>

#include "upad.h"

#include <string.h>
#include <stdlib.h>

#include "board.h"
#include "periph/gpio.h"

#include "xtimer.h"

#include "bdc.h"


static xtimer_ticks32_t wait_1_4;
static xtimer_ticks32_t wait_3_4;
static xtimer_ticks32_t wait_3_8;
static xtimer_ticks32_t wait_1_2;

static void
send_zero (void)
{
  gpio_clear (BKGD_PIN);
  xtimer_spin (wait_3_4);
  gpio_set (BKGD_PIN);
  xtimer_spin (wait_1_4);
}

static void
send_one (void)
{
  gpio_clear (BKGD_PIN);
  xtimer_spin (wait_1_4);
  gpio_set (BKGD_PIN);
  xtimer_spin (wait_3_4);
}


static void
send_byte (uint8_t byte)
{
  bkgd_talk ();

  for (size_t bit = 0; bit < CHAR_BIT * sizeof (byte); ++bit)
    {
      if ((byte << bit) & 0x80)
	send_one ();
      else
	send_zero ();
    }

  bkgd_listen ();
}


static bool
recv_bit (void)
{
  bkgd_talk ();            // 0 sixteenths
  gpio_clear (BKGD_PIN);
  xtimer_spin (wait_1_4);  // 4 sixteenths
  bkgd_listen ();
  xtimer_spin (wait_3_8);  // 10 sixteenths
  bool res =  gpio_read (BKGD_PIN);
  xtimer_spin (wait_3_8);  // 16 sixteenths
  return res;
}

static void
recv_bits (size_t n_bits, uint32_t *bits)
{
  *bits = 0x00000000;
  for (size_t b = 0; b < n_bits; ++b)
    {
      bool bit = recv_bit ();
      if (bit)
	{
	  *bits |= 0x01 << (n_bits - 1 - b);
	}
    }
}

void
bdc_delay (void)
{
  /* Wait half a bit period */
  xtimer_spin (wait_1_2);
}

struct bdc_command
{
  uint8_t cmd;  /* Command code */
  bool index;   /* True iff the command requires an index */
  int argc;     /* Number of bytes in the argument */
  int respc;    /* Expected number of bits (not bytes!!) in response */
};

static const struct bdc_command commands[] =
  {
    {/* NOP */          0x00, false, 0, 0},
    {/* ACK_DISABLE */  0x03, false, 0, 0},
    {/* ACK_ENABLE */   0x02, false, 0, 0},
    {/* GO */           0x08, false, 0, 0},
    {/* STEP */         0x09, false, 0, 0},
    {/* BACKGROUND */   0x04, false, 0, 0},
    {/* READ_R */       0x60, true,  0, 32},
    {/* WRITE_R */      0x40, true,  4, 0},
    {/* READ_BYTE */    0x30, false, 3, 8},
    {/* DUMP_BYTE */    0x32, false, 0, 8},
    {/* READ_WORD */    0x34, false, 3, 16},
    {/* DUMP_WORD */    0x36, false, 0, 16},
    {/* READ_DWORD */   0x38, false, 3, 32},
    {/* DUMP_DWORD */   0x3A, false, 0, 32},
    {/* WRITE_BYTE */   0x10, false, 4, 0},
    {/* FILL_BYTE */    0x12, false, 1, 0},
    {/* WRITE_WORD */   0x14, false, 5, 0},
    {/* FILL_WORD */    0x16, false, 2, 0},
    {/* WRITE_DWORD */  0x18, false, 7, 0},
    {/* FILL_DWORD */   0x1A, false, 4, 0},
    {/* READ_BDCCSR */  0x2D, false, 0, 16},
    {/* WRITE_BDCCSR */ 0x0D, false, 2, 0},
    {/* ERASE_FLASH */  0x95, false, 0, 0}
  };


uint32_t
execute_command (uint8_t command_byte, int argc, const uint8_t *args, int respc)
{
  /* Start the actual communication */
  send_byte (command_byte);
  for (int i = 0; i < argc; ++i)
    {
      send_byte (args[i]);
    }
  bdc_delay ();
  uint32_t response = 0x00;
  recv_bits (respc, &response);
  return response;
}

void
bdc_go (void)
{
  const struct bdc_command *read = &commands[GO];

  execute_command (read->cmd, read->argc, NULL, read->respc);
}

void
bdc_step (void)
{
  const struct bdc_command *read = &commands[STEP];

  execute_command (read->cmd, read->argc, NULL, read->respc);
}

void
bdc_background (void)
{
  const struct bdc_command *read = &commands[BACKGROUND];

  execute_command (read->cmd, read->argc, NULL, read->respc);
}


uint16_t
bdc_read_bdccsr (void)
{
  const struct bdc_command *read = &commands[READ_BDCCSR];

  return execute_command (read->cmd, read->argc, NULL, read->respc);
}

uint32_t
bdc_read_register (int reg)
{
  const struct bdc_command *read = &commands[READ_R];

  return execute_command (read->cmd + reg, read->argc, NULL, read->respc);
}


void
bdc_write_register (int reg, uint32_t value)
{
  const struct bdc_command *write = &commands[WRITE_R];

  uint8_t args[write->respc];
  args[0] = value >> 24;
  args[1] = value >> 16;
  args[2] = value >> 8;
  args[3] = value;

  execute_command (write->cmd + reg, write->argc, args, 0);
}

void
bdc_bulk_erase (void)
{
  const struct bdc_command *flash = &commands[ERASE_FLASH];

  /* Erase must be executed twice in succession, otherwise it will be
     ignored. */
  execute_command (flash->cmd, flash->argc, 0, flash->respc);
  execute_command (flash->cmd, flash->argc, 0, flash->respc);

  /* Wait until the erase operation is complete */
  while (bdc_read_bdccsr () & 0x0100)
    bdc_delay ();
}

void
bdc_write_bdccsr (uint16_t value)
{
  const struct bdc_command *write = &commands[WRITE_BDCCSR];

  uint8_t args[write->respc];
  args[0] = value >> 8;
  args[1] = value;
  execute_command (write->cmd, write->argc, args, write->respc);
}

void
bdc_read_byte (uint32_t address, uint8_t *byte)
{
  const struct bdc_command *read = &commands[READ_BYTE];

  uint8_t args[read->respc];
  args[0] = address >> 16;
  args[1] = address >> 8;
  args[2] = address;
  *byte = execute_command (read->cmd, read->argc, args, read->respc);
}

uint16_t
bdc_read_word (uint32_t address)
{
  const struct bdc_command *read = &commands[READ_WORD];

  uint8_t args[read->respc];
  args[0] = address >> 16;
  args[1] = address >> 8;
  args[2] = address;
  return execute_command (read->cmd, read->argc, args, read->respc);
}

void
bdc_read_double_byte (uint32_t address, uint8_t *value)
{
  uint16_t x = bdc_read_word (address);
  value[0] = x >> 8;
  value[1] = x & 0xFF;
}

void
bdc_read_quad_byte (uint32_t address, uint8_t *value)
{
  uint32_t x = bdc_read_dword (address);
  value[0] = x >> 24;
  value[1] = x >> 16;
  value[2] = x >> 8;
  value[3] = x & 0xFF;
}

uint32_t
bdc_read_dword (uint32_t address)
{
  const struct bdc_command *read = &commands[READ_DWORD];

  uint8_t args[read->respc];
  args[0] = address >> 16;
  args[1] = address >> 8;
  args[2] = address;
  return execute_command (read->cmd, read->argc, args, read->respc);
}

void
bdc_dump_byte (uint8_t *data)
{
  const struct bdc_command *read = &commands[DUMP_BYTE];

  *data = execute_command (read->cmd, read->argc, NULL, read->respc);
}


uint16_t
bdc_dump_word (void)
{
  const struct bdc_command *read = &commands[DUMP_WORD];

  return execute_command (read->cmd, read->argc, NULL, read->respc);
}

uint32_t
bdc_dump_dword (void)
{
  const struct bdc_command *read = &commands[DUMP_DWORD];

  return execute_command (read->cmd, read->argc, NULL, read->respc);
}

void
bdc_dump_double_byte (uint8_t *data)
{
  uint16_t x = bdc_dump_word ();

  data[0] = x >> 8;
  data[1] = x;
}

void
bdc_dump_quad_byte (uint8_t *data)
{
  uint32_t x = bdc_dump_dword ();

  data[0] = x >> 24;
  data[1] = x >> 16;
  data[2] = x >> 8;
  data[3] = x;
}

void
bdc_write_octet (uint32_t address, uint8_t value)
{
  const struct bdc_command *write = &commands[WRITE_BYTE];

  uint8_t args[write->respc];
  args[0] = address >> 16;
  args[1] = address >> 8;
  args[2] = address;
  args[3] = value;
  execute_command (write->cmd, write->argc, args, write->respc);
}


void
bdc_write_byte (uint32_t address, const uint8_t *value)
{
  const struct bdc_command *write = &commands[WRITE_BYTE];

  uint8_t args[write->respc];
  args[0] = address >> 16;
  args[1] = address >> 8;
  args[2] = address;
  args[3] = *value;
  execute_command (write->cmd, write->argc, args, write->respc);
}

void
bdc_write_quad_byte (uint32_t address, const uint8_t *value)
{
  const struct bdc_command *write = &commands[WRITE_DWORD];

  uint8_t args[write->respc];
  args[0] = address >> 16;
  args[1] = address >> 8;
  args[2] = address;
  args[3] = value[0];
  args[4] = value[1];
  args[5] = value[2];
  args[6] = value[3];
  execute_command (write->cmd, write->argc, args, write->respc);
}

void
bdc_write_double_byte (uint32_t address, const uint8_t *value)
{
  const struct bdc_command *write = &commands[WRITE_WORD];
  uint8_t args[write->respc];
  args[0] = address >> 16;
  args[1] = address >> 8;
  args[2] = address;
  args[3] = value[0];
  args[4] = value[1];
  execute_command (write->cmd, write->argc, args, write->respc);
}

void
bdc_write_word (uint32_t address, uint16_t value)
{
  const struct bdc_command *write = &commands[WRITE_WORD];

  uint8_t args[write->respc];
  args[0] = address >> 16;
  args[1] = address >> 8;
  args[2] = address;
  args[3] = value >> 8;
  args[4] = value;
  execute_command (write->cmd, write->argc, args, write->respc);
}

void
bdc_fill_byte (const uint8_t *value)
{
  const struct bdc_command *write = &commands[FILL_BYTE];

  uint8_t args[write->respc];
  args[0] = *value;
  execute_command (write->cmd, write->argc, args, write->respc);
}

void
bdc_fill_word (uint16_t value)
{
  const struct bdc_command *write = &commands[FILL_WORD];

  uint8_t args[write->respc];
  args[0] = value >> 8;
  args[1] = value;
  execute_command (write->cmd, write->argc, args, write->respc);
}

void
bdc_write_dword (uint32_t address, uint32_t value)
{
  const struct bdc_command *write = &commands[WRITE_DWORD];

  uint8_t args[write->respc];
  args[0] = address >> 16;
  args[1] = address >> 8;
  args[2] = address;
  args[3] = value >> 24;
  args[4] = value >> 16;
  args[5] = value >> 8;
  args[6] = value;
  execute_command (write->cmd, write->argc, args, write->respc);
}

void
bdc_fill_dword (uint32_t value)
{
  const struct bdc_command *write = &commands[FILL_DWORD];

  uint8_t args[write->respc];
  args[0] = value >> 24;
  args[1] = value >> 16;
  args[2] = value >> 8;
  args[3] = value;
  execute_command (write->cmd, write->argc, args, write->respc);
}

void
bdc_fill_double_byte (const uint8_t *value)
{
  const struct bdc_command *write = &commands[FILL_WORD];

  uint8_t args[write->respc];
  args[0] = value[0];
  args[1] = value[1];
  execute_command (write->cmd, write->argc, args, write->respc);
}

void
bdc_fill_quad_byte (const uint8_t *value)
{
  const struct bdc_command *write = &commands[FILL_DWORD];

  uint8_t args[write->respc];
  args[0] = value[0];
  args[1] = value[1];
  args[2] = value[2];
  args[3] = value[3];
  execute_command (write->cmd, write->argc, args, write->respc);
}


void
bdc_calibrate (uint32_t period)
{
  double factor = 1.0;
  wait_1_4 = xtimer_ticks_from_usec (1.0 * factor * period / 4.0);
  wait_3_4 = xtimer_ticks_from_usec (3.0 * factor * period / 4.0);
  wait_3_8 = xtimer_ticks_from_usec (3.0 * factor * period / 8.0);
  wait_1_2 = xtimer_ticks_from_usec (1.0 * factor * period / 2.0);
}



/* The Sync "command" */

#include "sema.h"

static uint32_t falltime;
static uint32_t risetime;
static sema_t s1;

static void
rising (void *arg __attribute__ ((unused)))
{
  risetime  = xtimer_now_usec ();
  sema_post (&s1);
}

static void
falling (void *arg __attribute__ ((unused)))
{
  falltime  = xtimer_now_usec ();
  gpio_init_int (BKGD_PIN, GPIO_IN, GPIO_RISING, rising, 0);
}

int
bdc_sync (uint32_t *sync_time)
{
  sema_create (&s1, 0);
  bkgd_talk ();
  xtimer_ticks32_t last_wakeup = xtimer_now ();
  gpio_set (BKGD_PIN);
  xtimer_periodic_wakeup (&last_wakeup, 100);
  gpio_clear (BKGD_PIN);
  xtimer_periodic_wakeup (&last_wakeup, 1000);
  gpio_set (BKGD_PIN);
  bkgd_listen ();
  gpio_init_int (BKGD_PIN, GPIO_IN, GPIO_FALLING, falling, 0);
  gpio_irq_enable (BKGD_PIN);
  int result = sema_wait_timed (&s1, 1000);
  bkgd_listen ();
  gpio_irq_disable (BKGD_PIN);

  if (result != 0)
    return -1;

  /* sync_time is 128 x T_{BDCCSI} */
  *sync_time = risetime - falltime;

  return 0;
}
