/* screen.c - User interface management (Readline)
 *
 * Copyright (C) 2004, 2005 Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include <config.h>
#include <unistd.h>		/* POSIX: STDIN_FILENO */
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#if defined(HAVE_READLINE_READLINE_H)
# include <readline/readline.h>
#elif defined(HAVE_READLINE_H)
# include <readline.h>
#endif
#if defined(HAVE_READLINE_HISTORY_H)
# include <readline/history.h>
#elif defined(HAVE_HISTORY_H)
# include <history.h>
#endif
#include "xalloc.h"		/* Gnulib */
#include "minmax.h"		/* Gnulib */
#include "xstrndup.h"		/* Gnulib */
#include "xvasprintf.h"		/* Gnulib */
#include "quotearg.h"		/* Gnulib */
#include "gettext.h"		/* Gnulib/GNU gettext */
#define _(s) gettext(s)
#define N_(s) gettext_noop(s)
#include "microdc.h"
#include "common/error.h"
#include "common/strleftcmp.h"
#include "common/bksearch.h"
#include "common/quoting.h"

/* Flow of messages in microdc:
 *
 *  main process:
 *    warn(error.c) -> warn_writer(error.c)  -> screen_warn_writer(screen.c) -> screen_writer(screen.c) -> flag_vputf(screen.c) -> log/screen
 *    die(error.c)  -> warn_writer(error.c)  -> screen_warn_writer(screen.c) -> screen_writer(screen.c) -> flag_vputf(screen.c) -> log/screen
 *                     screen_putf(screen.c) -> flag_putf(screen.c)          -> screen_writer(screen.c) -> flag_vputf(screen.c) -> log/screen
 *  user process:
 *    warn(error.c) -> warn_writer(error.c)  -> screen_warn_writer(screen.c) -> screen_writer(screen.c) -> user_screen_writer(user.c) -> ipc
 *    die(error.c)  -> warn_writer(error.c)  -> screen_warn_writer(screen.c) -> screen_writer(screen.c) -> user_screen_writer(user.c) -> ipc
 *                     screen_putf(screen.c) -> flag_putf(screen.c)          -> screen_writer(screen.c) -> user_screen_writer(user.c) -> ipc
 *
 */

typedef enum {
    SCREEN_UNINITIALIZED,	/* History not initialized */
    SCREEN_NO_HANDLER,		/* rl_callback_handler_install not called. */
    SCREEN_RL_CLEARED,		/* Readline has been cleared. Need to redisplay after. */
    SCREEN_RL_DISPLAYED,	/* Readline will be displayed. Need to clear first. */
    SCREEN_SUSPENDED,		/* A command using the screen is running. */
} ScreenState;

static void clear_rl();
static void user_input(char *line);
static int screen_warn_writer(const char *format, va_list args);
static void flag_vputf(DCDisplayFlag flag, const char *format, va_list args);

char *log_filename = NULL;
ScreenWriter screen_writer = flag_vputf;
static char *screen_prompt;
static FILE *log_fh = NULL;
static PtrV *suspend_msgs = NULL;
static ScreenState screen_state = SCREEN_UNINITIALIZED;

/* This piece of code was snatched from lftp, and modified somewhat by me.
 * I suggest a function called rl_clear() be added to readline. The
 * function clears the prompt and everything the user has written so far on
 * the line. The cursor is positioned at the beginning of the line that
 * contained the prompt. Note: This function doesn't modify the screen_state
 * variable.
 */
static void
clear_rl()
{
    extern char *rl_display_prompt;
#if HAVE__RL_MARK_MODIFIED_LINES
    extern int _rl_mark_modified_lines;
    int old_mark = _rl_mark_modified_lines;
#endif
    int old_end = rl_end;
    char *old_prompt = rl_display_prompt;

    rl_end = 0;
    rl_display_prompt = "";
    rl_expand_prompt("");
#if HAVE__RL_MARK_MODIFIED_LINES
    _rl_mark_modified_lines = 0;
#endif

    rl_redisplay();

    rl_end = old_end;
    rl_display_prompt = old_prompt;
#if HAVE__RL_MARK_MODIFIED_LINES
    _rl_mark_modified_lines = old_mark;
#endif
    if (rl_display_prompt == rl_prompt)
        rl_expand_prompt(rl_prompt);
}

bool
set_log_file(const char *new_filename, bool verbose)
{
    if (log_fh != NULL) {
	if (fclose(log_fh) != 0)
	    warn(_("%s: Cannot close file - %s\n"), quotearg(log_filename), errstr);
	log_fh = NULL;
	free(log_filename);
	log_filename = NULL;
    }
    if (new_filename == NULL) {
	if (verbose)
	    screen_putf(_("No longer logging to file.\n"));
	return true;
    }
    log_fh = fopen(new_filename, "a");
    if (log_fh == NULL) {
	screen_putf(_("%s: Cannot open file for appending - %s\n"), quotearg(new_filename), errstr);
	return false;
    }
    log_filename = xstrdup(new_filename);
    if (verbose)
	screen_putf(_("Logging to `%s'.\n"), quotearg(new_filename));
    return true;
}

/* Readline < 5.0 disables SA_RESTART on SIGWINCH for some reason.
 * This turns it back on.
 * This was partially copied from guile.
 */
static int
fix_winch(void)
{
    struct sigaction action;

    if (sigaction(SIGWINCH, NULL, &action) >= 0) {
    	action.sa_flags |= SA_RESTART;
    	sigaction(SIGWINCH, &action, NULL); /* Ignore errors */
    }
    return 0;
}

void
get_file_dir_part(const char *word, char **dir_part, const char **file_part)
{
    const char *fp;

    fp = strrchr(word, '/');
    if (fp == NULL) {
	*dir_part = xstrdup("");
	*file_part = word;
    } else {
	for (; fp > word && fp[-1] == '/'; fp--);
	*dir_part = xstrndup(word, fp - word + 1);
	for (fp++; *fp == '/'; fp++);
	*file_part = fp;
    }
}

DCCompletionEntry *
new_completion_entry(const char *input, const char *display)
{
    DCCompletionEntry *entry = xmalloc(sizeof(DCCompletionEntry));
    entry->input = input == NULL ? NULL : xstrdup(input);
    entry->display = display == NULL ? entry->input : xstrdup(display);
    entry->input_char = ' ';
    entry->input_format = "%s";
    entry->input_single_format = NULL;
    entry->input_single_full_format = NULL;
    entry->inhibit_quote = false;
    return entry;
}

void
free_completion_entry(DCCompletionEntry *entry)
{
    if (entry->display != entry->input)
        free(entry->display);
    free(entry->input);
    free(entry);
}

static char **
attempted_completion(const char *text, int start, int end)
{
    char **matches = NULL;
    char *word;
    bool quoted;
    PtrV *results = ptrv_new();

    /* Readline is strange. When completing a quoted word,
     * sometimes the start position is just after the
     * first quote. And sometimes it is on the first quote.
     * We need to manually detect which case it is, because
     * we need the quote.
     */

    quoted = char_is_quoted(rl_line_buffer, start)
               && ((start > 0 && rl_line_buffer[start-1] == '"') || rl_line_buffer[start] == '"');
    if (quoted && start > 0 && rl_line_buffer[start-1] == '"')
        start--;
    word = dequote_words_full(rl_line_buffer+start, false, false, true, true, rl_line_buffer+end);

    /* Do not attempt readline's default filename completion if our
     * completors fails to return any results.
     */
    rl_attempted_completion_over = 1;
    default_completion_selector(rl_line_buffer,start,end,word,results);

//printf("%d:%d [%s:%s]\n", start,end,text,rl_line_buffer);
    if (results->cur == 1) {
        DCCompletionEntry *entry = results->buf[0];
        char *match;

        if (entry->input_single_full_format != NULL && strcmp(word, entry->input) == 0) {
            match = xasprintf(entry->input_single_full_format, entry->input);
        } else if (entry->input_single_format != NULL) {
	    match = xasprintf(entry->input_single_format, entry->input);
	} else {
            match = xasprintf(entry->input_format, entry->input);
        }
	matches = xmalloc(sizeof(char *) * 2);
        matches[0] = quote_word_full(match, quoted, entry->inhibit_quote, ";", "#", false, true, true, true);

//printf("\n%d [[%s]=>[%s]] : %s\n", quoted, match, matches[0], word);
        matches[1] = NULL;
        rl_completion_append_character = entry->input_char;
        /*rl_completion_suppress_quote = entry->inhibit_quote;*/
        free_completion_entry(entry);
        free(match);
    } else if (results->cur > 1) {
        char *match = NULL;
        int match_len = 0;
        int c;

    	matches = xmalloc(sizeof(char *) * (results->cur + 2));
        for (c = 0; c < results->cur; c++) {
            DCCompletionEntry *entry = results->buf[c];

            /* Determine common leading characters for all matches. */
            if (c == 0) {
                match = xstrdup(entry->input);
                match_len = strlen(match);
            } else {
                int d;
                char c1, c2;
                for (d = 0; (c1 = match[d]) && (c2 = entry->input[d]); d++) {
	            if (c1 != c2)
	    	        break;
                }
                match_len = MIN(match_len, d);
            }

	    matches[c+1] = quote_word_full(entry->display, false, true, "", "", false, true, true, false);
	    free_completion_entry(entry);
        }

        match[match_len] = '\0';
        rl_completion_append_character = ' ';
	matches[0] =  quote_word_full(match, quoted, false, ";", "#", false, true, true, true);
        matches[c+1] = NULL;
        free(match);
    }

    ptrv_free(results);
    free(word);
    return matches;
}

/* This function is called via the warn_writer variable by warn() and
 * die() in error.c to print messages on screen properly.
 */
static int
screen_warn_writer(const char *format, va_list args)
{
    screen_writer(DC_DF_COMMON, format, args);
    return 0;
}

static void
flag_vputf(DCDisplayFlag flag, const char *format, va_list args)
{
    if (display_flags & flag) {
        if (screen_state == SCREEN_SUSPENDED) {
            ptrv_append(suspend_msgs, xvasprintf(format, args));
        } else {
            if (screen_state == SCREEN_RL_DISPLAYED) {
                clear_rl();
                screen_state = SCREEN_RL_CLEARED;
            }
            vprintf(format, args);
        }
    }
    if (log_fh != NULL && log_flags & flag)
        vfprintf(log_fh, format, args);
}

void
flag_putf(DCDisplayFlag flag, const char *format, ...)
{
    va_list args;

    va_start(args, format);
    screen_writer(flag, format, args);
    va_end(args);
}

/* This function is called by readline whenever the user has
 * entered a full line (usually by pressing enter).
 */
static void
user_input(char *line)
{
    /* Readline has already made way for us. */
    screen_state = SCREEN_RL_CLEARED;

    if (log_fh != NULL)
	fprintf(log_fh, "> %s\n", line == NULL ? "(null)" : line);
    
    if (line == NULL) {
        /* Ctrl+D was pressed on an empty line. */
        screen_putf("exit\n");
        running = false;
    } else if (line[0] != '\0') {
        add_history(line);
	/* XXX: handle input differently
	 * !shell_cmd
	 * /cmd
	 * depending on context: msg, say, cmd
	 */
        command_execute(line);
    }

    /* As soon as we exit this function, a new prompt will be displayed.
     */
    if (running) {
        if (screen_state != SCREEN_SUSPENDED)
            screen_state = SCREEN_RL_DISPLAYED;
    } else {
        rl_callback_handler_remove();
        FD_CLR(STDIN_FILENO, &read_fds);
        screen_state = SCREEN_NO_HANDLER;
    }
}

/* Move down one line (rl_on_new_line+redisplay), print a new prompt and
 * empty the input buffer. Unlike rl_clear this doesn't erase anything on
 * screen.
 *
 * This is usually called when a user presses Ctrl+C.
 * XXX: Should find a better way to do this (see lftp or bash).
 */
void
screen_erase_and_new_line(void)
{
    if (screen_state == SCREEN_RL_DISPLAYED) {
        rl_callback_handler_remove();
        putchar('\n');
        rl_callback_handler_install(screen_prompt, user_input);
    }
}

/* Suspend screen operations so that nothing in here
 * uses standard in or standard out. This is necessary when
 * running a system command in the terminal.
 */
void
screen_suspend(void)
{
    if (screen_state == SCREEN_RL_DISPLAYED || screen_state == SCREEN_RL_CLEARED) {
        rl_callback_handler_remove();
	FD_CLR(STDIN_FILENO, &read_fds);
        if (screen_state == SCREEN_RL_DISPLAYED)
            putchar('\n');
	suspend_msgs = ptrv_new();
	screen_state = SCREEN_SUSPENDED;
    }
}

/* Wake up screen after suspension.
 */
void
screen_wakeup(bool print_newline_first)
{
    if (screen_state == SCREEN_SUSPENDED) {
        int c;
        if (print_newline_first)
            putchar('\n');
        for (c = 0; c < suspend_msgs->cur; c++)
            fputs((char *) suspend_msgs->buf[c], stdout);
	ptrv_foreach(suspend_msgs, free);
	ptrv_free(suspend_msgs);
        screen_state = SCREEN_NO_HANDLER;
    }
}

/* Finish screen management. Usually called from main_finish.
 */
void
screen_finish(void)
{
    if (screen_state == SCREEN_SUSPENDED) {
        int c;
        for (c = 0; c < suspend_msgs->cur; c++)
            fputs((char *) suspend_msgs->buf[c], stdout);
	ptrv_foreach(suspend_msgs, free);
	ptrv_free(suspend_msgs);
    } else if (screen_state > SCREEN_NO_HANDLER) {
        rl_callback_handler_remove();
        if (screen_state == SCREEN_RL_DISPLAYED)
            putchar('\n');
	FD_CLR(STDIN_FILENO, &read_fds);
    }

    if (screen_state >= SCREEN_NO_HANDLER) {
    	char *path;

	/* Save history */
    	get_package_file("history", &path);
    	if (mkdirs_for_file(path, false) >= 0) {
	    if (write_history(path) != 0)
		warn(_("%s: Cannot write history - %s\n"), quotearg(path), errstr);
	}
	free(path);

    	//rl_basic_word_break_characters = ?
	//rl_completer_word_break_characters = ?
    	//rl_completion_display_matches_hook = ?
	rl_attempted_completion_function = NULL;
	//rl_char_is_quoted_p = NULL;
    	rl_pre_input_hook = NULL;

	warn_writer = default_warn_writer;
        screen_state = SCREEN_UNINITIALIZED;
        free(screen_prompt);

	set_log_file(NULL, false);
    }
}

/* Prepare the screen prior to waiting for events with select/poll/epoll.
 * Redisplay the prompt if it was cleared by a call to screen_(v)put(f).
 */
void
screen_prepare(void)
{
    if (screen_state == SCREEN_SUSPENDED)
        return;

    if (screen_state == SCREEN_UNINITIALIZED) {
    	char *path;

    	screen_state = SCREEN_NO_HANDLER;
	warn_writer = screen_warn_writer;
	screen_prompt = xasprintf("%s> ", PACKAGE);

	rl_readline_name = PACKAGE;
	rl_attempted_completion_function = attempted_completion;
	rl_completer_quote_characters = "\"";
	/* rl_basic_word_break_characters = ... */
	rl_special_prefixes = "\"";
	rl_completer_word_break_characters = " \t\n;\"";  // also \"
	/* rl_filename_quote_characters = " \t\n\\\"'>;|&()*?[]~!"; */
	/* rl_filename_quoting_function = quote_argument; */
	/* rl_filename_dequoting_function = dequote_argument_rl; */
	rl_char_is_quoted_p = char_is_quoted;
    	rl_pre_input_hook = fix_winch;

        using_history();
    	get_package_file("history", &path);
	if (read_history(path) != 0 && errno != ENOENT)
    	    warn(_("%s: Cannot read history - %s\n"), quotearg(path), errstr);
    }
    if (screen_state == SCREEN_NO_HANDLER) {
        rl_callback_handler_install(screen_prompt, user_input);
        FD_SET(STDIN_FILENO, &read_fds);
    }
    else if (screen_state == SCREEN_RL_CLEARED) {
        rl_set_prompt(screen_prompt);
        rl_redisplay();
    }
    screen_state = SCREEN_RL_DISPLAYED;
}

/* This function is called from the main loop when there's input for us to
 * read on stdin.
 */
void
screen_read_input(void)
{
    rl_callback_read_char();
}

/* Return the size of the screen.
 */
void
screen_get_size(int *rows, int *cols)
{
    int dummy;
    rl_get_screen_size(rows ? rows : &dummy, cols ? cols : &dummy);
}

void
set_screen_prompt(const char *prompt, ...)
{
    va_list args;

    if (screen_prompt != NULL)
	free(screen_prompt);
    va_start(args, prompt);
    screen_prompt = xvasprintf(prompt, args);
    va_end(args);
    if (screen_state != SCREEN_SUSPENDED)
        rl_set_prompt(screen_prompt);
}

/* Look up completion alternatives from a sorted list using binary search.
 */
void
sorted_list_completion_generator(const char *base, PtrV *results, void *items,
                                 size_t item_count, size_t item_size,
                                 size_t key_offset)
{
    const void *item;
    const void *last_item;

    if (bksearchrange(base, items, item_count, item_size,
	              key_offset, (comparison_fn_t) strleftcmp,
	              &item, &last_item)) {
        while (item <= last_item) {
	    char *key = xstrdup(*(const char **) (((const char *) item) + key_offset));
	    ptrv_append(results, new_completion_entry(key, NULL));
	    item = (const void *) (((const char *) item) + item_size);
        }
    }
}
