/* ipc.c - Generalized interprocess communication 
 *
 * Copyright (C) 2004-2005 Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <stdbool.h>		/* Gnulib/POSIX/C99 */
#include <stdint.h>		/* Gnulib/POSIX/C99 */
#include <stdarg.h>		/* C89 */
#include <string.h>		/* C89 */
#include <stdlib.h>		/* C89 */
#include "full-write.h"		/* Gnulib */
#include "full-read.h"		/* Gnulib */
#include "xalloc.h"		/* Gnulib */
#include "ipc_internal.h"
#include "ipc.h"

static bool
ipc_read(IPC *ipc, void *data, uint32_t datalen)
{
    ssize_t res;

    res = full_read(ipc->fd, data, datalen);
    if (res < datalen) {
    	ipc->handle_error(ipc, IPC_ERROR_READ, res, ipc->userdata);
    	return false;
    }
    return true;
}

static bool
ipc_read_checked(IPC *ipc, void *data, uint32_t datalen)
{
    if (datalen > ipc->header.size) {
        ipc->handle_error(ipc, IPC_ERROR_DATA, 0, ipc->userdata);
        return false;
    }
    ipc->header.size -= datalen;
    return ipc_read(ipc, data, datalen);
}

static bool
ipc_write(IPC *ipc, void *data, uint32_t datalen)
{
    ssize_t res;

    res = full_write(ipc->fd, data, datalen);
    if (res < datalen) {
    	ipc->handle_error(ipc, IPC_ERROR_WRITE, res, ipc->userdata);
    	return false;
    }
    return true;
}

bool
ipc_put(IPC *ipc, uint32_t id, ...)
{
    IPCMessageHeader header;
    bool result = true;
    va_list ap;

    header.id = id;
    header.size = 0;
    header.arg_count = 0;

    va_start(ap, id);
    for (;;) {
    	IPCArgType type = va_arg(ap, IPCArgType);
	IPCArgValue value;

	if (type == IPC_END)
	    break;
    	header.size += sizeof(type);
	header.arg_count++;

	switch (type) {
	case IPC_INT64:
	    va_arg(ap, int64_t);
	    header.size += sizeof(value.t_int64);
	    break;
	case IPC_INT32:
	    va_arg(ap, int32_t);
	    header.size += sizeof(value.t_int32);
	    break;
	case IPC_INT:
	    va_arg(ap, int);
	    header.size += sizeof(value.t_int);
	    break;
	case IPC_BOOL:
	    va_arg(ap, int); /* bool promotes to int */
	    header.size += sizeof(value.t_bool);
	    break;
	case IPC_STR:
	    value.t_str.chars = va_arg(ap, char *);
	    header.size += sizeof(value.t_str.len);
	    if (value.t_str.chars != NULL)
	    	header.size += strlen(value.t_str.chars);
	    break;
	default:
    	    ipc->handle_error(ipc, IPC_ERROR_DATA, -1, ipc->userdata);
	    result = false;
	    break;
	}
    }
    va_end(ap);
    if (!result)
    	return false;

    if (!ipc_write(ipc, &header, sizeof(header)))
    	return false;

    va_start(ap, id);
    while (result) {
    	IPCArgType type = va_arg(ap, IPCArgType);
	IPCArgValue value;

    	if (type == IPC_END)
	    break;
    	result = ipc_write(ipc, &type, sizeof(type));
    	if (!result)
	    break;

	switch (type) {
	case IPC_INT64:
	    value.t_int64 = va_arg(ap, int64_t);
    	    result = ipc_write(ipc, &value.t_int64, sizeof(value.t_int64));
	    break;
	case IPC_INT32:
	    value.t_int32 = va_arg(ap, int32_t);
    	    result = ipc_write(ipc, &value.t_int32, sizeof(value.t_int32));
	    break;
	case IPC_INT:
	    value.t_int = va_arg(ap, int);
    	    result = ipc_write(ipc, &value.t_int, sizeof(value.t_int));
	    break;
	case IPC_BOOL:
	    value.t_bool = va_arg(ap, int); /* bool promotes to int */
    	    result = ipc_write(ipc, &value.t_bool, sizeof(value.t_bool));
	    break;
	case IPC_STR:
	    value.t_str.chars = va_arg(ap, char *);
	    if (value.t_str.chars == NULL) {
	    	value.t_str.len = -1;
    	    	result = ipc_write(ipc, &value.t_str.len, sizeof(value.t_str.len));
	    } else {
	    	value.t_str.len = strlen(value.t_str.chars);
    	    	result = ipc_write(ipc, &value.t_str.len, sizeof(value.t_str.len));
		if (result)
		    result = ipc_write(ipc, value.t_str.chars, value.t_str.len);
	    }
	    break;
	default:
    	    ipc->handle_error(ipc, IPC_ERROR_DATA, id, ipc->userdata);
	    result = false;
	    break;
	}
    }
    va_end(ap);

    return result;
}

bool
ipc_get_id(IPC *ipc, uint32_t *id)
{
    if (!ipc->has_header) {
	if (!ipc_read(ipc, &ipc->header, sizeof(ipc->header)))
	    return false;
	ipc->has_header = true;
    }
    if (id != NULL)
    	*id = ipc->header.id;
    return true;
}

bool
ipc_handle(IPC *ipc)
{
    if (!ipc_get_id(ipc, NULL))
	return false;
    if (!ipc->handle_message(ipc, ipc->header.id, ipc->userdata))
	return false;
    ipc->has_header = false;
    return true;
}

static void
free_arg_values(IPCArgType *types, IPCArgValue *values, uint32_t count)
{
    uint32_t c;

    for (c = 0; c < count; c++) {
	switch (types[c]) {
    	case IPC_STR:
	    free(values[c].t_str.chars);
	    break;
	default:
	    break;
	}
    }
}

bool
ipc_get(IPC *ipc, uint32_t id, ...)
{
    bool result = true;
    uint32_t c;
    va_list ap;

    for (;;) {
    	if (!ipc_get_id(ipc, NULL))
	    return false;
	if (ipc->header.id == id)
	    break;
	if (!ipc->handle_message(ipc, ipc->header.id, ipc->userdata))
	    return false;
    	ipc->has_header = false;
    }
    ipc->has_header = false;

    IPCArgValue values[ipc->header.arg_count];
    IPCArgType types[ipc->header.arg_count];
    IPCArgRef refs[ipc->header.arg_count];

    for (c = 0; c < ipc->header.arg_count; c++) {
    	IPCArgType type;

	result = ipc_read_checked(ipc, &type, sizeof(IPCArgType));
	if (!result)
	    break;
	types[c] = type;

	switch (type) {
	case IPC_INT64:
	    result = ipc_read_checked(ipc, &values[c].t_int64, sizeof(values[c].t_int64));
	    break;
	case IPC_INT32:
	    result = ipc_read_checked(ipc, &values[c].t_int32, sizeof(values[c].t_int32));
	    break;
	case IPC_INT:
	    result = ipc_read_checked(ipc, &values[c].t_int, sizeof(values[c].t_int));
	    break;
	case IPC_BOOL:
	    result = ipc_read_checked(ipc, &values[c].t_bool, sizeof(values[c].t_bool));
	    break;
	case IPC_STR:
	    result = ipc_read_checked(ipc, &values[c].t_str.len, sizeof(values[c].t_str.len));
	    if (result) {
	    	if (values[c].t_str.len == -1) {
		    values[c].t_str.chars = NULL;
		} else {
    		    values[c].t_str.chars = xmalloc(values[c].t_str.len+1);
		    values[c].t_str.chars[values[c].t_str.len] = '\0';
		    result = ipc_read_checked(ipc, values[c].t_str.chars, values[c].t_str.len);
		    if (!result)
			free(values[c].t_str.chars);
		}
	    }
    	    break;
	default:
    	    ipc->handle_error(ipc, IPC_ERROR_DATA, 0, ipc->userdata);
	    result = false;
	    break;
	}
    }
    if (result && ipc->header.size != 0) {
    	ipc->handle_error(ipc, IPC_ERROR_DATA, 0, ipc->userdata);
	result = false;
    }

    if (!result) {
	free_arg_values(types, values, c);
	return false;
    }

    va_start(ap, id);
    for (c = 0; c < ipc->header.arg_count; c++) {
    	IPCArgType type = va_arg(ap, IPCArgType);

	if (type != types[c]) {
    	    ipc->handle_error(ipc, IPC_ERROR_DATA, 0, ipc->userdata);
	    result = false;
	    break;
	}

	switch (type) {
	case IPC_INT64:
	    refs[c].t_int64 = va_arg(ap, int64_t *);
	    break;
	case IPC_INT32:
	    refs[c].t_int32 = va_arg(ap, int32_t *);
	    break;
	case IPC_INT:
	    refs[c].t_int = va_arg(ap, int *);
	    break;
	case IPC_BOOL:
	    refs[c].t_bool = va_arg(ap, bool *);
	    break;
	case IPC_STR:
	    refs[c].t_str = va_arg(ap, char **);
	    break;
	default:
	    break;
	}
    }
    if (result && va_arg(ap, IPCArgType) != IPC_END) {
    	ipc->handle_error(ipc, IPC_ERROR_DATA, 0, ipc->userdata);
	result = false;
    }
    va_end(ap);

    if (!result) {
	free_arg_values(types, values, ipc->header.arg_count);
	return false;
    }

    for (c = 0; c < ipc->header.arg_count; c++) {
	switch (types[c]) {
	case IPC_INT64:
	    *(refs[c].t_int64) = values[c].t_int64;
	    break;
	case IPC_INT32:
	    *(refs[c].t_int32) = values[c].t_int32;
	    break;
	case IPC_INT:
	    *(refs[c].t_int) = values[c].t_int;
	    break;
	case IPC_BOOL:
	    *(refs[c].t_bool) = values[c].t_bool;
	    break;
	case IPC_STR:
	    *(refs[c].t_str) = values[c].t_str.chars;
	    break;
	default:
	    break;
	}
    }

    return true;
}

IPC *
ipc_new(int fd, IPCMessageFunction handle_message, IPCErrorFunction handle_error, void *userdata)
{
    IPC *ipc = xmalloc(sizeof(IPC));
    
    ipc->fd = fd;
    ipc->handle_message = handle_message;
    ipc->handle_error = handle_error;
    ipc->has_header = false;
    ipc->userdata = userdata;

    return ipc;
}

void
ipc_free(IPC *ipc)
{
    free(ipc);
}
