/* fs.c - Local and remote file system management
 *
 * Copyright (C) 2004-2005 Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <assert.h>		/* ? */
#include <sys/types.h>		/* ? */
#include <sys/stat.h>		/* ? */
#include <unistd.h>		/* POSIX */
#include <fcntl.h>		/* ? */
#include <stdlib.h>		/* C89 */
#include <dirent.h>		/* ? */
#include <sys/types.h>		/* ? */
#include <inttypes.h>		/* ? */
#include "full-read.h"		/* Gnulib */
#include "xalloc.h"		/* Gnulib */
#include "minmax.h"		/* Gnulib */
#include "full-write.h"		/* Gnulib */
#include "xvasprintf.h"		/* Gnulib */
#include "gettext.h"		/* Gnulib/GNU gettext */
#define _(s) gettext(s)
#define N_(s) gettext_noop(s)
#include "common/common.h"
#include "common/error.h"
#include "common/substrcmp.h"
#include "common/intparse.h"
#include "common/strbuf.h"
#include "common/strleftcmp.h"
#include "microdc.h"

#define MAX_DIR_DEPTH 32

DCFileList *our_filelist = NULL;

static DCFileList *
new_file_node(const char *name, DCFileType type) /* XXX: add DCFileList *parent to this prototype? */
{
    DCFileList *node;

    node = xmalloc(sizeof(DCFileList));
    node->parent = NULL;
    node->name = xstrdup(name);
    node->type = type;
    switch (node->type) {
    case DC_TYPE_DIR:
    	node->u.dir.children = hmap_new();
	node->u.dir.totalsize = 0;
	break;
    case DC_TYPE_REG:
    	node->u.reg.size = 0;
    default:
    	break;
    }
    return node;
}

void
filelist_free(DCFileList *node)
{
    switch (node->type) {
    case DC_TYPE_REG:
    	break;
    case DC_TYPE_DIR:
    	hmap_foreach(node->u.dir.children, filelist_free);
	hmap_free(node->u.dir.children);
    	break;
    }
    
    free(node->name);
    free(node);
}

DCFileList *
parse_decoded_dclst(char *decoded, uint32_t decoded_len)
{
    DCFileList *node;
    uint32_t c;
    PtrV *dirs;

    dirs = ptrv_new();
    node = new_file_node("", DC_TYPE_DIR);
    ptrv_append(dirs, node);
    for (c = 0; c < decoded_len; c++) {
    	DCFileList *oldnode;
	char *name;
    	int depth;

	for (; c < decoded_len && decoded[c] == '\n'; c++);
    	depth = 1;
	for (; c < decoded_len && decoded[c] == '\t'; c++)
	    depth++;
	if (c >= decoded_len)
	    break; /* Premature end */
	if (decoded[c] == '\r') {
	    c++; /* skip LF */
	    continue; /* Skipping bad line */
	}

    	name = decoded + c;
    	for (; c < decoded_len && decoded[c] != '\r' && decoded[c] != '|'; c++);
	if (c >= decoded_len)
	    break;
	if (decoded[c] == '|') {
	    char *sizestr;
	    uint64_t size;

	    decoded[c] = '\0';
    	    sizestr = decoded+c+1;
    	    for (c++; c < decoded_len && decoded[c] != '\r'; c++);
	    if (c >= decoded_len)
	    	break; /* Premature end */
	    decoded[c] = '\0';
    	    c++; /* skip LF */
	    if (!parse_uint64(sizestr, &size))
	    	continue; /* Skipping bad line */

	    node = new_file_node(name, DC_TYPE_REG);
	    node->u.reg.size = size;
	} else {
	    decoded[c] = '\0';
    	    c++; /* skip LF */
	    node = new_file_node(name, DC_TYPE_DIR);
	}

	if (depth < dirs->cur)
	    ptrv_remove_range(dirs, depth, dirs->cur);
	oldnode = dirs->buf[dirs->cur-1];
	hmap_put(oldnode->u.dir.children, node->name, node);

	if (node->type == DC_TYPE_REG) {
	    DCFileList *t;
	    for (t = oldnode; t != NULL; t = t->parent)
	    	t->u.dir.totalsize += node->u.reg.size;
	}
	node->parent = oldnode;
	if (node->type == DC_TYPE_DIR)
	    ptrv_append(dirs, node);
    }

    node = dirs->buf[0];
    ptrv_free(dirs); /* ignore non-empty */
    return node;
}

DCFileList *
filelist_lookup(DCFileList *node, const char *filename)
{
    HMapIterator it;
    const char *end;

    if (*filename != '/')
    	return NULL;
    for (filename++; *filename == '/'; filename++);
    end = strchr(filename, '/');
    if (end == NULL) {
	if (!*filename)
	    return node;
    	end = filename + strlen(filename);
    }
    if (node->type != DC_TYPE_DIR)
    	return NULL;
    if (substrcmp("..", filename, end-filename) == 0) {
    	DCFileList *parent = (node->parent == NULL ? node : node->parent);
       	return (*end == '\0' ? parent : filelist_lookup(parent, end));
    }
    if (substrcmp(".", filename, end-filename) == 0)
    	return (*end == '\0' ? node : filelist_lookup(node, end));

    hmap_iterator(node->u.dir.children, &it);
    while (it.has_next(&it)) {
    	DCFileList *subnode = it.next(&it);
    	if (substrcmp(subnode->name, filename, end-filename) == 0)
	    return (*end == '\0' ? subnode : filelist_lookup(subnode, end));
    }

    return NULL;
}

char *
filelist_get_path(DCFileList *node)
{
    StrBuf *sb;

    if (node->parent == NULL)
	return xstrdup("/"); /* root */

    sb = strbuf_new();
    while (node->parent != NULL) {
	strbuf_prepend(sb, node->name);
	strbuf_prepend_char(sb, '/');
    	node = node->parent;
    }

    return strbuf_free_to_string(sb);
}

static int
file_node_compare(const void *i1, const void *i2)
{
    const DCFileList *f1 = *(const DCFileList **) i1;
    const DCFileList *f2 = *(const DCFileList **) i2;

    if (f1->type != f2->type)
	return f1->type - f2->type;

    return strcmp(f1->name, f2->name);
}

static DCFileList **
get_sorted_file_list(DCFileList *node, uint32_t *out_count)
{
    HMapIterator it;
    DCFileList **items;
    uint32_t count;
    uint32_t c;

    assert(node->type == DC_TYPE_DIR);
    count = hmap_size(node->u.dir.children);
    items = xmalloc((count+1) * sizeof(DCFileList *));
    hmap_iterator(node->u.dir.children, &it);
    for (c = 0; c < count; c++)
        items[c] = it.next(&it);
    items[count] = NULL;
    qsort(items, count, sizeof(DCFileList *), file_node_compare);

    if (out_count != NULL)
        *out_count = count;

    return items;
}

void
filelist_list_recursively(DCFileList *node, char *basepath, int skip)
{
    if (node->type == DC_TYPE_DIR) {
	char *subpath;
	DCFileList **items;
	uint32_t c;

        subpath = skip > 0 ? xstrdup(basepath) : catfiles(basepath, node->name);
        items = get_sorted_file_list(node, NULL);
	for (c = 0; items[c] != NULL; c++)
	    filelist_list_recursively(items[c], subpath, MAX(skip-1, 0));
	free(items);
	free(subpath);
    } else {
	char *tmp;

	tmp = catfiles(basepath, node->name);
	screen_putf("%7" PRIu64 "M %s%s\n", node->u.reg.size/(1024*1024), tmp, "");
	free(tmp);
    }
}

void
filelist_list(DCFileList *node, bool long_mode)
{
    uint32_t maxlen;
    uint64_t maxsize;

    maxlen = 0;
    maxsize = 0;
    if (node->type == DC_TYPE_DIR) {
        HMapIterator it;

	hmap_iterator(node->u.dir.children, &it);
	while (it.has_next(&it)) {
    	    DCFileList *subnode = it.next(&it);

	    switch (subnode->type) {
	    case DC_TYPE_REG:
		maxsize = max(maxsize, subnode->u.reg.size);
        	maxlen = max(maxlen, strlen(subnode->name));
        	break;
            case DC_TYPE_DIR:
		maxsize = max(maxsize, subnode->u.dir.totalsize);
        	maxlen = max(maxlen, strlen(subnode->name)+1);
        	break;
            }
	}
    } else {
    	maxsize = node->u.reg.size;
    	maxlen = strlen(node->name);
    }

    if (long_mode) {
        char *format;

        format = xasprintf("%%%dlluM %%s%%s\n", maxsize == 0 ? 1 : ilog10(maxsize/(1024*1024)));
	if (node->type == DC_TYPE_DIR) {
	    DCFileList **items;
	    uint32_t c;

            items = get_sorted_file_list(node, NULL);
	    for (c = 0; items[c] != NULL; c++) {
		switch (items[c]->type) {
		case DC_TYPE_REG:
    		    screen_putf(format, items[c]->u.reg.size/(1024*1024), items[c]->name, "");
		    break;
		case DC_TYPE_DIR:
    		    screen_putf(format, items[c]->u.dir.totalsize/(1024*1024), items[c]->name, "/");
		    break;
		}
	    }
	    free(items);
	} else {
	    screen_putf(format, node->u.reg.size, node->name, "");
	}
	free(format);
    } else {
    	if (node->type == DC_TYPE_DIR) {
	    DCFileList **items;
            int cols;
	    int rows;
	    int row;
            int per_row;
            uint32_t count;

            items = get_sorted_file_list(node, &count);
            screen_get_size(NULL, &cols);
            per_row = MAX(1, (cols+2)/(maxlen+2));
	    rows = (count/per_row) + (count%per_row != 0);

	    for (row = 0; row < rows; row++) {
		uint32_t c;

		for (c = row; c < count; c += rows) {
		    DCFileList *item = items[c];
		    int extlen = 0;
		    int d;
    
		    switch (item->type) {
		    case DC_TYPE_REG:
			extlen = 0;
			screen_putf("%s", item->name);
			break;
		    case DC_TYPE_DIR:
			extlen = 1;
			screen_putf("%s/", item->name);
			break;
		    }
		    if (c+rows < count) {
			for (d = maxlen-strlen(items[c]->name)-extlen+2; d > 0; d--)
			    screen_putf(" ");
		    }
		}
		screen_putf("\n");
	    }
	    free(items);
	} else {
    	    screen_putf("%s\n", node->name);
	}
    }
}

DCFileList *
filelist_open(const char *filename)
{
    struct stat st;
    char *contents;
    char *decoded;
    uint32_t decoded_len;
    int fd;
    DCFileList *root;
    ssize_t res;

    if (stat(filename, &st) < 0) {
    	screen_putf(_("%s: Cannot get file status - %s\n"), filename, errstr);
	return NULL;
    }
    contents = malloc(st.st_size);
    if (contents == NULL) {
    	screen_putf("%s: %s\n", filename, errstr);
	return NULL;
    }
    fd = open(filename, O_RDONLY);
    if (fd < 0) {
    	screen_putf(_("%s: Cannot open file for reading - %s\n"), filename, errstr);
	return NULL;
    }
    res = full_read(fd, contents, st.st_size);
    if (res < st.st_size) {
    	if (res < 0)
    	    screen_putf(_("%s: Cannot read from file - %s\n"), filename, errstr);
	else
	    screen_putf(_("%s: Premature end of file\n"), filename);	/* XXX: really: file was truncated? */
	free(contents);
	if (close(fd) < 0)
    	    screen_putf(_("%s: Cannot close file - %s\n"), filename, errstr);
	return NULL;
    }
    if (close(fd) < 0)
    	screen_putf(_("%s: Cannot close file - %s\n"), filename, errstr);
    decoded = huffman_decode(contents, st.st_size, &decoded_len);
    free(contents);
    if (decoded == NULL) {
    	screen_putf(_("%s: Invalid data, cannot decode\n"), filename);
	return NULL;
    }
    root = parse_decoded_dclst(decoded, decoded_len);
    free(decoded);
    return root;
}

static void
dir_to_filelist(DCFileList *parent, const char *path)
{
    struct dirent *ep;
    DIR *dp;

    dp = opendir(path);
    if (dp == NULL) {
    	screen_putf(_("%s: Cannot open directory - %s\n"), path, errstr);
	return;
    }
    errno = 0;
    while ((ep = readdir(dp)) != NULL) {
    	struct stat st;
	char *fullname;

    	if (IS_CURRENT_DIR(ep->d_name) || IS_PARENT_DIR(ep->d_name))
	    continue;

	/* If we ran into looped symlinked dirs, stat will stop (errno=ELOOP). */

    	fullname = catfiles(path, ep->d_name);
    	if (stat(fullname, &st) < 0) {
	    screen_putf(_("%s: Cannot get file status - %s\n"), fullname, errstr);
	    free(fullname);
	    continue;
	}
	if (S_ISDIR(st.st_mode)) {
	    DCFileList *node = new_file_node(ep->d_name, DC_TYPE_DIR);
	    node->parent = parent;
	    dir_to_filelist(node, fullname);
	    parent->u.dir.totalsize += node->u.dir.totalsize;
	    hmap_put(parent->u.dir.children, node->name, node);
	}
	else if (S_ISREG(st.st_mode)) {
	    DCFileList *node = new_file_node(ep->d_name, DC_TYPE_REG);
	    node->parent = parent;
	    node->u.reg.size = st.st_size;
	    parent->u.dir.totalsize += node->u.reg.size;
	    hmap_put(parent->u.dir.children, node->name, node);
	}
	else {
	    screen_putf(_("%s: Not a regular file or directory, ignoring\n"), fullname);
	}
	free(fullname);
    }
    if (errno != 0)
    	screen_putf(_("%s: Cannot read directory - %s\n"), path, errstr);
    if (closedir(dp) < 0)
    	screen_putf(_("%s: Cannot close directory - %s\n"), path, errstr);
}

static void
filelist_to_string(DCFileList *node, StrBuf *sb, int level)
{
    if (level != 0)
    	strbuf_append_char_n(sb, level-1, '\t');

    if (node->type == DC_TYPE_REG) {
   	strbuf_appendf(sb, "%s|%" PRId64 "\r\n", node->name, node->u.reg.size);
    } else {
    	HMapIterator it;

    	if (level != 0)
   	    strbuf_appendf(sb, "%s\r\n", node->name);
	hmap_iterator(node->u.dir.children, &it);
	while (it.has_next(&it)) {
	    DCFileList *subnode = it.next(&it);
	    filelist_to_string(subnode, sb, level+1);
	}
    }
}

bool
filelist_create(const char *basedir)
{
    StrBuf *sb;
    char *indata;
    char *outdata;
    int fd;
    char *filename;
    uint32_t len;
    struct stat st;
    int i;
    DCFileList *root;

    root = new_file_node("", DC_TYPE_DIR);

    sb = strbuf_new();
    if (basedir != NULL) {
    	screen_putf(_("Scanning directory %s\n"), basedir);
    	dir_to_filelist(root, basedir);
	filelist_to_string(root, sb, 0);
    }

    len = strbuf_length(sb);
    indata = strbuf_free_to_string(sb);
    outdata = huffman_encode(indata, len, &len);
    free(indata);

    filename = catfiles(listing_dir, "MyList.DcLst");
    mkdirs_for_file(filename, true); /* Ignore errors */
    if (stat(filename, &st) < 0) {
    	if (errno != ENOENT)
	    warn(_("%s: Cannot get file status - %s\n"), filename, errstr);
    } else {
	if (unlink(filename) < 0)
	    warn(_("%s: Cannot remove file - %s\n"), filename, errstr);
    }
    i = ptrv_find(delete_files, filename, (comparison_fn_t) strcmp);
    if (i >= 0)
    	ptrv_remove_range(delete_files, i, i+1);

    fd = open(filename, O_CREAT|O_EXCL|O_WRONLY, 0666);
    if (fd < 0) {
    	screen_putf(_("%s: Cannot open file for writing - %s\n"), filename, errstr);
	filelist_free(root);
	free(outdata);
	free(filename);
	return false;
    }
    if (ptrv_find(delete_files, filename, (comparison_fn_t) strcmp) < 0)
    	ptrv_append(delete_files, xstrdup(filename));

    if (full_write(fd, outdata, len) < len) {
    	screen_putf(_("%s: Cannot write to file - %s\n"), filename, errstr);
	filelist_free(root); 
	free(outdata);
	free(filename);
	return false;
    }
    if (close(fd) < 0)
    	screen_putf(_("%s: Cannot close file - %s\n"), filename, errstr);
    free(filename);
    free(outdata);

    if (our_filelist != NULL)
    	filelist_free(our_filelist);
    our_filelist = root;
    my_share_size = our_filelist->u.dir.totalsize;

    return true;
}

/* Scan for bad components such as '..' in filename.
 */
static bool
check_filename(const char *inname)
{
    const char *t;
    const char *t2;

    for (t = inname; (t2 = strchr(t, '/')) != NULL; t = t2+1) {
    	if (t2-t == 2 && strncmp(t, "..", 2) == 0)
	    return false;
    }
    if (strcmp(t, "..") == 0)
    	return false;

    return true;
}

char *
resolve_upload_file(DCUserInfo *ui, const char *name)
{
    /* Ideally, we would check our MyList.DcLst and look up a
     * DCFileList node. Then we would get the path of this file
     * using filelist_get_path, and prepend share_dir to that.
     */
    if (!check_filename(name))
    	return NULL; /* Invalid filename - contains .. components */
    if (strcmp(name, "/MyList.DcLst") == 0)
        return catfiles(listing_dir, name+1);
    if (share_dir != NULL)
        return catfiles(share_dir, name);
    return NULL; /* Empty share directory */
}

char *
resolve_download_file(DCUserInfo *ui, const char *inname, const char *base_path)
{
    char *filename;

    /* Ideally, we would check the user's MyList.DcLst and look
     * up a DCFileList node.
     */
    if (!check_filename(inname))
    	return NULL;
    if (strcmp(inname, "/MyList.DcLst") == 0) {
        filename = xasprintf("%s/%s.MyList.DcLst", listing_dir, ui->nick);
	ptrv_append(delete_files, xstrdup(filename));
        mkdirs_for_file(filename, true); /* Ignore errors */
    } else {
        filename = catfiles(download_dir, inname+strlen(base_path));
        mkdirs_for_file(filename, false); /* Ignore errors */
    }

    return filename;
}

char *
translate_remote_to_local(const char *remotename)
{
    StrBuf *sb;

    sb = strbuf_new();
    strbuf_append_char(sb, '/');
    for (; *remotename != '\0'; remotename++) {
    	if (*remotename == '\\')
	    strbuf_append_char(sb, '/');
	else
	    strbuf_append_char(sb, *remotename);
    }

    return strbuf_free_to_string(sb);
}

char *
translate_local_to_remote(const char *localname)
{
    StrBuf *sb;

    sb = strbuf_new();
    localname++;
    for (; *localname != '\0'; localname++) {
    	if (*localname == '/')
	    strbuf_append_char(sb, '\\');
	else
	    strbuf_append_char(sb, *localname);
    }

    return strbuf_free_to_string(sb);
}

static bool
local_fs_completion_generator(const char *dir_part, const char *file_part, int state, DCCompletionEntry *ce, bool dirsonly)
{
    static DIR *dir;

    if (state == 0)
	dir = opendir(*dir_part ? dir_part : ".");
    
    if (dir != NULL) {
    	struct dirent *de;

	while ((de = readdir(dir)) != NULL) { /* Ignore errors */
	    if (*(de->d_name) == '.' && *file_part != '.')
		continue;
	    if (strleftcmp(file_part, de->d_name) == 0) {
		struct stat st;
		char *full;

		full = catfiles(*dir_part ? dir_part : ".", de->d_name);
		if (lstat(full, &st) < 0) {
		    free(full);
		    continue;
		}

		if (S_ISDIR(st.st_mode)) {
		    ce->input_char = '/';
		    ce->display_char = '/';
		} else if (S_ISLNK(st.st_mode)) {
		    if (stat(full, &st) >= 0 && S_ISDIR(st.st_mode)) {
		    	ce->input_char = '\0';
			ce->input_append_single_full = '/';
		    } else {
	    	    	if (dirsonly) {
			    free(full);
		    	    continue;
			}
			ce->input_char = ' ';
		    }
		    ce->display_char = '@';
		} else {
	    	    if (dirsonly) {
			free(full);
		    	continue;
		    }
		    ce->input_char = ' ';
		    if (access(full, X_OK) >= 0) {
		    	ce->display_char = '*';
		    } else {
		    	ce->display_char = '\0';
		    }
		}
		ce->str = xstrdup(de->d_name);
		free(full);

		return true;
	    }
	}
    	closedir(dir); /* Ignore errors */
    }

    return false;
}

bool
local_path_completion_generator(const char *path, const char *base, int state, DCCompletionEntry *ce)
{
    return local_fs_completion_generator(path, base, state, ce, false);
}

bool
local_dir_completion_generator(const char *path, const char *base, int state, DCCompletionEntry *ce)
{
    return local_fs_completion_generator(path, base, state, ce, true);
}

static bool
remote_fs_completion_generator(const char *dir_path, const char *file_part, int state, DCCompletionEntry *ce, bool dirsonly)
{
    static DCFileList *dir;
    static HMapIterator dir_it;
    static uint32_t dir_entry_count;

    if (browse_list == NULL)
    	return false;

    if (state == 0) {
    	char *abspath = apply_cwd(dir_path);

	dir = filelist_lookup(browse_list, abspath);
	free(abspath);
    	if (dir == NULL || dir->type != DC_TYPE_DIR)
	    return false;
	hmap_iterator(dir->u.dir.children, &dir_it);
	dir_entry_count = 0;
    }

    if (dir == NULL)
    	return false;

    for (;;) {
    	DCFileList *node;
	DCFileList dummy_node;

	if (dir_entry_count == 0) { 	    /* Fake current directory node (".") */
    	    dummy_node.type = DC_TYPE_DIR;
	    dummy_node.name = ".";
	    node = &dummy_node;
    	    dir_entry_count++;
	} else if (dir_entry_count == 1) {  /* Fake current directory node ("..") */
   	    dummy_node.type = DC_TYPE_DIR;
	    dummy_node.name = "..";
	    node = &dummy_node;
    	    dir_entry_count++;
	} else {    	        	    /* Get next normal directory node */
	    if (!dir_it.has_next(&dir_it))
	    	break;
    	    node = dir_it.next(&dir_it);
    	    dir_entry_count++;
	}

	if (dirsonly && node->type != DC_TYPE_DIR)
	    continue;
	if (node->name[0] == '.' && file_part[0] != '.')
	    continue;

	if (strleftcmp(file_part, node->name) == 0) {
    	    ce->str = xstrdup(node->name);
	    if (node->type == DC_TYPE_DIR) {
		ce->input_char = '/';
		ce->display_char = '/';
	    } else {
		ce->input_char = ' ';
		ce->display_char = '\0';
	    }
	    return true;
	}
    }

    return false;
}

bool
remote_path_completion_generator(const char *path, const char *base, int state, DCCompletionEntry *ce)
{
    return remote_fs_completion_generator(path, base, state, ce, false);
}

bool
remote_dir_completion_generator(const char *path, const char *base, int state, DCCompletionEntry *ce)
{
    return remote_fs_completion_generator(path, base, state, ce, true);
}

char *
apply_cwd(const char *path)
{
    if (*path == '/') {
    	return xstrdup(path);
    } else {
    	return catfiles(browse_path, path);
    }
}
