/* command.c - Command input and basic commands
 *
 * Copyright (C) 2004-2005 Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <ctype.h>		/* C89 */
#include <string.h>		/* ? */
#include <sys/types.h>		/* ? */
#include <sys/stat.h>		/* ? */
#include <unistd.h>		/* POSIX */
#include <netdb.h>		/* ? */
#include <sys/socket.h>		/* ? */
#include <netinet/in.h>		/* ? */
#include <arpa/inet.h>		/* ? */
#include <inttypes.h>		/* ? */
#include "xvasprintf.h"		/* Gnulib */
#include "xstrndup.h"		/* Gnulib */
#include "xalloc.h"		/* Gnulib */
#include "gettext.h"		/* Gnulib/GNU gettext */
#define _(s) gettext(s)
#define N_(s) gettext_noop(s)
#include "common/error.h"
#include "common/strleftcmp.h"
#include "common/intparse.h"
#include "common/common.h"
#include "common/strbuf.h"
#include "microdc.h"

static void cmd_exit(char *cmd, char *args);
static void cmd_nick(char *cmd, char *args);
static void cmd_description(char *cmd, char *args);
static void cmd_email(char *cmd, char *args);
static void cmd_sharedir(char *cmd, char *args);
static void cmd_downloaddir(char *cmd, char *args);
static void cmd_listingdir(char *cmd, char *args);
static void cmd_active(char *cmd, char *args);
static void cmd_slots(char *cmd, char *args);
static void cmd_say(char *cmd, char *args);
static void cmd_msg(char *cmd, char *args);
static void cmd_raw(char *cmd, char *args);
static void cmd_disconnect(char *cmd, char *args);
static void cmd_connect(char *cmd, char *args);
static void cmd_grantslot(char *cmd, char *args);
static void cmd_browse(char *cmd, char *args);
static void cmd_pwd(char *cmd, char *args);
static void cmd_find(char *cmd, char *args);
static void cmd_ls(char *cmd, char *args);
static void cmd_cd(char *cmd, char *args);
static void cmd_get(char *cmd, char *args);
static void cmd_queue(char *cmd, char *args);
static void cmd_unqueue(char *cmd, char *args);
static void cmd_speed(char *cmd, char *args);
static void cmd_password(char *cmd, char *args);
static void cmd_who(char *cmd, char *args);
static void cmd_tag(char *cmd, char *args);
static void cmd_transfers(char *cmd, char *args);
static void cmd_cancel(char *cmd, char *args);
static void cmd_debug(char *cmd, char *args);
static void cmd_listenaddr(char *cmd, char *args);
static void cmd_listenport(char *cmd, char *args);
static void cmd_search(char *cmd, char *args);
static void cmd_results(char *cmd, char *args);
static void cmd_unsearch(char *cmd, char *args);

struct DCCommand {
    char *name;
    void (*handler)(char *cmd, char *args);
    DCCompletionFunctionType type;
    void *completor;
};

/* This structure must be sorted by command name, according to strcmp. */
static struct DCCommand commands[] = {
    { "active",     	cmd_active, 	    0, NULL },
    { "browse",     	cmd_browse, 	    DC_CPL_SIMPLE, user_or_myself_completion_generator },
    { "cancel",         cmd_cancel,         DC_CPL_SIMPLE, transfer_completion_generator },
    { "cd", 	    	cmd_cd,     	    DC_CPL_FILE, remote_dir_completion_generator },
    { "connect",    	cmd_connect, 	    0, NULL },
    { "debug",          cmd_debug,          0, NULL },
    { "description", 	cmd_description,    0, NULL },
    { "disconnect", 	cmd_disconnect,     0, NULL },
    { "downloaddir", 	cmd_downloaddir,    DC_CPL_FILE, local_dir_completion_generator },
    { "email",      	cmd_email,  	    0, NULL },
    { "exit", 	    	cmd_exit,   	    0, NULL },
    { "find",		cmd_find,	    DC_CPL_FILE, remote_path_completion_generator },
    { "get", 	    	cmd_get,    	    DC_CPL_FILE, remote_path_completion_generator },
    { "grantslot",  	cmd_grantslot,      DC_CPL_SIMPLE, user_completion_generator },
    { "listenaddr",	cmd_listenaddr,     0, NULL },
    { "listenport",  	cmd_listenport,     0, NULL },
    { "listingdir", 	cmd_listingdir,     DC_CPL_FILE, local_dir_completion_generator },
    { "ls", 	    	cmd_ls,     	    DC_CPL_FILE, remote_path_completion_generator },
    { "msg", 	    	cmd_msg,    	    DC_CPL_SIMPLE, user_completion_generator },
    { "nick", 	    	cmd_nick,   	    0, NULL },
    { "password",   	cmd_password, 	    0, NULL },
    { "pwd", 	    	cmd_pwd,    	    0, NULL },
    { "queue",      	cmd_queue,  	    DC_CPL_SIMPLE, user_completion_generator },
    { "raw", 	    	cmd_raw,    	    0, NULL },
    { "results",        cmd_results,        0, NULL },
    { "say", 	    	cmd_say,    	    DC_CPL_CUSTOM, say_user_completion_generator },
    { "search",         cmd_search,         0, NULL },
    { "sharedir",   	cmd_sharedir, 	    DC_CPL_FILE, local_path_completion_generator },
    { "slots",      	cmd_slots,  	    0, NULL },
    { "speed",      	cmd_speed,  	    DC_CPL_SIMPLE, speed_completion_generator },
    { "tag", 	    	cmd_tag,    	    0, NULL },
    { "transfers",   	cmd_transfers,	    0, NULL },
    { "unqueue",    	cmd_unqueue, 	    DC_CPL_SIMPLE, user_completion_generator },
    { "unsearch",       cmd_unsearch,       0, NULL },
    { "who", 	    	cmd_who,    	    DC_CPL_SIMPLE, user_completion_generator },
};
#define COMMAND_COUNT (sizeof(commands)/sizeof(struct DCCommand))

static char *speeds[] = {
    "Modem",
    "28.8Kbps",
    "33.6Kbps",
    "56Kbps",
    "Satellite",
    "ISDN",
    "DSL",
    "Cable",
    "LAN(T1)",
    "LAN(T3)",
};
#define SPEED_COUNT (sizeof(speeds)/sizeof(char *))

static int
queued_file_cmp(const char *filename, DCQueuedFile *qf)
{
    return strcmp(filename, qf->filename);
}

/* Do a binary search of the table of DCCommand's. The comparator
 * function is supplied so that one can search based on complete command
 * name (strcmp) or partial name (strleftcmp). Partial search is needed
 * when looking for completion alternatives.
 */
static int32_t
find_command(const char *name, int (*comparator)(const char *, const char *))
{
    uint32_t lower = 0;
    uint32_t upper = COMMAND_COUNT;

    while (lower < upper) {
    	uint32_t index;
	int c;

	index = (lower + upper) / 2;
	c = comparator(name, commands[index].name);
	if (c < 0) {
	    upper = index;
	} else if (c > 0) {
	    lower = index + 1;
	} else {
	    return index;
	}
    }

    return -1;
}

void *
get_completor(const char *buffer, int start, int end, DCCompletionFunctionType *type)
{
    char *cmd;
    int c;
    int c2;

    /* First command? */
    for (c = 0; c < start; c++) {
    	if (!isspace(buffer[c]))
	    break;
    }
    if (c >= start) {
	*type = DC_CPL_SIMPLE;
    	return command_completion_generator;
    }

    for (c2 = c+1; !isspace(buffer[c2]); c2++);

    cmd = xstrndup(buffer+c, c2-c);
    c = find_command(cmd, strcmp);
    free(cmd);
    if (c >= 0) {
	*type = commands[c].type;
    	return commands[c].completor;
    }

    return NULL;
}

/* Look up completion alternatives from command list. I guess the search
 * can get faster than this - but remember that we don't compare more
 * commands than we eventually return to readline.
 */
char *
command_completion_generator(const char *base, int state)
{
    static int completion_start;
    static int completion_count;

    if (state == 0) {
	int32_t c;
    	int32_t index;

	index = find_command(base, strleftcmp);
    	if (index >= 0) {
	    for (c = index-1; c >= 0; c--) {
	    	if (strleftcmp(base, commands[c].name))
		    break;
	    }
	    completion_start = c+1;
	    for (c = index+1; c < COMMAND_COUNT; c++) {
	    	if (strleftcmp(base, commands[c].name))
		    break;
	    }
	    completion_count = c - completion_start;
	    return xstrdup(commands[completion_start].name);
	}
    } else if (state < completion_count) {
    	return xstrdup(commands[completion_start+state].name);
    }

    return NULL;
}

/* Execute a command specification from a complete line.
 */
void
command_execute(char *line)
{
    char *cmd;
    char *args;
    int i;

    for (; isspace(*line); line++);
    if (!*line || *line == '#')
    	return;

    cmd = line;
    for (; *line && !isspace(*line); line++);
    args = NULL;
    if (*line) {
    	*line = '\0';
	for (line++; isspace(*line); line++);
	if (*line) {
	    args = line;
    	    i = strlen(args);
	    while (i > 0 && isspace(args[i-1]))
		args[--i] = '\0';
	}
    }

    i = find_command(cmd, strcmp);
    if (i >= 0) {
	commands[i].handler(cmd, args);
    } else {
	screen_putf(_("%s: Unknown command.\n"), cmd);
    }
}

static void
cmd_exit(char *cmd, char *args)
{
    running = false;
}

static void
cmd_nick(char *cmd, char *args)
{
    if (args == NULL) {
    	screen_putf(_("Current nick: %s\n"), my_nick);
	return;
    }
    if (strchr(args, '$') != NULL
    	|| strchr(args, '|') != NULL
	|| strchr(args, ' ') != NULL) {
    	screen_putf(_("Nick may not contain '$', '|' or space.\n"));
	return;
    }
    if (strlen(args) > 35) {
    	screen_putf(_("Nick is too long (max is 35 characters)\n"));
	return;
    }
    free(my_nick);
    my_nick = xstrdup(args);
}

static void
cmd_description(char *cmd, char *args)
{
    if (args == NULL) {
    	screen_putf(_("Current description: %s\n"), my_description);
	return;
    }
    if (strchr(args, '$') != NULL || strchr(args, '|') != NULL) {
    	screen_putf(_("Description may not contain '$' or '|' characters\n"));
	return;
    }
    if (strlen(args) > 35) {
    	screen_putf(_("Description is too long (max is 35 characters)\n"));
	return;
    }
    free(my_description);
    my_description = xstrdup(args);
}

static void
cmd_email(char *cmd, char *args)
{
    if (args == NULL) {
    	screen_putf(_("Current e-mail: %s\n"), my_email);
	return;
    }
    if (strchr(args, '$') != NULL || strchr(args, '|') != NULL) {
    	screen_putf(_("E-mail may not contain '$' or '|' characters\n"));
	return;
    }
    if (strlen(args) > 35) {
    	screen_putf(_("E-mail is too long (max is 35 characters)\n"));
	return;
    }
    free(my_email);
    my_email = xstrdup(args);
}

static void
cmd_sharedir(char *cmd, char *args)
{
    struct stat st;
    char *new_dir;

    new_dir = get_argument(args, 0, 1);
    if (args == NULL) {
    	if (share_dir == NULL) {
    	    screen_putf(_("No share directory set\n"));
	} else {
    	    screen_putf(_("Current share directory: %s\n"), share_dir);
    	    screen_putf(_("Currently sharing %" PRId64 " bytes (%" PRId64 " MB).\n"), my_share_size, my_share_size/(1024*1024));
	}
	return;
    }
    if (stat(new_dir, &st) < 0) {
    	screen_putf(_("%s: Cannot get file status - %s\n"), new_dir, errstr);
	return;
    }
    if (!S_ISDIR(st.st_mode)) {
    	screen_putf(_("%s: Not a directory\n"), new_dir);
	return;
    }
    set_share_dir(new_dir); /* Ignore errors */
    free(new_dir);
}

static void
cmd_downloaddir(char *cmd, char *args)
{
    struct stat st;
    char *new_dir;

    new_dir = get_argument(args, 0, 1);
    if (new_dir == NULL) {
    	screen_putf(_("Current download directory: %s\n"), download_dir);
	return;
    }
    if (stat(new_dir, &st) < 0) {
    	screen_putf(_("%s: Cannot get file status - %s\n"), new_dir, errstr);
	return;
    }
    if (!S_ISDIR(st.st_mode)) {
    	screen_putf(_("%s: Not a directory\n"), new_dir);
	return;
    }
    free(download_dir);
    download_dir = new_dir;
}

static void
cmd_listingdir(char *cmd, char *args)
{
    struct stat st;
    char *new_dir;

    new_dir = get_argument(args, 0, 1);
    if (new_dir == NULL) {
    	screen_putf(_("Current listing directory: %s\n"), listing_dir);
	return;
    }
    if (stat(new_dir, &st) < 0) {
    	screen_putf(_("%s: Cannot get file status - %s\n"), new_dir, errstr);
	return;
    }
    if (!S_ISDIR(st.st_mode)) {
    	screen_putf(_("%s: Not a directory\n"), new_dir);
	return;
    }
    free(listing_dir);
    listing_dir = new_dir;
}

static void
cmd_active(char *cmd, char *args)
{
    bool state;

    if (args == NULL) {
    	screen_putf(_("Current active setting: %d\n"), is_active ? 1 : 0);
	return;
    }
    if (strcmp(args, "0") == 0) {
    	state = false;
    } else if (strcmp(args, "1") == 0) {
    	state = true;
    } else {
    	screen_putf(_("Specify active as 0 or 1.\n"));
	return;
    }
    if (!set_active(state, listen_port))
    	screen_putf(_("Active setting not changed.\n"));
}

static void
cmd_slots(char *cmd, char *args)
{
    if (args == NULL) {
    	screen_putf(_("Current number of slots: %d\n"), my_ul_slots);
	return;
    }
    if (!parse_uint32(args, &my_ul_slots)) {
    	screen_putf(_("Invalid slot number `%s'\n"), args);
	return;
    }
}

static void
cmd_say(char *cmd, char *args)
{
    char *msg;

    if (args == NULL) {
    	screen_putf(_("Usage: %s MESSAGE..\n"), cmd);
	return;
    }
    if (hub_state < DC_HUB_LOGGED_IN) {
    	screen_putf(_("Not connected.\n"));
	return;
    }
    msg = escape_message(args);
    hub_putf("<%s> %s|", my_nick, msg); /* Ignore error */
    free(msg);
}

static void
cmd_msg(char *cmd, char *args)
{
    char *msg;
    char *user;
    DCUserInfo *ui;

    if (args != NULL) {
	user = args;
	for (; *args && !isspace(*args); args++);
	if (!*args) {
	    args = NULL;
    	} else {
	    *args = '\0';
	    for (args++; isspace(*args); args++);
	    if (!*args)
	    	args = NULL;
	}
    }
    if (args == NULL) {
    	screen_putf(_("Usage: %s USER MESSAGE..\n"), cmd);
	return;
    }
    if (hub_state < DC_HUB_LOGGED_IN) {
    	screen_putf(_("Not connected.\n"));
	return;
    }

    ui = hmap_get(hub_users, user);
    if (ui == NULL) {
    	screen_putf(_("%s: No such user on this hub\n"), user);
	return;
    }

    msg = escape_message(args);
    hub_putf("$To: %s From: %s $<%s> %s|", ui->nick, my_nick, my_nick, msg); /* Ignore error */
    free(msg);
}

static void
cmd_raw(char *cmd, char *args)
{
    if (args == NULL) {
    	screen_putf(_("Usage: %s DATA...\n"), cmd);
	return;
    }
    if (hub_state < DC_HUB_LOCK) {
    	screen_putf(_("Not connected.\n"));
	return;
    }
    hub_putf("%s", args);
}

static void
cmd_connect(char *cmd, char *args)
{
    char *portstr;
    struct sockaddr_in addr;

    if (args == NULL) {
    	screen_putf(_("Usage: %s HOST[:PORT]\n"), cmd);
	return;
    }
    if (hub_state != DC_HUB_DISCONNECTED) {
    	screen_putf(_("Already connected or connecting, disconnect first.\n"));
	return;
    }

    portstr = strchr(args, ':');
    if (portstr != NULL) {
    	*portstr = '\0';
    	if (!parse_uint16(portstr+1, &addr.sin_port)) {
	    screen_putf(_("Invalid port number %s\n"), portstr);
	    return;
	}
	addr.sin_port = htons(addr.sin_port);
    } else {
    	addr.sin_port = htons(DC_HUB_TCP_PORT);
    }

    if (!inet_aton(args, &addr.sin_addr)) {
    	struct hostent *he;

	screen_putf(_("Looking up IP address for %s\n"), args);
	he = gethostbyname(args);
	if (he == NULL) {
    	    screen_putf(_("%s: Cannot look up address - %s\n"), args, hstrerror(h_errno));
	    return;
	}
    	addr.sin_addr = *(struct in_addr *) he->h_addr;
    }

    addr.sin_family = AF_INET;

    hub_connect(&addr); /* Ignore errors */
}

static void
cmd_disconnect(char *cmd, char *args)
{
    if (hub_state == DC_HUB_DISCONNECTED) {
    	warn(_("Not connected.\n"));
    } else {
    	warn(_("Disconnecting from hub.\n"));
	hub_disconnect();
    }
}

static void
cmd_grantslot(char *cmd, char *args)
{
    DCUserInfo *ui;

    if (args == NULL) {
    	screen_putf(_("Usage: %s USER\n"), cmd);
	return;
    }
    if (hub_state != DC_HUB_LOGGED_IN) {
    	screen_putf(_("Not connected.\n"));
	return;
    }
    ui = hmap_get(hub_users, args);
    if (ui == NULL) {
    	screen_putf(_("%s: No such user on this hub\n"), args);
	return;
    }
    ui->slot_granted = true;
}

void
browse_none(void)
{	
    /* Clean up previous browse. */
    if (browse_list != NULL) {
	filelist_free(browse_list);
	free(browse_path);
	browse_list = NULL;
    }
    if (browse_user != NULL) {
	user_info_free(browse_user);
	browse_user = NULL;
    }
    browsing_myself = false;
}

static void
cmd_browse(char *cmd, char *args)
{
    DCUserInfo *ui;
    char *filename;
    struct stat st;

    if (args == NULL) {
    	if (!browsing_myself && browse_user == NULL) {
	    screen_putf(_("Not browsing any user.\n"));
	    return;
	}
	browse_none();
	update_prompt();
	return;
    }

    if (strcmp(my_nick, args) == 0) {
	browse_none();
	browse_list = our_filelist;
	browse_path = xstrdup("/");
	browse_user = NULL;
	browsing_myself = true;
	update_prompt();
	return;
    }

    ui = hmap_get(hub_users, args);
    if (ui == NULL) {
        screen_putf(_("%s: No such user on this hub\n"), args);
        return;
    }

    filename = xasprintf("%s/%s.MyList.DcLst", listing_dir, ui->nick);
    if (stat(filename, &st) < 0) {
	if (errno != ENOENT) {
            screen_putf(_("%s: Cannot get file status - %s\n"), filename, errstr);
            free(filename);
            return;
	}

        free(filename);
	if (ptrv_find(ui->download_queue, "/MyList.DcLst", (comparison_fn_t) queued_file_cmp) < 0) {
	    DCQueuedFile *queued = xmalloc(sizeof(DCQueuedFile));

	    queued->filename = xstrdup("/MyList.DcLst");
	    queued->base_path = xstrdup("/");
	    queued->flag = DC_TF_LIST;
	    ptrv_prepend(ui->download_queue, queued);
	}
    	if (!has_user_conn(ui, DC_DIR_RECEIVE) && ui->conn_count < DC_USER_MAX_CONN)
	    hub_connect_user(ui); /* Ignore errors */
	else
	    screen_putf(_("No free connections. Queued file for download.\n"));

	browse_none();
	browse_user = ui;
	browsing_myself = false;
	ui->refcount++;
	update_prompt();
	return;
    }

    browse_none();
    browse_list = filelist_open(filename);
    browse_path = xstrdup("/");
    browse_user = ui;
    browsing_myself = false;
    ui->refcount++;
    free(filename);
    update_prompt();
}

static void
cmd_pwd(char *cmd, char *args)
{
    if (browse_list == NULL) {
	if (browse_user == NULL) {
	    screen_putf(_("Not browsing any user.\n"));
	} else {
	    screen_putf(_("(%s) Waiting for file list.\n"), browse_user->nick);
	}
    } else {
	screen_putf(_("(%s) %s\n"),
		browsing_myself ? my_nick : browse_user->nick,
		browse_path);
    }
}

static void
cmd_cd(char *cmd, char *args)
{
    DCFileList *node;
    char *new_path;
    char *path;

    if (browse_list == NULL) {
	screen_putf(_("Not browsing any user.\n"));
	return;
    }

    new_path = get_argument(args, 0, 1);
    if (new_path == NULL) {
	path = xstrdup("/");
    } else {
	path = apply_cwd(new_path);
	free(new_path);
    }

    node = filelist_lookup(browse_list, path);
    if (node == NULL) {
    	screen_putf(_("%s: No such file or directory\n"), args);
	free(path);
	return;
    }
    if (node->type != DC_TYPE_DIR) {
    	screen_putf(_("%s: Not a directory\n"), args);
	free(path);
	return;
    }
    free(browse_path);
    browse_path = filelist_get_path(node);
    update_prompt();
}

static void
cmd_find(char *cmd, char *args)
{
    char *path;
    char *arg;
    DCFileList *node;

    if (browse_list == NULL) {
	screen_putf(_("Not browsing any user.\n"));
	return;
    }
    arg = get_argument(args, 0, 1);
    if (arg != NULL) {
	path = apply_cwd(arg);
	free(arg);
    } else {
	path = xstrdup(browse_path);
    }

    node = filelist_lookup(browse_list, path);
    if (node == NULL) {
	screen_putf(_("%s: No such file or directory\n"), path);
	free(path);
	return;
    }
    filelist_list_recursively(node, "", 1);
    free(path);
}

static void
cmd_ls(char *cmd, char *args)
{
    char *arg;
    char *path = NULL;
    DCFileList *node;
    bool long_mode = false;

    /* XXX: this code can probably be cleaned up a little... */

    if (browse_list == NULL) {
	screen_putf(_("Not browsing any user.\n"));
	return;
    }
    arg = get_argument(args, 0, 1);
    if (arg != NULL) {
	if (strcmp(arg, "-l") == 0) {
	    long_mode = true;
	    free(arg);
	    arg = get_argument(args, 1, 1);
	    if (arg != NULL) {
		path = apply_cwd(arg);
		free(arg);
	    }
	} else {
    	    path = apply_cwd(arg);
	    free(arg);
	    arg = get_argument(args, 1, 1);
	    if (arg != NULL && strcmp(arg, "-l") == 0) {
		long_mode = true;
		free(arg);
	    }
	}
    }
    if (path == NULL)
	path = xstrdup(browse_path);

    node = filelist_lookup(browse_list, path);
    if (node == NULL) {
	screen_putf(_("%s: No such file or directory\n"), path);
	free(path);
	return;
    }
    filelist_list(node, long_mode);
    free(path);
}

static bool
append_download_dir(DCUserInfo *ui, DCFileList *node, char *base_path, uint64_t *bytes, uint32_t *files)
{
    if (node->type == DC_TYPE_REG) {
	DCQueuedFile *queued;
    	char *path = filelist_get_path(node);

	if (ptrv_find(ui->download_queue, path, (comparison_fn_t) queued_file_cmp) >= 0) {
	    screen_putf(_("Queue already contains this file, ignoring\n"));
	    free(path);
	    return false;
	}

	queued = xmalloc(sizeof(DCQueuedFile));
	queued->filename = path;
	queued->base_path = xstrdup(base_path);
	queued->flag = DC_TF_NORMAL;
	ptrv_append(ui->download_queue, queued);

	(*bytes) += node->u.reg.size;
	(*files) ++;
	return true;
    }
    if (node->type == DC_TYPE_DIR) {
    	HMapIterator it;
	bool s = false;

	hmap_iterator(node->u.dir.children, &it);
	while (it.has_next(&it))
	    s = append_download_dir(ui, it.next(&it), base_path, bytes, files) || s;
	return s;
    }
    return false;
}

static void
cmd_get(char *cmd, char *args)
{
    DCFileList *node;
    char *path;
    char *file_name;
    uint64_t bytes = 0;
    uint32_t files = 0;

    /* Note: This command assumes that we are browsing some user.
     * If we in the future would allow browse-less getting,
     * i.e. get NICK FILE, then we must here add additional checks
     * found in cmd_browse, such as strcmp(my_nick, nick)==0 etc.
     */

    file_name = get_argument(args, 0, 1);
    if (file_name == NULL) {
    	screen_putf(_("Usage: %s FILE\n"), cmd);
	return;
    }
    if (browse_list == NULL) {
	screen_putf(_("Not browsing any user.\n"));
	return;
    }
    if (browsing_myself) {
	screen_putf(_("Cannot download files from myself.\n"));
	return;
    }

    path = apply_cwd(file_name);
    free(file_name);
    node = filelist_lookup(browse_list, path);
    if (node == NULL) {
	screen_putf(_("%s: No such file or directory\n"), path);
	free(path);
	return;
    }
    free(path);

    if (append_download_dir(browse_user, node, browse_path, &bytes, &files)) {
	char *fmt;

	fmt = ngettext("Downloading %s (%" PRId64 " bytes in %d file)\n",
		"Downloading %s (%" PRId64 " bytes in %d files)\n", files);
    	path = filelist_get_path(node);
	screen_putf(fmt, path, bytes, files);
	free(path);
    } else {
    	if (browse_user->download_queue->cur == 0) {
	    screen_putf(_("No files to download (empty directories).\n"));
	    return;
	}
    }
    if (!has_user_conn(browse_user, DC_DIR_RECEIVE) && browse_user->conn_count < DC_USER_MAX_CONN)
    	hub_connect_user(browse_user); /* Ignore errors */
    else {
	char *msg;

	msg = ngettext("No free connections. Queued file for download.\n",
		"No free connections. Queued files for download.\n", files);
	screen_putf(msg);
    }
}

static void
cmd_queue(char *cmd, char *args)
{
    uint32_t c;
    DCUserInfo *ui;

    if (args == NULL) {
    	screen_putf(_("Usage: %s USER\n"), cmd);
	return;
    }
    ui = hmap_get(hub_users, args);
    if (ui == NULL) {
        screen_putf(_("%s: No such user on this hub\n"), args);
	return;
    }
    for (c = 0; c < ui->download_queue->cur; c++) {
	DCQueuedFile *queued = ui->download_queue->buf[c];
	screen_putf(_("%d. [ %s ] %s\n"),
		c+1,
		queued->base_path,
		queued->filename + strlen(queued->base_path));
    }
}

static void
cmd_unqueue(char *cmd, char *args)
{
    DCUserInfo *user;
    char *t, *t2;
    int32_t sp, ep;

    if (args == NULL) {
    	screen_putf(_("Usage: %s USER RANGE\n"), cmd);
	return;
    }

    t = strchr(args, ' ');
    if (t == NULL) {
        sp = 1;
        ep = -1;
    } else {
        *t = '\0';
        for (t++; isspace(*t); t++);
        t2 = strchr(t, '-');
        if (t2 == NULL) { /* No dash. */
            if (!parse_uint32(t, &sp)) {
                screen_putf(_("Invalid range\n"));
                return;
            }
	    ep = sp+1;
        } else if (t == t2) { /* Starts with dash. */
            sp = 1;
            t++;
            if (!parse_uint32(t, &ep)) {
                screen_putf(_("Invalid range\n"));
                return;
            }
        } else if (t2[1] == '\0') { /* Ends with dash. */
            *t2 = '\0';
            if (!parse_uint32(t, &sp)) {
                screen_putf(_("Invalid range\n"));
                return;
            }
	    ep = -1;
	} else { /* Dash in middle */
            *t2 = '\0';
            t2++;
            if (!parse_uint32(t, &sp)) {
                screen_putf(_("Invalid range\n"));
                return;
            }
            if (!parse_uint32(t2, &ep)) {
                screen_putf(_("Invalid range\n"));
                return;
            }
        }
    }

    user = hmap_get(hub_users, args);
    if (user == NULL) {
        screen_putf(_("%s: No such user on this hub\n"), args);
        return;
    }

    if (sp == 0 || ep == 0 || sp > user->download_queue->cur || ep > user->download_queue->cur) {
        screen_putf(_("Invalid range\n"));
        return;
    }
    sp--;
    if (ep < 0)
        ep = user->download_queue->cur;
    else
    	ep--;

    for (; sp < ep; sp++)
	free_queued_file(user->download_queue->buf[sp]);
    ptrv_remove_range(user->download_queue, sp, ep);
}

char *
speed_completion_generator(const char *base, int state)
{
    static int completion_pos;

    if (state == 0) {
    	uint32_t c;

    	for (c = 0; c < SPEED_COUNT; c++) {
	    if (strleftcmp(base, speeds[c]) == 0) {
	    	completion_pos = c+1;
		return xstrdup(speeds[c]);
	    }
    	}
	return NULL;
    }

    if (completion_pos < SPEED_COUNT) {
    	if (strleftcmp(base, speeds[completion_pos]) == 0)
	    return xstrdup(speeds[completion_pos++]);
    }

    return NULL;
}

static void
cmd_speed(char *cmd, char *args)
{
    if (args == NULL) {
    	uint32_t c;

        screen_putf(_("Current speed: %s\n"), my_speed);
        screen_putf(_("Speed should be one of\n"));
	for (c = 0; c < SPEED_COUNT; c++)
	    screen_putf("  %s\n", speeds[c]);
	return;
    }
    if (strchr(args, '$') != NULL || strchr(args, '|') != NULL) {
        screen_putf(_("Speed may not contain '$' or '|' characters"));
        return;
    }
    free(my_speed);
    my_speed = xstrdup(args);
}

static void
cmd_password(char *cmd, char *args)
{
    if (args == NULL) {
    	if (my_password != NULL) {
    	    screen_putf(_("Removing current password.\n"));
	    free(my_password);
	    my_password = NULL;
	} else {
	    screen_putf(_("No current password set - not removed.\n"));
	}
	return;
    }
    if (strchr(args, '|') != NULL) {
    	screen_putf(_("Password may not contain '|' character"));
	return;
    }
    free(my_password);
    my_password = xstrdup(args);
}

static int
user_info_compare(const void *i1, const void *i2)
{
    const DCUserInfo *f1 = *(const DCUserInfo **) i1;
    const DCUserInfo *f2 = *(const DCUserInfo **) i2;

    return strcmp(f1->nick, f2->nick);
}

static void
cmd_who(char *cmd, char *args)
{
    uint32_t maxlen;
    HMapIterator it;
    uint32_t c;
    DCUserInfo **items;
    uint32_t count;
    uint32_t cols;
    StrBuf *out;

    if (hub_state < DC_HUB_LOGGED_IN) {
    	screen_putf(_("Not connected.\n"));
	return;
    }

    if (args != NULL) {
    	DCUserInfo *ui;

	ui = hmap_get(hub_users, args);
	if (ui == NULL) {
    	    screen_putf(_("%s: No such user on this hub\n"), args);
	} else {
    	    screen_putf(_("Nick: %s\n"), ui->nick);
    	    screen_putf(_("Description: %s\n"), ui->description);
    	    screen_putf(_("Speed: %s\n"), ui->speed);
    	    screen_putf(_("Level: %d\n"), ui->level);
    	    screen_putf(_("E-mail: %s\n"), ui->email);
    	    screen_putf(_("Operator: %d\n"), ui->is_operator);
    	    screen_putf(_("Share Size: %" PRId64 " bytes (%" PRId64 " MB)\n"), ui->share_size, ui->share_size/(1024*1024));
	}
    	return;
    }

    maxlen = 0;
    for (hmap_iterator(hub_users, &it); it.has_next(&it); ) {
    	DCUserInfo *ui = it.next(&it);
	maxlen = max(maxlen, strlen(ui->nick));
    }

    count = hmap_size(hub_users);
    items = xmalloc(count * sizeof(DCUserInfo *));
    hmap_iterator(hub_users, &it);
    for (c = 0; c < count; c++)
	items[c] = it.next(&it);
    qsort(items, count, sizeof(DCUserInfo *), user_info_compare);

    screen_get_size(NULL, &cols);

    out = strbuf_new();
    for (c = 0; c < count; c++) {
    	DCUserInfo *ui = items[c];

	strbuf_clear(out);
	strbuf_append(out, ui->nick);
	strbuf_append_char_n(out, maxlen+1-strlen(ui->nick), ' ');
	strbuf_appendf(out, "  %7" PRId64 "M", ui->share_size / (1024*1024));
	strbuf_append(out, ui->is_operator ? " op" : "   ");
	if (ui->download_queue->cur > 0)
	    strbuf_appendf(out, " (%3d)", ui->download_queue->cur);
	else
	    strbuf_append(out, "      ");
	strbuf_appendf(out, " %s", ui->description ? ui->description : "");
	if (strbuf_length(out) > cols)
	    strbuf_set_length(out, cols);
	screen_putf("%s\n", strbuf_buffer(out));
    }
    free(items);
    strbuf_free(out);
}

static void
cmd_tag(char *cmd, char *args)
{
    if (args == NULL) {
    	screen_putf(_("Current tag: %s\n"), my_tag);
	return;
    }
    if (strchr(args, '$') != NULL || strchr(args, '|') != NULL) {
    	screen_putf(_("Tag may not contain '$' or '|' characters"));
	return;
    }
    free(my_tag);
    my_tag = xstrdup(args);
}

static void
cmd_transfers(char *cmd, char *args)
{
    char *format;
    HMapIterator it;
    uint32_t maxlen = 0;
    time_t now = 0;

    hmap_iterator(user_conns, &it);
    while (it.has_next(&it)) {
    	DCUserConn *uc = it.next(&it);
	maxlen = max(maxlen, strlen(uc->name));
    }

    format = xasprintf("%%-%ds  %%s\n", maxlen);

    if (time(&now) < 0) {
    	warn(_("Cannot get current time - %s\n"), errstr);
	return;
    }

    hmap_iterator(user_conns, &it);
    while (it.has_next(&it)) {
    	DCUserConn *uc = it.next(&it);
	char *status;

    	/*if (!get_user_conn_status(uc))
	    continue;*/
	status = user_conn_status_to_string(uc, now);
	screen_putf(format, uc->name, status);
	free(status);
    }

    screen_putf(_("Upload slots: %d/%d  Download slots: %d/unlimited\n"), used_ul_slots, my_ul_slots, used_dl_slots);

    free(format);
}

static void
cmd_cancel(char *cmd, char *args)
{
    DCUserConn *uc;

    if (args == NULL) {
    	screen_putf(_("Usage: %s CONNECTION\n"), cmd);
	return;
    }

    uc = hmap_get(user_conns, args);
    if (uc == NULL) {
    	screen_putf(_("%s: No such user connection.\n"), args);
	return;
    }

    user_conn_cancel(uc);
}

static void
cmd_debug(char *cmd, char *args)
{
    if (args == NULL) {
    	screen_putf(_("Current debug level: %d\n"), debug_level);
	return;
    }
    if (!parse_uint32(args, &debug_level)) {
    	screen_putf(_("%s: Invalid integer\n"), args);
	return;
    }
}

static void
cmd_listenaddr(char *cmd, char *args)
{
    struct sockaddr_in addr;

    if (args == NULL) {
	if (force_listen_addr.s_addr == INADDR_NONE) {
	    screen_putf(_("Listening address not set.\n"));
	} else {
	    force_listen_addr.s_addr = INADDR_NONE;
	    screen_putf(_("Removing listening address.\n"));
	}
	//screen_putf(_("Current listening address: %s\n"), sockaddr_in_str(&local_addr));
	return;
    }

    if (!inet_aton(args, &addr.sin_addr)) {
	struct hostent *he;

	screen_putf(_("Looking up IP address for %s\n"), args);
	he = gethostbyname(args);
	if (he == NULL) {
	    screen_putf(_("%s: Cannot look up address - %s\n"), args, hstrerror(h_errno));
	    return;
	}

	addr.sin_addr = *(struct in_addr *) he->h_addr;
    }

    force_listen_addr = addr.sin_addr;
    screen_putf(_("Listening address set to %s.\n"), inet_ntoa(force_listen_addr));
}

static void
cmd_listenport(char *cmd, char *args)
{
    uint16_t port;
    if (args == NULL) {
    	screen_putf(_("Current listening port: %d\n"), listen_port);
	return;
    }
    if (!parse_uint16(args, &port)) {
    	screen_putf(_("%s: Invalid integer\n"), args);
	return;
    }
    if (!set_active(is_active, port))
    	screen_putf(_("Active setting not changed.\n"));
}

static void
cmd_search(char *cmd, char *args)
{
    if (args == NULL) {
    	screen_putf(_("Usage: %s STRING...\n"), cmd);
	return;
    }
    add_search_request(args); /* Ignore errors */
}

static void
cmd_results(char *cmd, char *args)
{
    uint32_t c;

    if (args == NULL) {
    	for (c = 0; c < our_searches->cur; c++) {
    	    DCSearchRequest *sd = our_searches->buf[c];
	    screen_putf(_("%d. Results: %d\n"), c+1, sd->responses->cur);
	}
    } else {
    	DCSearchRequest *sd;
	if (!parse_uint32(args, &c) || c == 0 || c-1 >= our_searches->cur) {
    	    screen_putf(_("Invalid search index.\n"));
	    return;
	}
	sd = our_searches->buf[c-1];
	for (c = 0; c < sd->responses->cur; c++) {
	    DCSearchResponse *sr = sd->responses->buf[c];
	    char *n;
	    char *t;

	    n = translate_remote_to_local(sr->filename);
	    if (sr->filetype == DC_TYPE_DIR) /* XXX: put into some function */
	    	t = "/";
	    else
	    	t = "";
	    screen_putf("%d. %s %s%s\n", c+1, sr->userinfo->nick, n, t);
	    free(n);
	}
    }
}

static void
cmd_unsearch(char *cmd, char *args)
{
    DCSearchRequest *sd;
    uint32_t index;

    if (args == NULL) {
    	screen_putf(_("Usage: %s INDEX\n"), cmd);
	return;
    }
    if (!parse_uint32(args, &index) || index == 0 || index-1 >= our_searches->cur) {
    	screen_putf(_("Invalid search index.\n"));
	return;
    }
    
    sd = our_searches->buf[index-1];
    ptrv_remove_range(our_searches, index-1, index);
    free_search_request(sd);    
}

void
update_prompt(void)
{
    if (browsing_myself || browse_user != NULL) {
	char *nick = browsing_myself ? my_nick : browse_user->nick;
	
	if (browse_list == NULL) {
	    set_screen_prompt("%s:(%s)> ", PACKAGE, nick);
	} else {
	    set_screen_prompt("%s:%s:%s> ", PACKAGE, nick, browse_path);
	}
    } else {
	set_screen_prompt("%s> ", PACKAGE);
    }
}
