/* screen.c - User interface management (Readline)
 *
 * Copyright (C) 2004-2005 Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#if defined(HAVE_READLINE_READLINE_H)
# include <readline/readline.h>
#elif defined(HAVE_READLINE_H)
# include <readline.h>
#endif
#if defined(HAVE_READLINE_HISTORY_H)
# include <readline/history.h>
#elif defined(HAVE_HISTORY_H)
# include <history.h>
#endif
#include "microdc.h"
#include "common/error.h"
#include "common/memory.h"
#include "common/strleftcmp.h"
#include "common/common.h"
#include "minmax.h"
#include "xstrndup.h"
#include "xvasprintf.h"

static void clear_rl();
static void user_input(char *line);
static int real_screen_vputf(const char *format, va_list args);
static char *screen_prompt;

printf_fn_t screen_writer = real_screen_vputf;

enum {
    SCREEN_NO_HISTORY,
    SCREEN_NO_HANDLER,		/* rl_callback_handler_install not called. */
    SCREEN_NORMAL,		/* Normal non-readline status. Can print right away. */
    SCREEN_RL_DISPLAYED,	/* Readline will be displayed. Need to clear first. */
    SCREEN_RL_CLEARED,		/* Readline has been cleared. Need to redisplay after. */
} screen_state = SCREEN_NO_HISTORY;

static char *
quote_argument(const char *string, int rtype, int cs)
{
    const char *s;
    char *result;
    char *r;

    if (cs == '"') {
        result = xmalloc(2 * strlen(string) + 3);
        r = result;
        *r++ = '"';

        for (s = string; *s; s++) {
	    switch (*s) {
	    case '"':
	    case '\\':
	        *r++ = '\\';
	    default:
	        *r++ = *s;
	        break;
            }
        }

        if (rtype != MULT_MATCH)
            *r++ = '"';
        *r = '\0';
    } else {
        result = xmalloc(2 * strlen(string) + 1);
        r = result;

        for (s = string; *s; s++) {
	    switch (*s) {
	    case ' ':
	    case '\t':
	    case '\n':
	    case '"':
	    case '\'':
	    case '\\':
	        *r++ = '\\';
	        *r++ = *s;
	        break;
            default:
	        *r++ = *s;
	        break;
            }
        }

        *r = '\0';
    }

    return result;
}

static char *
dequote_argument(const char *text, int quote_char, int count)
{
    const char *p;
    char *ret;
    char *r;
    int quoted;

    ret = xmalloc(strlen(text) + 1);
    quoted = quote_char;
    r = ret;
    for (p = text; p-text < count && *p; p++) {
	if (*p == '\\') { /* don't touch backslash and escaped chars */
	    *r++ = *++p;
	    if (p-text >= count && *p == '\0')
		break;
	    continue;
	}
	if (quoted != 0 && *p == quoted) {	/* close quote */
	    quoted = 0;
	    continue;
	}
	if (quoted == 0 && *p == '"') {	/* open quote */
	    quoted = *p;
	    continue;
	}
	*r++ = *p;
    }
    *r = '\0';

    return ret;
}

static const char *
skip_word(const char *p)
{
    int quoted;

    quoted = 0;
    for (; *p; p++) {
	if (*p == '\\') { /* don't touch backslash and escaped chars */
	    p++;
	    if (*p == '\0')
		break;
	    continue;
	}
	if (quoted != 0 && *p == quoted) {	/* Close quote. */
	    quoted = 0;
	    continue;
	}
	if (quoted == 0 && *p == '"') {	/* Open quote. */
	    quoted = *p;
	    continue;
	}
	if (quoted == 0 && (*p == ' ' || *p == '\t' || *p == '\n'))
	    break;
    }

    return p;
}

char *
get_argument(const char *p, int index, int count)
{
    char *space_chars = " \t\n";
    const char *start;

    if (p == NULL) /* necessary when no argument specified */
	return NULL;
    
    /* Skip initial whitespace */
    for (; *p && strchr(space_chars, *p) != NULL; p++);

    /* Skip words prior */
    for (; *p && index > 0; index--) {
	p = skip_word(p);
	for (; *p && strchr(space_chars, *p) != NULL; p++);
    }
    if (*p == '\0')
	return NULL;

    /* Skip 'count' number of words */
    start = p;
    for (; *p && count > 0; count--) {
	p = skip_word(p);
	if (count > 1)
	    for (; *p && strchr(space_chars, *p) != NULL; p++);
    }

    return dequote_argument(start, 0, p-start);
}

int
char_is_quoted(char *string, int index)
{
    bool escaped = false;
    int c;

    for (c = 0; c <= index; c++) {
	if (escaped) {
	    escaped = false;
	    if (c >= index)
		return 1;
	} else if (string[c] == '"') {
	    char quote = string[c];
	    c++;
	    for (; string[c] != '\0' && string[c] != quote; c++) {
		if (string[c] == '\\' && string[c+1] != '\0')
		    c++;
	    }
	    if (c >= index)
		return 1;
	} else if (string[c] == '\\') {
	    escaped = true;
	}
    }

    return 0;
}

/* Readline < 5.0 disables SA_RESTART on SIGWINCH for some reason.
 * This turns it back on.
 * This was copied from guile.
 */
static int
fix_winch(void)
{
    struct sigaction action;

    if (sigaction(SIGWINCH, NULL, &action) >= 0) {
    	action.sa_flags |= SA_RESTART;
    	sigaction(SIGWINCH, &action, NULL); /* Ignore errors */
    }

    return 0;
}

static char **
attempted_completion(const char *text, int start, int end)
{
    void *completor;
    DCCompletionFunctionType type;
    char **matches = NULL;
    char *dir_part = NULL;
    char *file_part = NULL;
    char *text_nq;
    char quoted;

    quoted = ((char_is_quoted(rl_line_buffer, start) &&
	    strchr(rl_completer_quote_characters, rl_line_buffer[start-1]) != NULL)
	    ? rl_line_buffer[start-1] : 0);
    text_nq = dequote_argument(text, quoted, INT_MAX);
   
    /* Do not attempt readline's default filename completion if our
     * completors fails to return any results.
     */
    rl_attempted_completion_over = 1;

    completor = get_completor(rl_line_buffer, start, end, &type);
    //completor = path_completion_generator;

    if (type == DC_CPL_FILE) {
	file_part = strrchr(text_nq, '/');
	if (file_part == NULL) {
	    dir_part = xstrdup("");
	    file_part = text_nq;
	} else {
	    for (; file_part > text_nq && file_part[-1] == '/'; file_part--);
	    dir_part = xstrndup(text_nq, file_part-text_nq+1);
	    for (file_part++; *file_part == '/'; file_part++);
	}
    } else if (type == DC_CPL_SIMPLE || type == DC_CPL_CUSTOM) {
	dir_part = NULL;
	file_part = text_nq;
    }

    if (completor != NULL) {
    	PtrV *completion_entries;
    	int state;
	int lcd = INT_MAX;
	uint32_t c;

    	completion_entries = ptrv_new();
	for (state = 0; true; state++) {
	    DCCompletionEntry ce = { NULL, ' ', '\0', '\0', '\0' };
	    
	    if (type == DC_CPL_SIMPLE) {
		DCCompletionFunction simple_completor = completor;
		ce.str = simple_completor(file_part, state);
		if (ce.str == NULL)
		    break;
	    } else if (type == DC_CPL_FILE) {
		DCFileCompletionFunction file_completor = completor;
		if (!file_completor(dir_part, file_part, state, &ce))
		    break;
	    } else if (type == DC_CPL_CUSTOM) {
		DCCustomCompletionFunction custom_completor = completor;
		if (!custom_completor(file_part, state, &ce))
		    break;
	    }
	    ptrv_append(completion_entries, xmemdup(&ce, sizeof(ce)));

    	    if (state > 0) {
    		DCCompletionEntry *oldce = completion_entries->buf[state-1];
		int c1, c2, si;

	    	for (si = 0; (c1 = oldce->str[si]) && (c2 = ce.str[si]); si++) {
		    if (c1 != c2)
		    	break;
    	    	}
	    	lcd = min(lcd, si);
	    }
	}

    	if (state == 0) { /* no matches */
	    ptrv_free(completion_entries);
	    matches = NULL;
	}
	else if (state == 1) { /* single match */
	    DCCompletionEntry *ce = completion_entries->buf[0];
	    char *tmp = NULL;

	    matches = xmalloc(sizeof(char *) * 2);
	    if (ce->input_append_single) {
		tmp = xasprintf("%s%c", ce->str, ce->input_append_single);
		free(ce->str);
		ce->str = tmp;
	    } else if (ce->input_append_single_full && strcmp(file_part, ce->str) == 0) {
		free(ce->str);
		ce->str = xasprintf("%s%c", file_part, ce->input_append_single_full);
	    }

	    if (type == DC_CPL_FILE) {
		tmp = catfiles(dir_part, ce->str);
	    } else if (type == DC_CPL_SIMPLE || type == DC_CPL_CUSTOM) {
		tmp = ce->str;
	    }
	    matches[0] = quote_argument(tmp, SINGLE_MATCH, quoted);
	    free(tmp);

	    matches[1] = NULL;
	    rl_completion_append_character = ce->input_char;
	    free(ce);
	    ptrv_free(completion_entries);
	}
    	else { /* multiple matches */
    	    DCCompletionEntry *ce = completion_entries->buf[0];
	    char *tmp = NULL;

    	    matches = xmalloc(sizeof(char *) * (state + 2));
	    matches[0] = xstrndup(ce->str, lcd);
	    if (type == DC_CPL_FILE) {
		tmp = catfiles(dir_part, matches[0]);
		free(matches[0]);
	    }
	    else if (type == DC_CPL_SIMPLE || type == DC_CPL_CUSTOM) {
		tmp = matches[0];
	    }
	    matches[0] = quote_argument(tmp, MULT_MATCH, quoted);
	    free(tmp);

	    for (c = 0; c < state; c++) {
	    	ce = completion_entries->buf[c];
		if (ce->display_char) {
		    matches[c+1] = xasprintf("%s%c", ce->str, ce->display_char);
		    free(ce->str);
		} else {
	    	    matches[c+1] = ce->str;
		}
	    }
	    matches[state+1] = NULL;
	    ptrv_foreach(completion_entries, free);
	    ptrv_free(completion_entries);
	}
    }

    free(text_nq);
    free(dir_part);

    return matches;
}

/* This piece of code was snatched from lftp, and modified somewhat by me.
 * I suggest a function called rl_clear() be added to readline. The
 * function clears the prompt and everything the user has written so far on
 * the line. The cursor is positioned at the beginning of the line that
 * contained the prompt Note: This function doesn't modify the screen_state
 * variable.
 */
static void
clear_rl()
{
    extern char *rl_display_prompt;
#if HAVE__RL_MARK_MODIFIED_LINES
    extern int _rl_mark_modified_lines;
    int old_mark = _rl_mark_modified_lines;
#endif
    int old_end = rl_end;
    char *old_prompt = rl_display_prompt;

    rl_end = 0;
    rl_display_prompt = "";
    rl_expand_prompt("");
#if HAVE__RL_MARK_MODIFIED_LINES
    _rl_mark_modified_lines = 0;
#endif

    rl_redisplay();

    rl_end = old_end;
    rl_display_prompt = old_prompt;
#if HAVE__RL_MARK_MODIFIED_LINES
    _rl_mark_modified_lines = old_mark;
#endif
    if (rl_display_prompt == rl_prompt)
        rl_expand_prompt(rl_prompt);
}

/* Print something on screen, va_list style.
 * __attribute__ is provided by mdc.h declaration.
 */
static int
real_screen_vputf(const char *format, va_list args)
{
    if (screen_state == SCREEN_RL_DISPLAYED) {
        clear_rl();
        screen_state = SCREEN_RL_CLEARED;
    }
    return vprintf(format, args);
}

/* Print something on screen, printf format style.
 * __attribute__ is provided by mdc.h declaration.
 */
void
screen_putf(const char *format, ...)
{
    va_list args;

    va_start(args, format);
    screen_writer(format, args);
    va_end(args);
}

/* This function is called by readline whenever the user has
 * entered a full line (usually by pressing enter).
 */
static void
user_input(char *line)
{
    /* Readline has already made way for us. */
    screen_state = SCREEN_NORMAL;

    if (line == NULL) {
        /* Ctrl+D was pressed on an empty line. */
        screen_putf("exit\n");
        running = false;
    } else {
        add_history(line);
        command_execute(line);
    }

    /* The following is necessary or readline will display
     * a new prompt before we exit.
     */
    if (!running) {
        rl_callback_handler_remove();
        screen_state = SCREEN_NO_HANDLER;
    }
}

/* Move down one line (rl_on_new_line+redisplay), print a new prompt and
 * empty the input buffer. Unlike rl_clear this doesn't erase anything on
 * screen.
 *
 * This is usually called when a user presses Ctrl+C.
 * XXX: Should find a better way to do this (see lftp or bash).
 */
void
screen_erase_and_new_line(void)
{
    if (screen_state != SCREEN_NO_HANDLER) {
        rl_callback_handler_remove();
        printf("\n");
        rl_callback_handler_install(screen_prompt, user_input);
    }
}

/* Finish screen management. Usually called from main_finish.
 */
void
screen_finish(void)
{
    if (screen_state > SCREEN_NO_HANDLER) {
        rl_callback_handler_remove();
        if (screen_state == SCREEN_RL_DISPLAYED)
    	    printf("\n");
	screen_state = SCREEN_NO_HANDLER;
    }

    if (screen_state > SCREEN_NO_HISTORY) {
    	char *path;

    	get_package_file("history", &path);
    	if (mkdirs_for_file(path, false) >= 0) {
	    if (write_history(path) != 0)
		warn("%s: Cannot write - %s\n", path, errstr);
	}
	free(path);

    	//rl_basic_word_break_characters = ?
	//rl_completer_word_break_characters = ?
    	//rl_completion_display_matches_hook = ?
	rl_attempted_completion_function = NULL;
	//rl_char_is_quoted_p = NULL;
    	rl_pre_input_hook = NULL;

	warn_writer = default_warn_writer;
        screen_state = SCREEN_NO_HISTORY;
    }
}

/* Prepare the screen prior to waiting for events with select/poll/epoll.
 * Redisplay the prompt if it was cleared by a call to screen_(v)put(f).
 */
void
screen_prepare(void)
{
    if (screen_state <= SCREEN_NO_HISTORY) {
    	char *path;

    	screen_state = SCREEN_NO_HANDLER;
	warn_writer = real_screen_vputf;
	screen_prompt = xasprintf("%s> ", PACKAGE);

	rl_readline_name = PACKAGE;
	rl_attempted_completion_function = attempted_completion;
	rl_completer_quote_characters = "\"";
	rl_completer_word_break_characters = " \t\n\"";
	/* rl_filename_quote_characters = " \t\n\\\"'>;|&()*?[]~!"; */
	/* rl_filename_quoting_function = quote_argument; */
	/* rl_filename_dequoting_function = dequote_argument_rl; */
	rl_char_is_quoted_p = char_is_quoted;
    	rl_pre_input_hook = fix_winch;

        using_history();
    	get_package_file("history", &path);
	if (read_history(path) != 0 && errno != ENOENT)
    	    warn("%s: Cannot read - %s\n", path, errstr);
    }
    if (screen_state == SCREEN_NO_HANDLER) {
        rl_callback_handler_install(screen_prompt, user_input);
    }
    else if (screen_state == SCREEN_RL_CLEARED) {
        rl_redisplay();
    }
    screen_state = SCREEN_RL_DISPLAYED;
}

/* This function is called from the main loop when there's input for us to
 * read on stdin.
 */
void
screen_read_input(void)
{
    rl_callback_read_char();
}

/* Return the size of the screen.
 */
void
screen_get_size(int *rows, int *cols)
{
    int dummy;
    rl_get_screen_size(rows ? rows : &dummy, cols ? cols : &dummy);
}

void
set_screen_prompt(const char *prompt, ...)
{
    va_list args;

    if (screen_prompt != NULL)
	free(screen_prompt);
    va_start(args, prompt);
    screen_prompt = xvasprintf(prompt, args);
    va_end(args);
    rl_set_prompt(screen_prompt);
}
