/*
 * @(#)MibTreeModel.java
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * As a special exception, the copyright holders of this library 
 * give you permission to link this library with independent modules
 * to produce an executable, regardless of the license terms of 
 * these independent modules, and to copy and distribute the 
 * resulting executable under terms of your choice, provided that 
 * you also meet, for each linked independent module, the terms and 
 * conditions of the license of that module. An independent module 
 * is a module which is not derived from or based on this library. 
 * If you modify this library, you may extend this exception to your 
 * version of the library, but you are not obligated to do so. If 
 * you do not wish to do so, delete this exception statement from 
 * your version. 
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib;

import javax.swing.event.TreeModelListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import net.percederberg.mib.symbol.Symbol;
import net.percederberg.mib.symbol.TypeSymbol;
import net.percederberg.mib.type.SnmpTrapType;

/**
 * A tree model interface to a MIB file. This class adapts the Mib
 * class API to a TreeModel, making it possible to insert MIB:s into
 * some types of graphical components.
 *
 * @author   Per Cederberg, per@percederberg.net
 * @version  1.1
 */
public class MibTreeModel extends Object implements TreeModel {

    /**
     * The root node.
     */
    private TreeSymbol root;

    /**
     * The traps root node.
     */
    private TreeSymbol traps;

    /**
     * The types root node.
     */
    private TreeSymbol types;

    /**
     * The enterprises node.
     */
    private Symbol enterprises;

    /**
     * Creates a MIB tree model for the specified MIB.
     *
     * @param mib      the MIB object
     */
    public MibTreeModel(Mib mib) {

        // Sets the various root nodes
        this.root = new TreeSymbol(mib.toString());
        this.enterprises = mib.getSymbol("enterprises");
        if (this.enterprises != null) {
            this.root.addChild(this.enterprises);
        }
        this.traps = new TreeSymbol(this.root, "TRAPS");
        this.types = new TreeSymbol(this.root, "TEXTUAL CONVENTIONS");

        // Adds traps and types
        addTraps(mib);
        addTypes(mib);
    }

    /**
     * Adds all the trap symbols as children to the tree symbol for
     * traps.
     *
     * @param   mib         the mib containing the symbols
     */
    private void addTraps(Mib mib) {
        Symbol      sym;

        for (int i = 0; i < mib.getSymbolCount(); i++) {
            sym = mib.getSymbol(i);
            if (sym.getType() instanceof SnmpTrapType) {
                this.traps.addChild(sym);
            }
        }
    }

    /**
     * Adds all the type symbols as children to the tree symbol for
     * types.
     *
     * @param   mib         the mib containing the symbols
     */
    private void addTypes(Mib mib) {
        Symbol      sym;

        for (int i = 0; i < mib.getSymbolCount(); i++) {
            sym = mib.getSymbol(i);
            if (sym instanceof TypeSymbol) {
                this.types.addChild(sym);
            }
        }
    }

    /**
     * Returns the root of the tree. Returns null only if the tree has
     * no nodes.
     *
     * @return  the root of the tree
     */
    public Object getRoot() {
        return root;
    }

    /**
     * Returns a child node at the specified index. The parent node
     * must be a node previously obtained from this tree model. The
     * index should be in the range 0 to getChildCount(parent) - 1.
     *
     * @param parent  the parent node in this tree model
     * @param index   the child index
     *
     * @return the child node at the specified index, or
     *         null if not found
     */
    public Object getChild(Object parent, int index) {
        return ((Symbol) parent).childAt(index);
    }

    /**
     * Returns the number of child nodes. This method returns zero (0)
     * if the node is a leaf or if it has no children. The parent node
     * must be a node previously obtained from this tree model.
     *
     * @param parent  the parent node in this tree model
     *
     * @return the number of children to the specified node
     */
    public int getChildCount(Object parent) {
        return ((Symbol) parent).children();
    }

    /**
     * Checks if the specified node is a leaf. It is possible for this
     * method to return false even if the node has no children. A
     * directory in a filesystem, for example, may contain no files;
     * the node representing the directory still isn't a leaf,
     * however.
     *
     * @param node    a node in this tree model
     *
     * @return true if the node is a leaf, or
     *         false otherwise
     */
    public boolean isLeaf(Object node) {
        return getChildCount(node) == 0
            && !(node instanceof TreeSymbol);
    }

    /**
     * Called when the user has altered the value for the item
     * identified by <I>path</I> to <I>newValue</I>. If
     * <I>newValue</I> signifies a truly new value the model should
     * post a treeNodesChanged event.
     *
     * @param path      path to the node that the user has altered
     * @param newValue  the new value from the TreeCellEditor
     */
    public void valueForPathChanged(TreePath path, Object newValue) {
        // Ignored
    }

    /**
     * Returns the index of child in parent.
     *
     * @param parent  the parent node
     * @param child   the child node
     *
     * @return the child node index, or
     *         -1 if not present
     */
    public int getIndexOfChild(Object parent, Object child) {
        int      i = 0;
        Symbol   symbol = (Symbol) parent;

        while (i < symbol.children()) {
            if (symbol.childAt(i) == child) {
                return i;
            }
            i++;
        }
        return -1;
    }

    /**
     * Adds a listener for the TreeModelEvent posted after the tree
     * changes.
     *
     * @param l    the listener to add
     *
     * @see #removeTreeModelListener
     */
    public void addTreeModelListener(TreeModelListener l) {
        // Ignored
    }

    /**
     * Removes a listener previously added.
     *
     * @param l    the listener to remove
     *
     * @see #addTreeModelListener
     */
    public void removeTreeModelListener(TreeModelListener l) {
        // Ignored
    }


    /**
     * A class for allowing symbol nodes that are not really symbols 
     * from the MIB file. Typically three tree symbols will be 
     * created; the root, the "TEXUAL CONVENTIONS" node and the 
     * "TRAPS" node.
     */
    private class TreeSymbol extends Symbol {
    
        /**
         * Creates a new root tree symbol with the given name.
         *
         * @param name     the root name (i.e. the file name)
         */
        public TreeSymbol(String name) {
            this.name = name;
        }
    
        /**
         * Creates a new tree symbol with the given parent and name.
         *
         * @param parent     the parent symbol
         * @param name       the symbol name
         */
        public TreeSymbol(Symbol parent, String name) {
            this.name = name;
            this.setParent(parent);
        }
    
        /**
         * Overrules the protected definition of addChild from symbol
         * in order to add children that have no registered parent.
         *
         * @param child    the child to add
         */
        public void addChild(Symbol child) {
            super.addChild(child);
        }
    
        /**
         * Returns an empty string.
         *
         * @return an empty string
         */
        public String getOID() {
            return "";
        }
    
        /**
         * Throws an exception, since this operation is not allowed for
         * tree symbols.
         *
         * @param id     the new OID
         *
         * @exception UnsupportedOperationException thrown as this
         *                operation is not supported
         */
        public void setOID(int id) throws UnsupportedOperationException {
            throw new UnsupportedOperationException(
                "Cannot assign an id to " + this.toString());
        }
    }
}
