/*
 * @(#)StringType.java         1.0            6 September 1999
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2002 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib.type;

/**
 * A class for handling string type information.
 *
 * @version  1.0
 * @author   Per Cederberg, per@percederberg.net
 */
public class StringType extends Type {

   /**
    * Maximum allowed string length, or -1 for infinite.
    */
   private SizeConstraint size;

   /**
    * Creates a new string type with unrestricted length.
    */
   public StringType() {
      this(null);
   }

   /**
    * Creates a new string type with length restrictions.
    *
    * @param  size      the size constraints
    */
   public StringType(SizeConstraint size) {
      this.size = size;
   }

   /**
    * Checks if this type is equals to some other object. It will only
    * be reported equal to other string types with the same constraints.
    *
    * @param  obj     an object
    *
    * @return true if the objects are equal, or
    *         false otherwise
    */
   public boolean equals(Object obj) {
      if (obj instanceof StringType) {
         StringType t = (StringType)obj;
         if (size == null) {
            return this.size == t.size;
         } else {
            return this.size.equals(t.size);
         }
      } else if (obj instanceof NamedType) {
         NamedType t = (NamedType)obj;
         return t.equals(this);
      } else {
         return false;
      }
   }

   /**
    * Returns a string description of this type.
    *
    * @return a string description of the type
    */
   public String toString() {
      if (size == null) {
         return "String";
      } else {
         return "String [" + size + "]";
      }
   }

   /**
    * Transfers the type information from this type to a type converter
    * object. The calling conventions declared in the TypeConverter
    * class are followed.
    *
    * @param   converter     a type converter
    */
   public void transferType(TypeConverter converter) {
      converter.createString();
      if (size != null) {
         size.transferConstraint(converter);
      }
   }

}

