/*
 * @(#)DepthFirstAdapter.java       1.2          9 August 1999
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2002 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib.asn1.analysis;

import net.percederberg.mib.asn1.node.Node;

/**
 * This class provides a depth first tree traversal implementation for
 * the parse tree. This class should be subclassed and the appropriate
 * inXXX and outXXX methods overridden for handling various types of
 * parse tree node analysis. All the case methods are declared final
 * to avoid overriding them.
 *
 * @version  1.2
 * @author   Per Cederberg, per@percederberg.net
 */
public class DepthFirstAdapter extends TreeWalker implements Switch {

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inStart(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outStart(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inStart method and afterwards the
    * outStart method.
    *
    * @param node    a parse tree node
    */
   public final void caseStart(Node node) {
      inStart(node);
      defaultCase(node);
      outStart(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inModuleDefinition(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outModuleDefinition(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inModuleDefinition method and afterwards the
    * outModuleDefinition method.
    *
    * @param node    a parse tree node
    */
   public final void caseModuleDefinition(Node node) {
      inModuleDefinition(node);
      defaultCase(node);
      outModuleDefinition(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTagDefault(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTagDefault(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inTagDefault method and afterwards the
    * outTagDefault method.
    *
    * @param node    a parse tree node
    */
   public final void caseTagDefault(Node node) {
      inTagDefault(node);
      defaultCase(node);
      outTagDefault(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inModuleIdentifier(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outModuleIdentifier(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inModuleIdentifier method and afterwards the
    * outModuleIdentifier method.
    *
    * @param node    a parse tree node
    */
   public final void caseModuleIdentifier(Node node) {
      inModuleIdentifier(node);
      defaultCase(node);
      outModuleIdentifier(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inModuleBody(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outModuleBody(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inModuleBody method and afterwards the
    * outModuleBody method.
    *
    * @param node    a parse tree node
    */
   public final void caseModuleBody(Node node) {
      inModuleBody(node);
      defaultCase(node);
      outModuleBody(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inExports(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outExports(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inExports method and afterwards the
    * outExports method.
    *
    * @param node    a parse tree node
    */
   public final void caseExports(Node node) {
      inExports(node);
      defaultCase(node);
      outExports(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inImports(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outImports(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inImports method and afterwards the
    * outImports method.
    *
    * @param node    a parse tree node
    */
   public final void caseImports(Node node) {
      inImports(node);
      defaultCase(node);
      outImports(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSymbolsFromModuleList(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSymbolsFromModuleList(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSymbolsFromModuleList method and afterwards the
    * outSymbolsFromModuleList method.
    *
    * @param node    a parse tree node
    */
   public final void caseSymbolsFromModuleList(Node node) {
      inSymbolsFromModuleList(node);
      defaultCase(node);
      outSymbolsFromModuleList(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSymbolsFromModule(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSymbolsFromModule(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSymbolsFromModule method and afterwards the
    * outSymbolsFromModule method.
    *
    * @param node    a parse tree node
    */
   public final void caseSymbolsFromModule(Node node) {
      inSymbolsFromModule(node);
      defaultCase(node);
      outSymbolsFromModule(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSymbolList(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSymbolList(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSymbolList method and afterwards the
    * outSymbolList method.
    *
    * @param node    a parse tree node
    */
   public final void caseSymbolList(Node node) {
      inSymbolList(node);
      defaultCase(node);
      outSymbolList(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSymbol(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSymbol(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSymbol method and afterwards the
    * outSymbol method.
    *
    * @param node    a parse tree node
    */
   public final void caseSymbol(Node node) {
      inSymbol(node);
      defaultCase(node);
      outSymbol(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inAssignmentList(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outAssignmentList(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inAssignmentList method and afterwards the
    * outAssignmentList method.
    *
    * @param node    a parse tree node
    */
   public final void caseAssignmentList(Node node) {
      inAssignmentList(node);
      defaultCase(node);
      outAssignmentList(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inAssignment(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outAssignment(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inAssignment method and afterwards the
    * outAssignment method.
    *
    * @param node    a parse tree node
    */
   public final void caseAssignment(Node node) {
      inAssignment(node);
      defaultCase(node);
      outAssignment(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inMacroDefinition(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outMacroDefinition(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inMacroDefinition method and afterwards the
    * outMacroDefinition method.
    *
    * @param node    a parse tree node
    */
   public final void caseMacroDefinition(Node node) {
      inMacroDefinition(node);
      defaultCase(node);
      outMacroDefinition(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inMacroBody(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outMacroBody(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inMacroBody method and afterwards the
    * outMacroBody method.
    *
    * @param node    a parse tree node
    */
   public final void caseMacroBody(Node node) {
      inMacroBody(node);
      defaultCase(node);
      outMacroBody(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSkipToEND(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSkipToEND(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSkipToEND method and afterwards the
    * outSkipToEND method.
    *
    * @param node    a parse tree node
    */
   public final void caseSkipToEND(Node node) {
      inSkipToEND(node);
      defaultCase(node);
      outSkipToEND(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inMacroReference(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outMacroReference(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inMacroReference method and afterwards the
    * outMacroReference method.
    *
    * @param node    a parse tree node
    */
   public final void caseMacroReference(Node node) {
      inMacroReference(node);
      defaultCase(node);
      outMacroReference(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTypeAssignment(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTypeAssignment(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inTypeAssignment method and afterwards the
    * outTypeAssignment method.
    *
    * @param node    a parse tree node
    */
   public final void caseTypeAssignment(Node node) {
      inTypeAssignment(node);
      defaultCase(node);
      outTypeAssignment(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inType(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outType(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inType method and afterwards the
    * outType method.
    *
    * @param node    a parse tree node
    */
   public final void caseType(Node node) {
      inType(node);
      defaultCase(node);
      outType(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inBuiltinType(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outBuiltinType(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inBuiltinType method and afterwards the
    * outBuiltinType method.
    *
    * @param node    a parse tree node
    */
   public final void caseBuiltinType(Node node) {
      inBuiltinType(node);
      defaultCase(node);
      outBuiltinType(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inIntegerType(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outIntegerType(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inIntegerType method and afterwards the
    * outIntegerType method.
    *
    * @param node    a parse tree node
    */
   public final void caseIntegerType(Node node) {
      inIntegerType(node);
      defaultCase(node);
      outIntegerType(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inNamedNumberList(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outNamedNumberList(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inNamedNumberList method and afterwards the
    * outNamedNumberList method.
    *
    * @param node    a parse tree node
    */
   public final void caseNamedNumberList(Node node) {
      inNamedNumberList(node);
      defaultCase(node);
      outNamedNumberList(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inNamedNumber(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outNamedNumber(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inNamedNumber method and afterwards the
    * outNamedNumber method.
    *
    * @param node    a parse tree node
    */
   public final void caseNamedNumber(Node node) {
      inNamedNumber(node);
      defaultCase(node);
      outNamedNumber(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSignedNumber(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSignedNumber(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSignedNumber method and afterwards the
    * outSignedNumber method.
    *
    * @param node    a parse tree node
    */
   public final void caseSignedNumber(Node node) {
      inSignedNumber(node);
      defaultCase(node);
      outSignedNumber(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inStringType(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outStringType(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inStringType method and afterwards the
    * outStringType method.
    *
    * @param node    a parse tree node
    */
   public final void caseStringType(Node node) {
      inStringType(node);
      defaultCase(node);
      outStringType(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inBitStringType(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outBitStringType(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inBitStringType method and afterwards the
    * outBitStringType method.
    *
    * @param node    a parse tree node
    */
   public final void caseBitStringType(Node node) {
      inBitStringType(node);
      defaultCase(node);
      outBitStringType(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSequenceType(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSequenceType(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSequenceType method and afterwards the
    * outSequenceType method.
    *
    * @param node    a parse tree node
    */
   public final void caseSequenceType(Node node) {
      inSequenceType(node);
      defaultCase(node);
      outSequenceType(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSequenceOfType(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSequenceOfType(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSequenceOfType method and afterwards the
    * outSequenceOfType method.
    *
    * @param node    a parse tree node
    */
   public final void caseSequenceOfType(Node node) {
      inSequenceOfType(node);
      defaultCase(node);
      outSequenceOfType(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSetType(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSetType(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSetType method and afterwards the
    * outSetType method.
    *
    * @param node    a parse tree node
    */
   public final void caseSetType(Node node) {
      inSetType(node);
      defaultCase(node);
      outSetType(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSetOfType(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSetOfType(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSetOfType method and afterwards the
    * outSetOfType method.
    *
    * @param node    a parse tree node
    */
   public final void caseSetOfType(Node node) {
      inSetOfType(node);
      defaultCase(node);
      outSetOfType(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inElementTypeList(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outElementTypeList(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inElementTypeList method and afterwards the
    * outElementTypeList method.
    *
    * @param node    a parse tree node
    */
   public final void caseElementTypeList(Node node) {
      inElementTypeList(node);
      defaultCase(node);
      outElementTypeList(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inElementType(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outElementType(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inElementType method and afterwards the
    * outElementType method.
    *
    * @param node    a parse tree node
    */
   public final void caseElementType(Node node) {
      inElementType(node);
      defaultCase(node);
      outElementType(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inNamedType(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outNamedType(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inNamedType method and afterwards the
    * outNamedType method.
    *
    * @param node    a parse tree node
    */
   public final void caseNamedType(Node node) {
      inNamedType(node);
      defaultCase(node);
      outNamedType(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inChoiceType(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outChoiceType(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inChoiceType method and afterwards the
    * outChoiceType method.
    *
    * @param node    a parse tree node
    */
   public final void caseChoiceType(Node node) {
      inChoiceType(node);
      defaultCase(node);
      outChoiceType(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inEnumeratedType(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outEnumeratedType(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inEnumeratedType method and afterwards the
    * outEnumeratedType method.
    *
    * @param node    a parse tree node
    */
   public final void caseEnumeratedType(Node node) {
      inEnumeratedType(node);
      defaultCase(node);
      outEnumeratedType(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSelectionType(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSelectionType(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSelectionType method and afterwards the
    * outSelectionType method.
    *
    * @param node    a parse tree node
    */
   public final void caseSelectionType(Node node) {
      inSelectionType(node);
      defaultCase(node);
      outSelectionType(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTaggedType(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTaggedType(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inTaggedType method and afterwards the
    * outTaggedType method.
    *
    * @param node    a parse tree node
    */
   public final void caseTaggedType(Node node) {
      inTaggedType(node);
      defaultCase(node);
      outTaggedType(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTag(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTag(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inTag method and afterwards the
    * outTag method.
    *
    * @param node    a parse tree node
    */
   public final void caseTag(Node node) {
      inTag(node);
      defaultCase(node);
      outTag(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inClassNumber(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outClassNumber(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inClassNumber method and afterwards the
    * outClassNumber method.
    *
    * @param node    a parse tree node
    */
   public final void caseClassNumber(Node node) {
      inClassNumber(node);
      defaultCase(node);
      outClassNumber(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inClass(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outClass(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inClass method and afterwards the
    * outClass method.
    *
    * @param node    a parse tree node
    */
   public final void caseClass(Node node) {
      inClass(node);
      defaultCase(node);
      outClass(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inAnyType(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outAnyType(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inAnyType method and afterwards the
    * outAnyType method.
    *
    * @param node    a parse tree node
    */
   public final void caseAnyType(Node node) {
      inAnyType(node);
      defaultCase(node);
      outAnyType(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inDefinedType(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outDefinedType(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inDefinedType method and afterwards the
    * outDefinedType method.
    *
    * @param node    a parse tree node
    */
   public final void caseDefinedType(Node node) {
      inDefinedType(node);
      defaultCase(node);
      outDefinedType(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inConstraintList(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outConstraintList(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inConstraintList method and afterwards the
    * outConstraintList method.
    *
    * @param node    a parse tree node
    */
   public final void caseConstraintList(Node node) {
      inConstraintList(node);
      defaultCase(node);
      outConstraintList(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inConstraint(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outConstraint(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inConstraint method and afterwards the
    * outConstraint method.
    *
    * @param node    a parse tree node
    */
   public final void caseConstraint(Node node) {
      inConstraint(node);
      defaultCase(node);
      outConstraint(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inValueConstraint(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outValueConstraint(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inValueConstraint method and afterwards the
    * outValueConstraint method.
    *
    * @param node    a parse tree node
    */
   public final void caseValueConstraint(Node node) {
      inValueConstraint(node);
      defaultCase(node);
      outValueConstraint(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inValueRange(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outValueRange(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inValueRange method and afterwards the
    * outValueRange method.
    *
    * @param node    a parse tree node
    */
   public final void caseValueRange(Node node) {
      inValueRange(node);
      defaultCase(node);
      outValueRange(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inLowerEndPoint(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outLowerEndPoint(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inLowerEndPoint method and afterwards the
    * outLowerEndPoint method.
    *
    * @param node    a parse tree node
    */
   public final void caseLowerEndPoint(Node node) {
      inLowerEndPoint(node);
      defaultCase(node);
      outLowerEndPoint(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inUpperEndPoint(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outUpperEndPoint(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inUpperEndPoint method and afterwards the
    * outUpperEndPoint method.
    *
    * @param node    a parse tree node
    */
   public final void caseUpperEndPoint(Node node) {
      inUpperEndPoint(node);
      defaultCase(node);
      outUpperEndPoint(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSizeConstraint(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSizeConstraint(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSizeConstraint method and afterwards the
    * outSizeConstraint method.
    *
    * @param node    a parse tree node
    */
   public final void caseSizeConstraint(Node node) {
      inSizeConstraint(node);
      defaultCase(node);
      outSizeConstraint(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inAlphabetConstraint(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outAlphabetConstraint(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inAlphabetConstraint method and afterwards the
    * outAlphabetConstraint method.
    *
    * @param node    a parse tree node
    */
   public final void caseAlphabetConstraint(Node node) {
      inAlphabetConstraint(node);
      defaultCase(node);
      outAlphabetConstraint(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inValueAssignment(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outValueAssignment(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inValueAssignment method and afterwards the
    * outValueAssignment method.
    *
    * @param node    a parse tree node
    */
   public final void caseValueAssignment(Node node) {
      inValueAssignment(node);
      defaultCase(node);
      outValueAssignment(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inValue(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outValue(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inValue method and afterwards the
    * outValue method.
    *
    * @param node    a parse tree node
    */
   public final void caseValue(Node node) {
      inValue(node);
      defaultCase(node);
      outValue(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inDefinedValue(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outDefinedValue(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inDefinedValue method and afterwards the
    * outDefinedValue method.
    *
    * @param node    a parse tree node
    */
   public final void caseDefinedValue(Node node) {
      inDefinedValue(node);
      defaultCase(node);
      outDefinedValue(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inBuiltinValue(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outBuiltinValue(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inBuiltinValue method and afterwards the
    * outBuiltinValue method.
    *
    * @param node    a parse tree node
    */
   public final void caseBuiltinValue(Node node) {
      inBuiltinValue(node);
      defaultCase(node);
      outBuiltinValue(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inBooleanValue(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outBooleanValue(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inBooleanValue method and afterwards the
    * outBooleanValue method.
    *
    * @param node    a parse tree node
    */
   public final void caseBooleanValue(Node node) {
      inBooleanValue(node);
      defaultCase(node);
      outBooleanValue(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSpecialRealValue(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSpecialRealValue(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSpecialRealValue method and afterwards the
    * outSpecialRealValue method.
    *
    * @param node    a parse tree node
    */
   public final void caseSpecialRealValue(Node node) {
      inSpecialRealValue(node);
      defaultCase(node);
      outSpecialRealValue(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inNullValue(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outNullValue(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inNullValue method and afterwards the
    * outNullValue method.
    *
    * @param node    a parse tree node
    */
   public final void caseNullValue(Node node) {
      inNullValue(node);
      defaultCase(node);
      outNullValue(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inNamedValue(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outNamedValue(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inNamedValue method and afterwards the
    * outNamedValue method.
    *
    * @param node    a parse tree node
    */
   public final void caseNamedValue(Node node) {
      inNamedValue(node);
      defaultCase(node);
      outNamedValue(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inObjectIdentifierValue(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outObjectIdentifierValue(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inObjectIdentifierValue method and afterwards the
    * outObjectIdentifierValue method.
    *
    * @param node    a parse tree node
    */
   public final void caseObjectIdentifierValue(Node node) {
      inObjectIdentifierValue(node);
      defaultCase(node);
      outObjectIdentifierValue(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inObjIdComponentList(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outObjIdComponentList(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inObjIdComponentList method and afterwards the
    * outObjIdComponentList method.
    *
    * @param node    a parse tree node
    */
   public final void caseObjIdComponentList(Node node) {
      inObjIdComponentList(node);
      defaultCase(node);
      outObjIdComponentList(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inObjIdComponent(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outObjIdComponent(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inObjIdComponent method and afterwards the
    * outObjIdComponent method.
    *
    * @param node    a parse tree node
    */
   public final void caseObjIdComponent(Node node) {
      inObjIdComponent(node);
      defaultCase(node);
      outObjIdComponent(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inNameAndNumberForm(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outNameAndNumberForm(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inNameAndNumberForm method and afterwards the
    * outNameAndNumberForm method.
    *
    * @param node    a parse tree node
    */
   public final void caseNameAndNumberForm(Node node) {
      inNameAndNumberForm(node);
      defaultCase(node);
      outNameAndNumberForm(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inBinaryString(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outBinaryString(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inBinaryString method and afterwards the
    * outBinaryString method.
    *
    * @param node    a parse tree node
    */
   public final void caseBinaryString(Node node) {
      inBinaryString(node);
      defaultCase(node);
      outBinaryString(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inHexString(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outHexString(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inHexString method and afterwards the
    * outHexString method.
    *
    * @param node    a parse tree node
    */
   public final void caseHexString(Node node) {
      inHexString(node);
      defaultCase(node);
      outHexString(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inCharString(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outCharString(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inCharString method and afterwards the
    * outCharString method.
    *
    * @param node    a parse tree node
    */
   public final void caseCharString(Node node) {
      inCharString(node);
      defaultCase(node);
      outCharString(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inNumber(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outNumber(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inNumber method and afterwards the
    * outNumber method.
    *
    * @param node    a parse tree node
    */
   public final void caseNumber(Node node) {
      inNumber(node);
      defaultCase(node);
      outNumber(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inIdentifier(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outIdentifier(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inIdentifier method and afterwards the
    * outIdentifier method.
    *
    * @param node    a parse tree node
    */
   public final void caseIdentifier(Node node) {
      inIdentifier(node);
      defaultCase(node);
      outIdentifier(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inModuleReference(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outModuleReference(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inModuleReference method and afterwards the
    * outModuleReference method.
    *
    * @param node    a parse tree node
    */
   public final void caseModuleReference(Node node) {
      inModuleReference(node);
      defaultCase(node);
      outModuleReference(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTypeReference(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTypeReference(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inTypeReference method and afterwards the
    * outTypeReference method.
    *
    * @param node    a parse tree node
    */
   public final void caseTypeReference(Node node) {
      inTypeReference(node);
      defaultCase(node);
      outTypeReference(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inDefinedMacroType(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outDefinedMacroType(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inDefinedMacroType method and afterwards the
    * outDefinedMacroType method.
    *
    * @param node    a parse tree node
    */
   public final void caseDefinedMacroType(Node node) {
      inDefinedMacroType(node);
      defaultCase(node);
      outDefinedMacroType(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inDefinedMacroName(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outDefinedMacroName(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inDefinedMacroName method and afterwards the
    * outDefinedMacroName method.
    *
    * @param node    a parse tree node
    */
   public final void caseDefinedMacroName(Node node) {
      inDefinedMacroName(node);
      defaultCase(node);
      outDefinedMacroName(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSnmpObjectTypeMacroType(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSnmpObjectTypeMacroType(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSnmpObjectTypeMacroType method and afterwards the
    * outSnmpObjectTypeMacroType method.
    *
    * @param node    a parse tree node
    */
   public final void caseSnmpObjectTypeMacroType(Node node) {
      inSnmpObjectTypeMacroType(node);
      defaultCase(node);
      outSnmpObjectTypeMacroType(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSnmpTrapTypeMacroType(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSnmpTrapTypeMacroType(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSnmpTrapTypeMacroType method and afterwards the
    * outSnmpTrapTypeMacroType method.
    *
    * @param node    a parse tree node
    */
   public final void caseSnmpTrapTypeMacroType(Node node) {
      inSnmpTrapTypeMacroType(node);
      defaultCase(node);
      outSnmpTrapTypeMacroType(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSnmpAccessPart(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSnmpAccessPart(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSnmpAccessPart method and afterwards the
    * outSnmpAccessPart method.
    *
    * @param node    a parse tree node
    */
   public final void caseSnmpAccessPart(Node node) {
      inSnmpAccessPart(node);
      defaultCase(node);
      outSnmpAccessPart(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSnmpStatusPart(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSnmpStatusPart(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSnmpStatusPart method and afterwards the
    * outSnmpStatusPart method.
    *
    * @param node    a parse tree node
    */
   public final void caseSnmpStatusPart(Node node) {
      inSnmpStatusPart(node);
      defaultCase(node);
      outSnmpStatusPart(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSnmpDescrPart(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSnmpDescrPart(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSnmpDescrPart method and afterwards the
    * outSnmpDescrPart method.
    *
    * @param node    a parse tree node
    */
   public final void caseSnmpDescrPart(Node node) {
      inSnmpDescrPart(node);
      defaultCase(node);
      outSnmpDescrPart(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSnmpReferPart(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSnmpReferPart(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSnmpReferPart method and afterwards the
    * outSnmpReferPart method.
    *
    * @param node    a parse tree node
    */
   public final void caseSnmpReferPart(Node node) {
      inSnmpReferPart(node);
      defaultCase(node);
      outSnmpReferPart(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSnmpIndexPart(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSnmpIndexPart(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSnmpIndexPart method and afterwards the
    * outSnmpIndexPart method.
    *
    * @param node    a parse tree node
    */
   public final void caseSnmpIndexPart(Node node) {
      inSnmpIndexPart(node);
      defaultCase(node);
      outSnmpIndexPart(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTypeOrValueList(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTypeOrValueList(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inTypeOrValueList method and afterwards the
    * outTypeOrValueList method.
    *
    * @param node    a parse tree node
    */
   public final void caseTypeOrValueList(Node node) {
      inTypeOrValueList(node);
      defaultCase(node);
      outTypeOrValueList(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTypeOrValue(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTypeOrValue(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inTypeOrValue method and afterwards the
    * outTypeOrValue method.
    *
    * @param node    a parse tree node
    */
   public final void caseTypeOrValue(Node node) {
      inTypeOrValue(node);
      defaultCase(node);
      outTypeOrValue(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSnmpDefValPart(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSnmpDefValPart(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSnmpDefValPart method and afterwards the
    * outSnmpDefValPart method.
    *
    * @param node    a parse tree node
    */
   public final void caseSnmpDefValPart(Node node) {
      inSnmpDefValPart(node);
      defaultCase(node);
      outSnmpDefValPart(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inSnmpVarPart(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outSnmpVarPart(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inSnmpVarPart method and afterwards the
    * outSnmpVarPart method.
    *
    * @param node    a parse tree node
    */
   public final void caseSnmpVarPart(Node node) {
      inSnmpVarPart(node);
      defaultCase(node);
      outSnmpVarPart(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inVarTypes(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outVarTypes(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the inVarTypes method and afterwards the
    * outVarTypes method.
    *
    * @param node    a parse tree node
    */
   public final void caseVarTypes(Node node) {
      inVarTypes(node);
      defaultCase(node);
      outVarTypes(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inEOF(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outEOF(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseEOF(Node node) {
      inEOF(node);
      defaultCase(node);
      outEOF(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTDOT(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTDOT(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTDOT(Node node) {
      inTDOT(node);
      defaultCase(node);
      outTDOT(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTCOMMA(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTCOMMA(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTCOMMA(Node node) {
      inTCOMMA(node);
      defaultCase(node);
      outTCOMMA(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTSEMI_COLON(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTSEMI_COLON(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTSEMI_COLON(Node node) {
      inTSEMI_COLON(node);
      defaultCase(node);
      outTSEMI_COLON(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTLEFTPAREN(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTLEFTPAREN(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTLEFTPAREN(Node node) {
      inTLEFTPAREN(node);
      defaultCase(node);
      outTLEFTPAREN(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTRIGHTPAREN(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTRIGHTPAREN(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTRIGHTPAREN(Node node) {
      inTRIGHTPAREN(node);
      defaultCase(node);
      outTRIGHTPAREN(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTLEFTBRACE(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTLEFTBRACE(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTLEFTBRACE(Node node) {
      inTLEFTBRACE(node);
      defaultCase(node);
      outTLEFTBRACE(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTRIGHTBRACE(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTRIGHTBRACE(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTRIGHTBRACE(Node node) {
      inTRIGHTBRACE(node);
      defaultCase(node);
      outTRIGHTBRACE(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTLEFTBRACKET(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTLEFTBRACKET(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTLEFTBRACKET(Node node) {
      inTLEFTBRACKET(node);
      defaultCase(node);
      outTLEFTBRACKET(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTRIGHTBRACKET(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTRIGHTBRACKET(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTRIGHTBRACKET(Node node) {
      inTRIGHTBRACKET(node);
      defaultCase(node);
      outTRIGHTBRACKET(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTMINUS(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTMINUS(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTMINUS(Node node) {
      inTMINUS(node);
      defaultCase(node);
      outTMINUS(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTLESSTHAN(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTLESSTHAN(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTLESSTHAN(Node node) {
      inTLESSTHAN(node);
      defaultCase(node);
      outTLESSTHAN(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTBAR(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTBAR(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTBAR(Node node) {
      inTBAR(node);
      defaultCase(node);
      outTBAR(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTASSIGN(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTASSIGN(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTASSIGN(Node node) {
      inTASSIGN(node);
      defaultCase(node);
      outTASSIGN(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTDEFINITIONS(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTDEFINITIONS(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTDEFINITIONS(Node node) {
      inTDEFINITIONS(node);
      defaultCase(node);
      outTDEFINITIONS(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTEXPLICIT(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTEXPLICIT(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTEXPLICIT(Node node) {
      inTEXPLICIT(node);
      defaultCase(node);
      outTEXPLICIT(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTIMPLICIT(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTIMPLICIT(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTIMPLICIT(Node node) {
      inTIMPLICIT(node);
      defaultCase(node);
      outTIMPLICIT(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTTAGS(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTTAGS(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTTAGS(Node node) {
      inTTAGS(node);
      defaultCase(node);
      outTTAGS(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTBEGIN(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTBEGIN(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTBEGIN(Node node) {
      inTBEGIN(node);
      defaultCase(node);
      outTBEGIN(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTEND(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTEND(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTEND(Node node) {
      inTEND(node);
      defaultCase(node);
      outTEND(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTEXPORTS(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTEXPORTS(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTEXPORTS(Node node) {
      inTEXPORTS(node);
      defaultCase(node);
      outTEXPORTS(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTIMPORTS(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTIMPORTS(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTIMPORTS(Node node) {
      inTIMPORTS(node);
      defaultCase(node);
      outTIMPORTS(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTFROM(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTFROM(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTFROM(Node node) {
      inTFROM(node);
      defaultCase(node);
      outTFROM(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTMACRO(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTMACRO(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTMACRO(Node node) {
      inTMACRO(node);
      defaultCase(node);
      outTMACRO(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTINTEGER(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTINTEGER(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTINTEGER(Node node) {
      inTINTEGER(node);
      defaultCase(node);
      outTINTEGER(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTREAL(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTREAL(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTREAL(Node node) {
      inTREAL(node);
      defaultCase(node);
      outTREAL(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTBOOLEAN(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTBOOLEAN(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTBOOLEAN(Node node) {
      inTBOOLEAN(node);
      defaultCase(node);
      outTBOOLEAN(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTNULL(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTNULL(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTNULL(Node node) {
      inTNULL(node);
      defaultCase(node);
      outTNULL(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTBIT(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTBIT(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTBIT(Node node) {
      inTBIT(node);
      defaultCase(node);
      outTBIT(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTOCTET(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTOCTET(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTOCTET(Node node) {
      inTOCTET(node);
      defaultCase(node);
      outTOCTET(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTSTRING(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTSTRING(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTSTRING(Node node) {
      inTSTRING(node);
      defaultCase(node);
      outTSTRING(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTENUMERATED(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTENUMERATED(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTENUMERATED(Node node) {
      inTENUMERATED(node);
      defaultCase(node);
      outTENUMERATED(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTSEQUENCE(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTSEQUENCE(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTSEQUENCE(Node node) {
      inTSEQUENCE(node);
      defaultCase(node);
      outTSEQUENCE(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTSET(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTSET(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTSET(Node node) {
      inTSET(node);
      defaultCase(node);
      outTSET(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTOF(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTOF(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTOF(Node node) {
      inTOF(node);
      defaultCase(node);
      outTOF(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTCHOICE(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTCHOICE(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTCHOICE(Node node) {
      inTCHOICE(node);
      defaultCase(node);
      outTCHOICE(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTUNIVERSAL(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTUNIVERSAL(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTUNIVERSAL(Node node) {
      inTUNIVERSAL(node);
      defaultCase(node);
      outTUNIVERSAL(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTAPPLICATION(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTAPPLICATION(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTAPPLICATION(Node node) {
      inTAPPLICATION(node);
      defaultCase(node);
      outTAPPLICATION(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTPRIVATE(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTPRIVATE(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTPRIVATE(Node node) {
      inTPRIVATE(node);
      defaultCase(node);
      outTPRIVATE(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTANY(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTANY(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTANY(Node node) {
      inTANY(node);
      defaultCase(node);
      outTANY(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTDEFINED(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTDEFINED(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTDEFINED(Node node) {
      inTDEFINED(node);
      defaultCase(node);
      outTDEFINED(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTBY(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTBY(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTBY(Node node) {
      inTBY(node);
      defaultCase(node);
      outTBY(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTOBJECT(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTOBJECT(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTOBJECT(Node node) {
      inTOBJECT(node);
      defaultCase(node);
      outTOBJECT(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTIDENTIFIER(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTIDENTIFIER(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTIDENTIFIER(Node node) {
      inTIDENTIFIER(node);
      defaultCase(node);
      outTIDENTIFIER(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTINCLUDES(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTINCLUDES(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTINCLUDES(Node node) {
      inTINCLUDES(node);
      defaultCase(node);
      outTINCLUDES(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTMIN(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTMIN(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTMIN(Node node) {
      inTMIN(node);
      defaultCase(node);
      outTMIN(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTMAX(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTMAX(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTMAX(Node node) {
      inTMAX(node);
      defaultCase(node);
      outTMAX(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTSIZE(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTSIZE(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTSIZE(Node node) {
      inTSIZE(node);
      defaultCase(node);
      outTSIZE(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTWITH(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTWITH(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTWITH(Node node) {
      inTWITH(node);
      defaultCase(node);
      outTWITH(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTCOMPONENT(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTCOMPONENT(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTCOMPONENT(Node node) {
      inTCOMPONENT(node);
      defaultCase(node);
      outTCOMPONENT(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTCOMPONENTS(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTCOMPONENTS(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTCOMPONENTS(Node node) {
      inTCOMPONENTS(node);
      defaultCase(node);
      outTCOMPONENTS(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTPRESENT(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTPRESENT(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTPRESENT(Node node) {
      inTPRESENT(node);
      defaultCase(node);
      outTPRESENT(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTABSENT(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTABSENT(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTABSENT(Node node) {
      inTABSENT(node);
      defaultCase(node);
      outTABSENT(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTOPTIONAL(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTOPTIONAL(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTOPTIONAL(Node node) {
      inTOPTIONAL(node);
      defaultCase(node);
      outTOPTIONAL(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTDEFAULT(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTDEFAULT(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTDEFAULT(Node node) {
      inTDEFAULT(node);
      defaultCase(node);
      outTDEFAULT(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTTRUE(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTTRUE(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTTRUE(Node node) {
      inTTRUE(node);
      defaultCase(node);
      outTTRUE(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTFALSE(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTFALSE(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTFALSE(Node node) {
      inTFALSE(node);
      defaultCase(node);
      outTFALSE(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTPLUS_INFINITY(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTPLUS_INFINITY(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTPLUS_INFINITY(Node node) {
      inTPLUS_INFINITY(node);
      defaultCase(node);
      outTPLUS_INFINITY(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTMINUS_INFINITY(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTMINUS_INFINITY(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTMINUS_INFINITY(Node node) {
      inTMINUS_INFINITY(node);
      defaultCase(node);
      outTMINUS_INFINITY(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTOBJECT_TYPE(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTOBJECT_TYPE(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTOBJECT_TYPE(Node node) {
      inTOBJECT_TYPE(node);
      defaultCase(node);
      outTOBJECT_TYPE(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTTRAP_TYPE(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTTRAP_TYPE(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTTRAP_TYPE(Node node) {
      inTTRAP_TYPE(node);
      defaultCase(node);
      outTTRAP_TYPE(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTSYNTAX(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTSYNTAX(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTSYNTAX(Node node) {
      inTSYNTAX(node);
      defaultCase(node);
      outTSYNTAX(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTACCESS(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTACCESS(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTACCESS(Node node) {
      inTACCESS(node);
      defaultCase(node);
      outTACCESS(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTSTATUS(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTSTATUS(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTSTATUS(Node node) {
      inTSTATUS(node);
      defaultCase(node);
      outTSTATUS(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTDESCRIPTION(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTDESCRIPTION(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTDESCRIPTION(Node node) {
      inTDESCRIPTION(node);
      defaultCase(node);
      outTDESCRIPTION(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTREFERENCE(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTREFERENCE(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTREFERENCE(Node node) {
      inTREFERENCE(node);
      defaultCase(node);
      outTREFERENCE(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTINDEX(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTINDEX(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTINDEX(Node node) {
      inTINDEX(node);
      defaultCase(node);
      outTINDEX(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTDEFVAL(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTDEFVAL(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTDEFVAL(Node node) {
      inTDEFVAL(node);
      defaultCase(node);
      outTDEFVAL(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTENTERPRISE(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTENTERPRISE(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTENTERPRISE(Node node) {
      inTENTERPRISE(node);
      defaultCase(node);
      outTENTERPRISE(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTVARIABLES(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTVARIABLES(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTVARIABLES(Node node) {
      inTVARIABLES(node);
      defaultCase(node);
      outTVARIABLES(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTBINSTRING(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTBINSTRING(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTBINSTRING(Node node) {
      inTBINSTRING(node);
      defaultCase(node);
      outTBINSTRING(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTHEXSTRING(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTHEXSTRING(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTHEXSTRING(Node node) {
      inTHEXSTRING(node);
      defaultCase(node);
      outTHEXSTRING(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTCSTRING(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTCSTRING(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTCSTRING(Node node) {
      inTCSTRING(node);
      defaultCase(node);
      outTCSTRING(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTUCASEFIRST_IDENT(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTUCASEFIRST_IDENT(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTUCASEFIRST_IDENT(Node node) {
      inTUCASEFIRST_IDENT(node);
      defaultCase(node);
      outTUCASEFIRST_IDENT(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTLCASEFIRST_IDENT(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTLCASEFIRST_IDENT(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTLCASEFIRST_IDENT(Node node) {
      inTLCASEFIRST_IDENT(node);
      defaultCase(node);
      outTLCASEFIRST_IDENT(node);
   }

   /**
    * Called upon entering a node of this type. By default calls
    * the defaultIn method.
    *
    * @param node   a parse tree node
    */
   protected void inTNUMBER(Node node) {
      defaultIn(node);
   }

   /**
    * Called upon exiting a node of this type. By default calls
    * the defaultOut method.
    *
    * @param node   a parse tree node
    */
   protected void outTNUMBER(Node node) {
      defaultOut(node);
   }

   /**
    * Traverses all the children to a node of this type, first
    * calling the appropriate inXXX method and afterwards the
    * corresponding outXXX method.
    *
    * @param node    a parse tree node
    */
   public final void caseTNUMBER(Node node) {
      inTNUMBER(node);
      defaultCase(node);
      outTNUMBER(node);
   }

   /**
    * A default method called from all inXXX methods. Override
    * this method in a subclass to handle the default case.
    *
    * @param  node   a parse tree node
    */
   protected void defaultIn(Node node) {
   }

   /**
    * A default method called from all outXXX methods. Override
    * this method in a subclass to handle the default case.
    *
    * @param  node   a parse tree node
    */
   protected void defaultOut(Node node) {
   }

   /**
    * Traverses the child nodes recursively by switching them
    * on this object.
    *
    * @param  node   a parse tree node
    */
   protected final void defaultCase(Node node) {
       int   i = 0;

       while (i < node.children()) {
          node.childAt(i).apply(this);
          i++;
       }
   }
}
