/*=======================================================================
 * Project: MediaTeX
 * Module : Get archive from a seda v0.2 archive transfer
 *
 * Example using libmediatex.a

 MediaTex is an Electronic Records Management System
 Copyright (C) 2016  Nicolas Roche
 
 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 any later version.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 =======================================================================*/

#include "mediatex.h"
#include "misc.h"

#include <locale.h>

#include "seda02.h"

// Sax entry point
extern int parse_xml_file (BisonData* bisonData, const char *filename);

#undef MEMORY_CVSPRINT_MAX
#define MEMORY_CVSPRINT_MAX 1*GIGA

/*=======================================================================
 * Function   : extractSedaZipFile
 * Description: extract zip file
 * Synopsis   : int extractSedaZipFile(char* zipSedaPath)
 * Input      : char* zipSedaPath
 * Output     : TRUE on success
 * Note       : -o to force replacement
 =======================================================================*/
int 
extractSedaZipFile(char* zipSedaPath)
{
  int rc = FALSE;
  char* argv[] = {"/usr/bin/unzip", "-o", 0, 0};

  // call unzip
  argv[2] = zipSedaPath;
  if (!execScript(argv, 0, 0, FALSE)) goto error;

  rc = TRUE;
 error:
  if (!rc) {
     logMain(LOG_ERR, "%s", "extractSedaZipFile fails");
  }
  return rc;
}

/*=======================================================================
 * Function   : removeZipDirectory
 * Description: remove the extracted zip directory
 * Synopsis   : int removeZipDirectory(char* zipPath)
 * Input      : char* zipPath
 * Output     : TRUE on success
 =======================================================================*/
int 
removeZipDirectory(char* zipPath)
{
  int rc = FALSE;
  char* argv[] = {"/bin/rm", "-fr", 0, 0};

  // call rm
  argv[2] = zipPath;
  if (!execScript(argv, 0, 0, FALSE)) goto error;

  rc = TRUE;
 error:
  if (!rc) {
     logMain(LOG_ERR, "%s", "removeZipDirectory fails");
  }
  return rc;
}

/*=======================================================================
 * Function   : addZipFileContainer
 * Description: Load the main extraction container
 * Synopsis   : 
 * Input      : BisonData data
 *              char* zipSedaPath
 *              char* xmlSedaPath
 * Output     : TRUE on success
 =======================================================================*/
int 
addSedaZipFileContainer(BisonData* data, 
			char* zipSedaPath, char* xmlSedaPath)
{
  int rc = FALSE;
  char hash[MAX_SIZE_MD5+1];
  off_t size = 0;
  Archive* archive = 0;
  Carac* carac = 0;
 
  hash[MAX_SIZE_MD5] = 0;
  if (!(carac = addCarac(data->coll, "format"))) goto error;

  // zip file
  if (!getArchiveIds(zipSedaPath, hash, &size)) goto error;
  if (!(archive = addArchive(data->coll, hash, size))) goto error;
  if (!(data->zipSedaFile = addContainer(data->coll, ZIP, archive))) goto error;
  if (!addAssoCarac(data->coll, carac, ARCH, archive, "ZIP"));

  /*
  // xml file
  if (!getArchiveIds(xmlSedaPath, hash, &size)) goto error;
  if (!(data->xmlSedaFile = addArchive(data->coll, hash, size))) goto error;
  if (!addAssoCarac(data->coll, carac, ARCH, data->xmlSedaFile, "XML"));
  */

  rc = TRUE;
 error:
  if (!rc) {
     logMain(LOG_ERR, "%s", "addSedaZipFileContainer fails");
  }
  return rc;
}

/*=======================================================================
 * Function   : usage
 * Description: Print the usage.
 * Synopsis   : static void usage(char* programName)
 * Input      : programName = the name of the program; usually
 *                                  argv[0].
 * Output     : N/A
 =======================================================================*/
static void 
usage(char* programName)
{
  mdtxUsage(programName);
  fprintf(stderr, "\n\t\t-b basename");
  mdtxOptions();
  fprintf(stderr, "  ---\n"
	  "  -b, --basename\tseda v0.2 XML transfer basename\n");
  return;
}


/*=======================================================================
 * Function   : main 
 * Author     : Nicolas ROCHE
 * modif      : 2010/12/10
 * Description: Entry point for mdtx wrapper
 * Synopsis   : ./mdtx
 * Input      : stdin
 * Output     : rtfm
 =======================================================================*/
int 
main(int argc, char** argv)
{
  char zipSedaPath[128];
  char xmlSedaPath[128];
  char tmp[128];
  BisonData data;
  CvsFile fd = {0, 0, 0, FALSE, 0, cvsCatOpen, cvsCatPrint};
  // ---
  int rc = 0;
  int cOption = EOF;
  char* programName = *argv;
  char* options = MDTX_SHORT_OPTIONS"b:";
  struct option longOptions[] = {
    {"sedaBaseName", required_argument, 0, 'b'},
    MDTX_LONG_OPTIONS,
    {0, 0, 0, 0}
  };

  setlocale (LC_ALL, "");
  setlocale(LC_NUMERIC, "C"); // so as printf do not write comma in float

  // import mdtx environment
  env.cvsprintMax = 1*GIGA;
  getEnv(&env);
  memset(&data, 0, sizeof(BisonData));
  
  // parse the command line
  while((cOption = getopt_long(argc, argv, options, longOptions, 0)) 
	!= EOF) {
    switch(cOption) {
      
    case 'b':
      if(isEmptyString(optarg)) {
	fprintf(stderr, 
		"%s: nil or empty argument for the seda basename\n",
		programName);
	rc = 2;
      }
      else {
	if (!(data.basename = 
	      (char*)malloc(sizeof(char) * strlen(optarg) + 1))) {
	  fprintf(stderr, 
		  "%s: cannot allocate memory for the seda basename\n", 
		  programName);
	  rc = 3;
	}
	else {
	  strcpy(data.basename, optarg);
	}
      }
      break;

      GET_MDTX_OPTIONS; // generic options
    }
    if (rc) goto optError;
  }
      
  // export mdtx environment
  env.allocDiseaseCallBack = clientDiseaseAll;
  if (!setEnv(programName, &env)) goto optError;
 
  /************************************************************************/
  if (!data.basename) {
    usage(argv[0]); // expect a basename to work
    goto optError;
  }

  logMain(LOG_INFO, "xml2txt: %s", data.basename);

  // get input file's path
  if (sprintf(zipSedaPath, "%s.zip", data.basename)<=0) goto error;
  if (sprintf(xmlSedaPath, "%s.xml", data.basename)<=0) goto error;
  
  // initialise data used by bison
  if (!(data.coll = addCollection("seda02"))) goto error;
  if (!(data.coll->extractTree = createExtractTree())) goto error;
  if (!(data.coll->catalogTree = createCatalogTree())) goto error;
  if (!addCarac(data.coll, "sha1")) goto error;
  if (!(data.locataire = addRole(data.coll, "Locataire"))) goto error;
  if (!(data.transferringAgency = 
	addRole(data.coll, "Transferring Agency"))) goto error;
  if (!(data.archivalAgency = addRole(data.coll, "Archival Agency")))
      goto error;

  // extract zip content
  if (!extractSedaZipFile(zipSedaPath)) goto error;
  
  // build the extraction container from the zip file
  if (!addSedaZipFileContainer(&data, zipSedaPath, xmlSedaPath)) 
    goto error;
  
  // ask sax/bison to parse the seda xml input file
  if (!parse_xml_file(&data, xmlSedaPath)) goto error;

  // remove the extracted zip directory
  if (!removeZipDirectory(data.basename)) goto error;
  
  // add zip file to the catalog
  if (!addArchiveToDocument(data.coll, data.zipSedaFile->parent, 
			    data.transfer)) goto error;
  /*
  if (!addArchiveToDocument(data.coll, data.xmlSedaFile, 
			    data.transfer)) goto error;
  */

  // serialize the resulting mediatex metadata
  if (!(data.coll->user = createString("mdtx"))) goto error;
  if (sprintf(tmp, "%s_ext", data.basename) <= 0) goto error;
  if (!(data.coll->extractDB = createString(tmp))) goto error;
  if (sprintf(tmp, "%s_cat", data.basename) <= 0) goto error;
  if (!(data.coll->catalogDB = createString(tmp))) goto error;
  data.coll->memoryState |= EXPANDED;
  env.noRegression = TRUE;
  logMain(LOG_NOTICE, "serialize '%s'", data.coll->extractDB);
  if (!(serializeExtractTree(data.coll, &fd))) goto error;
  logMain(LOG_NOTICE, "serialize '%s'", data.coll->catalogDB);
  if (!(serializeCatalogTree(data.coll, &fd))) goto error;
  
  rc = TRUE;
  /************************************************************************/

  rc = TRUE;
 error:
  if (data.basename) free(data.basename);
  freeConfiguration();
  ENDINGS;
  rc=!rc;
 optError:
  exit(rc);
}

/* Local Variables: */
/* mode: c */
/* mode: font-lock */
/* mode: auto-fill */
/* End: */
