/*=======================================================================
 * Version: $Id: txt2sql.c,v 1.2 2015/07/02 12:14:32 nroche Exp $
 * Project: MediaTeX
 * Module : Reversibility use case
 *
 * Example using libmediatex.a

 MediaTex is an Electronic Records Management System
 Copyright (C) 2016  Nicolas Roche
 
 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 any later version.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 =======================================================================*/

#include <mediatex.h>
#include <client/commonHtml.h>
#include <client/catalogHtml.h>

#include <locale.h>
#include <avl.h>

typedef struct Data {
  Collection *coll;
  char* sqlFile;
  FILE* fd;
} Data;

/*=======================================================================
 * Function   : sqlString
 * Description: replace special caracteres
 * Synopsis   : char* sqlString(char* string)
 * Input      : char* string : the string to rewrite
 * Output     : char* : modified input buffer
 =======================================================================*/
char*
sqlString(char* string, char* buffer)
{
  char* ptr = buffer;
  int len = 0;
  char car = 0;
  int i;

  len = strlen(string);

  for (i=0; i<len; ++i) {
    car = string[i];

    switch (car) {
    case '\'':
      *ptr++ = car;
      *ptr++ = car;
      break;
    default:
      *ptr++ = car;
    }
  }

  *ptr=0;
  return buffer;
}

/*=======================================================================
 * Function   : sqlString1
 * Description: replace special caracteres
 * Synopsis   : char* sqlString1(char* string)
 * Input      : char* string : the string to rewrite
 * Output     : char* : a static buffer
 =======================================================================*/
char*
sqlString1(char* string)
{
  static char buffer[128];
  return sqlString(string, buffer);
}

/*=======================================================================
 * Function   : sqlString2
 * Description: replace special caracteres
 * Synopsis   : char* sqlString2(char* string)
 * Input      : char* string : the string to rewrite
 * Output     : char* : a static buffer
 =======================================================================*/
char*
sqlString2(char* string)
{
  static char buffer[128];
  return sqlString(string, buffer);
}

/*=======================================================================
 * Function   : sqlString3
 * Description: replace special caracteres
 * Synopsis   : char* sqlString3(char* string)
 * Input      : char* string : the string to rewrite
 * Output     : char* : a static buffer
 =======================================================================*/
char*
sqlString3(char* string)
{
  static char buffer[128];
  return sqlString(string, buffer);
}

/*=======================================================================
 * Function   : sqlCreateTables
 * Description: create all sql tables
 * Synopsis   : void sqlCreateTables(Data* data)
 * Input      : Data* data
 * Output     : N/A
 =======================================================================*/
void
sqlCreateTables(Data* data)
{
  fprintf(data->fd, "\n"
	  "DROP INDEX IF EXISTS %s_%s_numloc;\n"
	  "DROP INDEX IF EXISTS %s_%s_numint;\n"
	  "DROP INDEX IF EXISTS %s_%s_famname;\n"
	  "DROP INDEX IF EXISTS %s_%s_persname;\n"
	  "DROP INDEX IF EXISTS %s_%s_annee;\n"
	  "DROP INDEX IF EXISTS %s_%s_mois;\n"
	  "DROP TABLE IF EXISTS %s_%s;\n"
	  "\n"
	  "CREATE TABLE %s_%s (\n"

	  /* ha76 metadata */
	  "    numavis integer,\n"
	  "    numloc CHAR(32),\n"
	  "    numint CHAR(32),\n"
	  "    famname CHAR(32),\n"
	  "    persname CHAR(32),\n"
	  "    annee smallint,\n"
	  "    mois smallint,\n"

	  /* presentation */
	  "    title CHAR(128),\n"
	  "    file CHAR(32),\n"

	  /* mdtx metadata */
	  "    urlData CHAR(128),\n"
	  "    urlMetadata CHAR(128),\n"

	  "    PRIMARY KEY (numavis)\n"
	  ");\n"
	  "CREATE INDEX %s_%s_numloc ON %s_%s(numloc);\n"
	  "CREATE INDEX %s_%s_numint ON %s_%s(numint);\n"
	  "CREATE INDEX %s_%s_famname ON %s_%s(famname);\n"
	  "CREATE INDEX %s_%s_persname ON %s_%s(persname);\n"
	  "CREATE INDEX %s_%s_annee ON %s_%s(annee);\n"
	  "CREATE INDEX %s_%s_mois ON %s_%s(mois);\n\n",
	 env.confLabel, data->coll->label,
	 env.confLabel, data->coll->label,
	 env.confLabel, data->coll->label,
	 env.confLabel, data->coll->label,
	 env.confLabel, data->coll->label,
	 env.confLabel, data->coll->label,
	 env.confLabel, data->coll->label,
	 env.confLabel, data->coll->label,
	 env.confLabel, data->coll->label,
	 env.confLabel, data->coll->label,
	 env.confLabel, data->coll->label,
	 env.confLabel, data->coll->label,
	 env.confLabel, data->coll->label,
	 env.confLabel, data->coll->label,
	 env.confLabel, data->coll->label,
	 env.confLabel, data->coll->label,
	 env.confLabel, data->coll->label,
	 env.confLabel, data->coll->label,
	 env.confLabel, data->coll->label,
	 env.confLabel, data->coll->label);

  return;
}


/*=======================================================================
 * Function   : sqlAvis
 * Description: Sqlise mediatex metadata
 * Synopsis   : int sqlAvis(Data* data)
 * Input      : Data* data
 * Output     : TRUE on success
 =======================================================================*/
int 
sqlAvis(Data* data)
{ 
  int rc = FALSE;
  Archive* archive = 0;
  Document* document = 0;
  AssoRole* assoRole = 0;
  AssoCarac* assoCarac = 0;
  AVLNode *node = 0;
  RGIT* curr = 0;
  int numavis = 0;
  char numloc[128];
  char numint[128];
  char* famname = 0;
  char* persname = 0;
  short int annee = 0;
  short int mois = 0;
  char* title = 0;
  char* file = 0;
  char url[128];
  int i = 0;
  
  checkCollection(data->coll);
  if (!data->coll->archives) goto error;
  logMain(LOG_DEBUG, "sql %s avis", data->coll->label);

  env.progBar.cur = 0;
  env.progBar.max = avl_count(data->coll->archives);
  startProgBar("serialize");

  // loop on archives
  if (!avl_count(data->coll->archives)) {
    logMain(LOG_WARNING, "%s", "no archive found");
    goto end;
  }

  fprintf(data->fd, "INSERT INTO %s_%s VALUES", 
	 env.confLabel, data->coll->label);

  for (node = data->coll->archives->head; node; node = node->next) {
    archive = node->item;
    numavis = 0;
    numloc[0] = 0;
    numint[0] = 0;
    famname = 0;
    persname = 0;
    annee = 0;
    mois = 0;
    title = 0;
    file = 0;

    logMain(LOG_DEBUG, "sql archive (%s:%lli)...", 
	    archive->hash, archive->size);

    // one and only one document per archive
    if (archive->documents->nbItems != 1) {
      logMain(LOG_ERR, "find %i document(s) (expected 1)",
	      archive->documents->nbItems);
      goto error;
    }
    if (!(document = archive->documents->head->it)) {
      logMain(LOG_ERR, "%s", "no document");
      goto error;
    }

    logMain(LOG_DEBUG, "... alias %s", document->label);

    if (document->assoRoles->nbItems == 2) {
      logMain(LOG_DEBUG, "skip archive transfert archive %s", 
	      document->label);
      continue;
    }

    // one and only one human per archive
    if (document->assoRoles->nbItems != 1) {
      logMain(LOG_ERR, "find %i document(s) (expected 1)",
	      document->assoRoles->nbItems);
      goto error;
    }
    if (!(assoRole = document->assoRoles->head->it)) {
      logMain(LOG_ERR, "%s", "no role association");
      goto error;
    }

    // get archive caracs
    curr = 0;
    while ((assoCarac = rgNext_r(archive->assoCaracs, &curr))) {

      if (!strcmp(assoCarac->carac->label, "Description")) {
	file = assoCarac->value;
      }
    }

    // get document caracs
    curr = 0;
    while ((assoCarac = rgNext_r(document->assoCaracs, &curr))) {

      if (!strcmp(assoCarac->carac->label, "Content Description")) {
	title = sqlString1(assoCarac->value);
      }
      if (!strcmp(assoCarac->carac->label, "numavis")) {
	if (!sscanf(assoCarac->value, "%i", &numavis)) goto error;
      }
      if (!strcmp(assoCarac->carac->label, "numloc")) {
	if (!sscanf(assoCarac->value, "%s", numloc)) goto error;
      }
      if (!strcmp(assoCarac->carac->label, "numint")) {
	if (!sscanf(assoCarac->value, "%s", numint)) goto error;
      }
      if (!strcmp(assoCarac->carac->label, "annee")) {
	if (!sscanf(assoCarac->value, "%hi", &annee)) goto error;
      }
      if (!strcmp(assoCarac->carac->label, "mois")) {
	if (!sscanf(assoCarac->value, "%hi", &mois)) goto error;
      }
    }

    if (!numavis) {
      logMain(LOG_ERR, "%s", "cannot find numavis (primary key)");
      goto error;
    }

    // get human carrac
    famname = sqlString2(assoRole->human->firstName);
    persname = sqlString3(assoRole->human->secondName); 

    // get url on mediatex static html pages
    if (!getDocumentUri(url, "/", document->id)) goto error;
   
    fprintf(data->fd, "%s\n  ", i?",":"");
    fprintf(data->fd, "('%i', '%s', '%s', '%s', '%s', '%hi', '%hi',"
	   " '%s', '%s',"
	   " '~%s/cgi/get.cgi?hash=%s&size=%lli',"
	   " '~%s/index%s')", 
	   numavis, numloc, numint, famname, persname, annee, mois,
	   title, file,
	   data->coll->user, archive->hash, archive->size,
	   data->coll->user, url);
    ++env.progBar.cur;
    ++i;
  }
  fprintf(data->fd, ";\n\n");

 end:
  logMain(LOG_NOTICE, "%i avis serialized", i);
  rc = TRUE;
 error:
  if (!rc) {
    logMain(LOG_ERR, "%s", "sqlAvis fails");
  }
  stopProgBar();
  return rc;
}

/*=======================================================================
 * Function   : usage
 * Description: Print the usage.
 * Synopsis   : static void usage(char* programName)
 * Input      : programName = the name of the program; usually
 *                                  argv[0].
 * Output     : N/A
 =======================================================================*/
static void 
usage(char* programName)
{
  mdtxUsage(programName);
  fprintf(stderr, "\n\t\t-i label [ -o sqlFile ]");
  mdtxOptions();
  fprintf(stderr, "  ---\n"
	  "  -i, --label\t\tcollection's label\n"
	  "  -o, --sqlFile\t\toutput file\n");
  return;
}


/*=======================================================================
 * Function   : main 
 * Author     : Nicolas ROCHE
 * modif      : 2015/05/28
 * Description: Entry point for mdtx wrapper
 * Synopsis   : ./mdtx
 * Input      : stdin
 * Output     : rtfm
 =======================================================================*/
int 
main(int argc, char** argv)
{
  Data data;
  char* optargs[] = {0, 0};
  // ---
  int rc = 0;
  int cOption = EOF;
  char* programName = *argv;
  char* options = MDTX_SHORT_OPTIONS"i:o:";
  struct option longOptions[] = {
    MDTX_LONG_OPTIONS,
    {"label", required_argument, 0, 'i'},
    {"sqlFile", required_argument, 0, 'o'},
    {0, 0, 0, 0}
  };

  memset (&data, 0, sizeof(Data));
  data.fd = stdout;
  setlocale (LC_ALL, "");
  setlocale (LC_NUMERIC, "C"); // so as printf do not write comma in float

  // import mdtx environment
  getEnv(&env);

  // parse the command line
  while ((cOption = getopt_long(argc, argv, options, longOptions, 0)) 
	 != EOF) {
    switch(cOption) {
      
    case 'i':
      optargs[0] = optarg;
      break;
      
    case 'o':
      optargs[1] = optarg;
      break;
      
      GET_MDTX_OPTIONS; // generic options
    }
    if (rc) goto optError;
  }
  
  // export mdtx environment
  env.allocDiseaseCallBack = 0;
  if (!setEnv(programName, &env)) goto optError;
  
  /************************************************************************/
  if (!optargs[0]) {
    usage(argv[0]); // options expected
    goto optError;
  }
  
  logMain(LOG_INFO, "%s", "** txt2sql: %s **");
  
  if (optargs[1]) {
    if (!(data.sqlFile = createString(optargs[1]))) goto error;
    if (!(data.fd = fopen(data.sqlFile, "w"))) {
      logMain(LOG_ERR, "fopen %s fails: %s", 
	      data.sqlFile, strerror(errno));
      goto error;
    }
  }

  if (!(data.coll = mdtxGetCollection(optargs[0]))) goto error;
  if (!loadCollection(data.coll, CTLG)) goto error;
  sqlCreateTables(&data);
  if (!sqlAvis(&data)) goto error;

  if (data.sqlFile  && fclose(data.fd)) {
    logMain(LOG_ERR, "fclose fails: %s", strerror(errno));
    goto error;
  }

  if (!releaseCollection(data.coll, CTLG)) goto error;
  /************************************************************************/

  freeConfiguration();
  rc = TRUE;
 error:
  freeConfiguration();
  ENDINGS;
  rc=!rc;
 optError:
  exit(rc);
}

/* Local Variables: */
/* mode: c */
/* mode: font-lock */
/* mode: auto-fill */
/* End: */
