/*=======================================================================
 * Version: $Id: txt2sql.c,v 1.3 2014-04-28 16:07:45 nroche Exp $
 * Project: MediaTeX
 * Module : Reversibility use case
 *
 * Example using libmediatex.a

 MediaTex is an Electronic Records Management System
 Copyright (C) 2014  Nicolas Roche
 
 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 any later version.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 =======================================================================*/

#include "mediatex.h"
#include "misc/log.h"
#include "misc/command.h"
#include "memory/confTree.h"
#include "common/openClose.h"
#include "client/commonHtml.h"
#include "client/catalogHtml.h"

#include <locale.h>
#include <avl.h>

GLOBAL_STRUCT_DEF_BIN;

/*=======================================================================
 * Function   : sqlCreateTables
 * Description: create all sql tables
 * Synopsis   : void sqlCreateTables()
 * Input      : N/A
 * Output     : N/A
 =======================================================================*/
void
sqlCreateTables(Collection* coll)
{
  printf("\n"
	 "DROP TABLE IF EXISTS %s_%s_category_childs;\n"
	 "DROP TABLE IF EXISTS %s_%s_category_documents;\n"
	 "DROP TABLE IF EXISTS %s_%s_category_humans;\n"
	 "DROP TABLE IF EXISTS %s_%s_document_archives;\n"
	 "DROP TABLE IF EXISTS %s_%s_document_role_humans;\n"
	 "\n"
	 "DROP TABLE IF EXISTS %s_%s_archive_caracs;\n"
	 "DROP TABLE IF EXISTS %s_%s_document_caracs;\n"
	 "DROP TABLE IF EXISTS %s_%s_human_caracs;\n"
	 "DROP TABLE IF EXISTS %s_%s_category_caracs;\n"
	 "\n"
	 "DROP TABLE IF EXISTS %s_%s_container_archives;\n"
	 "DROP TABLE IF EXISTS %s_%s_server_archives;\n"
	 "\n"
	 "DROP TABLE IF EXISTS %s_%s_servers;\n"
	 "DROP TABLE IF EXISTS %s_%s_containers;\n"
	 "DROP TABLE IF EXISTS %s_%s_archives;\n"
	 "DROP TABLE IF EXISTS %s_%s_documents;\n"
	 "DROP TABLE IF EXISTS %s_%s_humans;\n"
	 "DROP TABLE IF EXISTS %s_%s_categories;\n"
	 "DROP TABLE IF EXISTS %s_%s_roles;\n"
	 "DROP TABLE IF EXISTS %s_%s_caracs;\n"
	 "\n"
	 "-- Entities\n"
	 "\n"
	 "CREATE TABLE %s_%s_archives (\n"
	 "    hash VARCHAR(32),\n"
	 "    size int8,\n"
	 "    score real,\n"
	 "    url VARCHAR(128),\n"
	 "    PRIMARY KEY (hash, size)\n"
	 ");\n"
	 "\n"
	 "CREATE TABLE %s_%s_containers (\n"
	 "    hash VARCHAR(32),\n"
	 "    size int8,\n"
	 "    FOREIGN KEY (hash, size) "
	 "REFERENCES %s_%s_archives(hash, size),\n"
	 "    type VARCHAR(5),\n"
	 "    PRIMARY KEY (hash, size, type)\n"
	 ");\n"
	 "\n"
	 "CREATE TABLE %s_%s_servers (\n"
	 "    fingerprint VARCHAR(32),\n"
	 "    hostname VARCHAR(64),\n"
	 "    PRIMARY KEY (fingerprint)\n"
	 ");\n"
	 "\n"
	 "CREATE TABLE %s_%s_documents (\n"
	 "    label VARCHAR(128),\n"
	 "    url VARCHAR(128),\n"
	 "    PRIMARY KEY (label)\n"
	 ");\n"
	 "\n"
	 "CREATE TABLE %s_%s_humans (\n"
	 "    name1 VARCHAR(64),\n"
	 "    name2 VARCHAR(64),\n"
	 "    url VARCHAR(128),\n"
	 "    PRIMARY KEY (name1, name2)\n"
	 ");\n"
	 "\n"
	 "CREATE TABLE %s_%s_categories (\n"
	 "    label  VARCHAR(128),\n"
	 "    url VARCHAR(128),\n"
	 "    PRIMARY KEY (label)\n"
	 ");\n"
	 "\n"
	 "CREATE TABLE %s_%s_roles (\n"
	 "    label VARCHAR(128),\n"
	 "    url VARCHAR(128),\n"
	 "    PRIMARY KEY (label)\n"
	 ");\n"
	 "\n"
	 "CREATE TABLE %s_%s_caracs (\n"
	 "    label VARCHAR(128),\n"
	 "    PRIMARY KEY (label)\n"
	 ");\n"
	 "\n"
	 "-- Associations\n"
	 "\n"
	 "CREATE TABLE %s_%s_container_archives (\n"
	 "    fatherHash VARCHAR(32),\n"
	 "    fatherSize int8,\n"
	 "    FOREIGN KEY (fatherHash, fatherSize) "
	 "REFERENCES %s_%s_archives(hash, size),\n"
	 "    childHash VARCHAR(32),\n"
	 "    childSize int8,\n"
	 "    FOREIGN KEY (childHash, childSize) "
	 "REFERENCES %s_%s_archives(hash, size),\n"
	 "    path VARCHAR (512),\n"
	 "    PRIMARY KEY (fatherHash, fatherSize, childHash, childSize)\n"
	 ");\n"
	 "\n"
	 "CREATE TABLE %s_%s_server_archives (\n"
	 "    server VARCHAR(32) "
	 "REFERENCES %s_%s_servers(fingerprint),\n"
	 "    hash VARCHAR(32),\n"
	 "    size int8,\n"
	 "    FOREIGN KEY (hash, size) "
	 "REFERENCES %s_%s_archives(hash, size),\n"
	 "    score real,\n"
	 "    PRIMARY KEY (server, hash, size)\n"
	 ");\n"
	 "\n"
	 "CREATE TABLE %s_%s_archive_caracs (\n"
	 "    hash  VARCHAR(32),\n"
	 "    size  int8, \n"
	 "    FOREIGN KEY (hash, size) "
	 "REFERENCES %s_%s_archives(hash, size),\n"
	 "    carac VARCHAR(128) "
	 "REFERENCES %s_%s_caracs(label),\n"
	 "    value VARCHAR(128),\n"
	 "    PRIMARY KEY (hash, size, carac)\n"
	 ");\n"
	 "\n"
	 "CREATE TABLE %s_%s_document_caracs (\n"
	 "    document VARCHAR(128) "
	 "REFERENCES %s_%s_documents(label),\n"
	 "    carac    VARCHAR(128) "
	 "REFERENCES %s_%s_caracs(label),\n"
	 "    value    VARCHAR(128),\n"
	 "    PRIMARY KEY (document, carac)\n"
	 ");\n"
	 "\n"
	 "CREATE TABLE %s_%s_human_caracs (\n"
	 "    name1 VARCHAR(64),\n"
	 "    name2 VARCHAR(64),\n" 
	 "    FOREIGN KEY (name1, name2) "
	 "REFERENCES %s_%s_humans(name1, name2),\n"
	 "    carac    VARCHAR(128) "
	 "REFERENCES %s_%s_caracs(label),\n"
	 "    value    VARCHAR(128),\n"
	 "    PRIMARY KEY (name1, name2, carac)\n"
	 ");\n"
	 "\n"
	 "CREATE TABLE %s_%s_category_caracs (\n"
	 "    category VARCHAR(128) "
	 "REFERENCES %s_%s_categories(label),\n"
	 "    carac    VARCHAR(128) "
	 "REFERENCES %s_%s_caracs(label),\n"
	 "    value    VARCHAR(128),\n"
	 "    PRIMARY KEY (category, carac)\n"
	 ");\n"
	 "\n"
	 "CREATE TABLE %s_%s_category_documents (\n"
	 "    category VARCHAR(128) "
	 "REFERENCES %s_%s_categories(label),\n"
	 "    document VARCHAR(128) "
	 "REFERENCES %s_%s_documents(label),\n"
	 "    PRIMARY KEY (category, document)\n"
	 ");\n"
	 "\n"
	 "CREATE TABLE %s_%s_category_childs (\n"
	 "    father VARCHAR(128) "
	 "REFERENCES %s_%s_categories(label),\n"
	 "    child  VARCHAR(128) "
	 "REFERENCES %s_%s_categories(label),\n"
	 "    PRIMARY KEY (father, child)\n"
	 ");\n"
	 "\n"
	 "CREATE TABLE %s_%s_category_humans (\n"
	 "    category VARCHAR(128) "
	 "REFERENCES %s_%s_categories(label),\n"
	 "    name1 VARCHAR(64),\n"
	 "    name2 VARCHAR(64),\n" 
	 "    FOREIGN KEY (name1, name2) "
	 "REFERENCES %s_%s_humans(name1, name2),\n"
	 "    PRIMARY KEY (category, name1, name2)\n"
	 ");\n"
	 "\n"
	 "CREATE TABLE %s_%s_document_archives (\n"
	 "    label VARCHAR(128) "
	 "REFERENCES %s_%s_documents(label),\n"
	 "    hash VARCHAR(32),\n"
	 "    size int8,\n"
	 "    FOREIGN KEY (hash, size) "
	 "REFERENCES %s_%s_archives(hash, size),\n"
	 "    PRIMARY KEY (label, hash, size)\n"
	 ");\n"
	 "CREATE TABLE %s_%s_document_role_humans (\n"
	 "    role VARCHAR(128) "
	 "REFERENCES %s_%s_roles(label),\n"
	 "    document VARCHAR(128) "
	 "REFERENCES %s_%s_documents(label),\n"
	 "    name1 VARCHAR(64),\n"
	 "    name2 VARCHAR(64),\n" 
	 "    FOREIGN KEY (name1, name2) "
	 "REFERENCES %s_%s_humans(name1, name2),\n"
	 "    PRIMARY KEY (role, document, name1, name2)\n"
	 ");\n"
	 "\n",
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label,
	 env.confLabel, coll->label, env.confLabel, coll->label);

  return;
}


/*=======================================================================
 * Function   : sqlExtractTree 
 * Description: Sql a catalog to latex working place
 * Synopsis   : int sqlExtractTree(Collection* coll)
 * Input      : Collection* coll = what to sql
 * Output     : TRUE on success
 =======================================================================*/
int 
sqlExtractTree(Collection* coll)
{ 
  int rc = FALSE;
  ExtractTree* self = NULL;
  Archive* archive = NULL;
  Container* container = NULL;
  FromAsso* fromAsso = NULL;
  AVLNode *node = NULL;
  AVLNode *node2 = NULL;
  int i = 0;
  
  checkCollection(coll);
  if (!(self = coll->extractTree)) goto error;
  logEmit(LOG_DEBUG, "sql %s extract tree", coll->label);


  // archives
  if (avl_count(coll->archives)) {
    i = 0;
    printf("INSERT INTO %s_%s_archives VALUES", 
	   env.confLabel, coll->label);
    for(node = coll->archives->head; node; node = node->next){
      archive = node->item;
   
      printf("%s\n  ", i?",":"");
      printf("  ('%s', '%lli', '%5.2f', "
	     "'~%s/cgi/get.cgi?hash=%s&size=%lli')", 
	     archive->hash, archive->size, archive->extractScore,
	     coll->user, archive->hash, archive->size);
      i = 1;
    }
    printf(";\n\n");
  }


  // containers
  if (avl_count(self->containers)) {
    i = 0;
    printf("INSERT INTO %s_%s_containers VALUES",
	   env.confLabel, coll->label);
    for(node = self->containers->head; node; node = node->next) {
      container = node->item;

      printf("%s\n  ", i?",":"");
      printf("  ('%s', '%lli', '%s')", 
	     container->parent->hash, container->parent->size, 
	     strEType(container->type));
      i = 1;
    }
    printf(";\n\n");
  }
 
  // contents
  if (avl_count(self->containers)) {
    i = 0;
    printf("INSERT INTO %s_%s_container_archives VALUES",
	   env.confLabel, coll->label);
    for(node = self->containers->head; node; node = node->next) {
      container = node->item;
      
      // loop on childs
      for(node2 = container->childs->head; node2; node2 = node2->next) {
	fromAsso = node2->item;
	
	printf("%s\n  ", i?",":"");
	printf("  ('%s', '%lli', '%s', '%lli', '%s')", 
	       container->parent->hash, container->parent->size, 
	       fromAsso->archive->hash, fromAsso->archive->size,
	       fromAsso->path);
	i = 1;
      }
    }
    printf(";\n\n");
  }

  rc = TRUE;
 error:
  if (!rc) {
    logEmit(LOG_ERR, "%s", "sqlExtractTree fails");
  }
  return rc;
}


/*=======================================================================
 * Function   : sqlServerTree 
 * Description: Sql a catalog to latex working place
 * Synopsis   : int sqlServerTree(Collection* coll)
 * Input      : Collection* coll = what to sql
 * Output     : TRUE on success
 =======================================================================*/
int 
sqlServerTree(Collection* coll)
{ 
  int rc = FALSE;
  ServerTree* self = NULL;
  Server* server = NULL;
  Image* image = NULL;
  int i = 0;
  int nb = 0;
  
  checkCollection(coll);
  if (!(self = coll->serverTree)) goto error;
  logEmit(LOG_DEBUG, "sql %s server tree", coll->label);


  // servers
  if (!isEmptyRing(self->servers)) {
    i = 0;
    printf("INSERT INTO %s_%s_servers VALUES",
	   env.confLabel, coll->label);
    rgRewind(self->servers);
    while((server = rgNext(self->servers)) != NULL) {
      printf("%s\n  ", i?",":"");
      printf("  ('%s', '%s')", 
	     server->fingerPrint, server->host);
      i = 1;
      if (!isEmptyRing(server->images)) nb = 1;
    }
    printf(";\n\n");
  }

  // images (for each server)
  if (nb) {
    i = 0;
    printf("INSERT INTO %s_%s_server_archives VALUES",
	   env.confLabel, coll->label);
    rgRewind(self->servers);
    while((server = rgNext(self->servers)) != NULL) {
      rgRewind(server->images);

      // loop on images
      rgRewind(server->images);
      while((image = rgNext(server->images)) != NULL) {
      printf("%s\n  ", i?",":"");
      printf("  ('%s', '%s', '%lli', %5.2f)", 
	     image->server->fingerPrint, 
	     image->archive->hash, image->archive->size,
	     image->score);
      i = 1;
      }
    }
    printf(";\n\n");
  }

  rc = TRUE;
 error:
  if (!rc) {
    logEmit(LOG_ERR, "%s", "sqlExtractTree fails");
  }
  return rc;
}


/*=======================================================================
 * Function   : sqlCatalogTree
 * Description: Sql a catalog to latex working place
 * Synopsis   : int sqlCatalogTree(Collection* coll)
 * Input      : Collection* coll = what to sql
 * Output     : TRUE on success
 =======================================================================*/
int
sqlCatalogTree(Collection* coll)
{
  int rc = FALSE;
  CatalogTree* self = NULL;
  AVLNode *node = NULL;
  Human* human = NULL;
  Archive* archive = NULL;
  Document* document = NULL;
  Category* category = NULL;
  Category* child = NULL;
  Role* role = NULL;
  AssoRole* assoRole = NULL;
  char url[128];
  int i;
  int nbChilds = 0, nbHums = 0, nbDocs = 0, nbArchs = 0, nbAssoRole = 0;

  checkCollection(coll);
  if (!(self = coll->catalogTree)) goto error;
  logEmit(LOG_DEBUG, "sql %s document tree", coll->label);

  // categories
  if (!isEmptyRing(self->categories)) {
    i = 0;
    printf("INSERT INTO %s_%s_categories VALUES",
	   env.confLabel, coll->label);
    rgRewind(self->categories);
    while((category = rgNext(self->categories)) != NULL) {
      if (!getCateListUri(url, "/", category->id, 1)) goto error;
      printf("%s\n  ", i?",":"");
      printf("  ('%s', '~%s/index%s')", category->label, coll->user, url);
      i = 1;
      if (!isEmptyRing(category->childs)) nbChilds = 1;
      if (!isEmptyRing(category->humans)) nbHums = 1;
      if (!isEmptyRing(category->documents)) nbDocs = 1;
    }
    printf(";\n\n");
  }

  // documents
  if (avl_count(self->documents)) {
    i = 0;
    printf("INSERT INTO %s_%s_documents VALUES",
	   env.confLabel, coll->label);
    for(node = self->documents->head; node; node = node->next) {
      document = (Document*)node->item;
      if (!getDocumentUri(url, "/", document->id)) goto error;
      printf("%s\n  ", i?",":"");
      printf("  ('%s', '~%s/index%s')", document->label, coll->user, url);
      i = 1;
      if (!isEmptyRing(document->archives)) nbArchs = 1;
    }
    printf(";\n\n");
  }

  // humans
  if (avl_count(self->humans)) {
    i = 0;
    printf("INSERT INTO %s_%s_humans VALUES",
	   env.confLabel, coll->label);
    for(node = self->humans->head; node; node = node->next) {
      human = (Human*)node->item;
      if (!getHumanUri(url, "/", human->id)) goto error;
      printf("%s\n  ", i?",":"");
      printf("  ('%s', '%s', '~%s/index%s')", 
	     human->firstName, human->secondName, coll->user, url);
      i = 1;
    }
    printf(";\n\n");
  }

  // role
  if (!isEmptyRing(self->roles)) {
    i = 0;
    printf("INSERT INTO %s_%s_roles VALUES",
	   env.confLabel, coll->label);
    rgRewind(self->roles);
    while ((role = rgNext(self->roles)) != NULL) {
      if (!getRoleListUri(url, "/", role->id, 1)) goto error;
      printf("%s\n  ", i?",":"");
      printf("  ('%s', '~%s/index%s')", role->label, coll->user, url);
      i = 1;
      if (!isEmptyRing(role->assos)) nbAssoRole = 1;
    }
    printf(";\n\n");
  }

  // category's parents
  if (nbChilds) {
    i = 0;
    printf("INSERT INTO %s_%s_category_childs VALUES",
	   env.confLabel, coll->label);
    rgRewind(self->categories);
    while ((category = rgNext(self->categories)) != NULL) {
      
      rgRewind(category->childs);
      while ((child = rgNext(category->childs)) != NULL) {
	
	printf("%s\n  ", i?",":"");
	printf("  ('%s', '%s')", 
	       category->label, child->label);
	i = 1;
      }
    }
    printf(";\n\n");
  }
    
  // category's humans
  if (nbHums) {
    i = 0;
    printf("INSERT INTO %s_%s_category_humans VALUES",
	   env.confLabel, coll->label);
    rgRewind(self->categories);
    while ((category = rgNext(self->categories)) != NULL) {
      
      rgRewind(category->humans);
      while ((human = rgNext(category->humans)) != NULL) {
	
	printf("%s\n  ", i?",":"");
	printf("  ('%s', '%s', '%s')", 
	       category->label, human->firstName, human->secondName);
	i = 1;
      }
    }
    printf(";\n\n");
  }

  // category's documents
  if (nbDocs) {
    i = 0;
    printf("INSERT INTO %s_%s_category_documents VALUES",
	   env.confLabel, coll->label);
    rgRewind(self->categories);
    while ((category = rgNext(self->categories)) != NULL) {
      
      rgRewind(category->documents);
      while ((document = rgNext(category->documents)) != NULL) {
	
	printf("%s\n  ", i?",":"");
	printf("  ('%s', '%s')", 
	       category->label, document->label);
	i = 1;
      }
    }
    printf(";\n\n");
  }

  // document's archives
  if (nbArchs) {
    i = 0;
    printf("INSERT INTO %s_%s_document_archives VALUES",
	   env.confLabel, coll->label);
    for(node = self->documents->head; node; node = node->next) {
      document = (Document*)node->item;

      rgRewind(document->archives);
      while ((archive = rgNext(document->archives)) != NULL) {

	printf("%s\n  ", i?",":"");
	printf("  ('%s', '%s', '%lli')", 
	       document->label, archive->hash, archive->size);
	i = 1;
      }
    }
    printf(";\n\n");
  }

  // role association
  if (nbAssoRole) {
    i = 0;
    printf("INSERT INTO %s_%s_document_role_humans VALUES",
	   env.confLabel, coll->label);
    rgRewind(self->roles);
    while ((role = rgNext(self->roles)) != NULL) {

      rgRewind(role->assos);
      while ((assoRole = rgNext(role->assos)) != NULL) {
	
	printf("%s\n  ", i?",":"");
	printf("  ('%s', '%s', '%s', '%s')",
	       role->label, assoRole->document->label,
	       assoRole->human->firstName, assoRole->human->secondName);
	i = 1;
      }
    }
    printf(";\n\n");
  }

  rc = TRUE;
 error:
  if (!rc) {
    logEmit(LOG_ERR, "%s", "sqlCatalogTree fails");
  }
  return rc;
}

/*=======================================================================
 * Function   : sqlCarac
 * Description: Sql all caracs from an Entity.
 * Synopsis   : int sqlCaracLoop(RG* assoCaracs)
 * Input      : RG* caracs: array where to concatenate new caracs
              : RG* assoCaracs: caracs to add 
 * Output     : RG* caracs
 *            : TRUE on success
 =======================================================================*/
int
sqlCarac(RG* caracs, RG* assoCaracs)
{
  int rc = FALSE;
  AssoCarac* assoCarac = NULL;

  rgRewind(assoCaracs);
  while((assoCarac = rgNext(assoCaracs)) != NULL) {
      
    if (!rgHaveItem(caracs, assoCarac->carac) &&
	!rgInsert(caracs, assoCarac->carac)) goto error;
  }

  rc = TRUE;
 error:
  return rc;
}

/*=======================================================================
 * Function   : sqlCaracs
 * Description: Sql all caracs.
 * Synopsis   : int sqlCarac(Collection* coll)
 * Input      : Collection* coll
 * Output     : TRUE on success
 =======================================================================*/
int
sqlCaracs(Collection* coll)
{
  int rc = FALSE;
  RG* caracs = NULL;
  Carac* carac = NULL;
  AVLNode *node = NULL;
  RG* ring = NULL;
  Category* category = NULL;
  Human* human = NULL;
  Archive* archive = NULL;
  Document* document = NULL;
  int i = 0;

  checkCollection(coll);
  logEmit(LOG_DEBUG, "%s", "sql Caracs");

  if (!(caracs = createRing())) goto error;

  // archives
  if (avl_count(coll->archives)) {
    for(node = coll->archives->head; node; node = node->next){
      archive = node->item;
      if (!sqlCarac(caracs, archive->assoCaracs)) goto error;
    }
  }

  // categories
  ring = coll->catalogTree->categories;
  rgRewind(ring);
  while((category = rgNext(ring)) != NULL) {
    if (!sqlCarac(caracs, category->assoCaracs)) goto error;
  }
  
  // humans
  if (avl_count(coll->catalogTree->humans)) {
    for(node = coll->catalogTree->humans->head; node; node = node->next) {
      human = node->item;
      if (!sqlCarac(caracs, human->assoCaracs)) goto error;
    }
  }

  // documents
  if (avl_count(coll->catalogTree->documents)) {
    for(node = coll->catalogTree->documents->head; node; node = node->next){
      document = node->item;
      if (!sqlCarac(caracs, document->assoCaracs)) goto error;
    }
  }

  // sql caracs
  if (!isEmptyRing(caracs)) {
    printf("INSERT INTO %s_%s_caracs VALUES",
	   env.confLabel, coll->label);
    rgRewind(caracs);
    while((carac = rgNext(caracs)) != NULL) {
      printf("%s\n  ('%s')", i?",":"", carac->label);
      i = 1;
    }
    printf(";\n\n");
  }

  rc = TRUE;
 error:
  if (!rc) {
    logEmit(LOG_ERR, "%s", "cannot sql all caracs");
  }
  caracs = destroyOnlyRing(caracs);
  return(rc);
}

/*=======================================================================
 * Function   : sqlAssoCaracs
 * Description: Sql all caracs.
 * Synopsis   : int sqlAssoCarac(Collection* coll)
 * Input      : Collection* coll
 * Output     : TRUE on success
 =======================================================================*/
int
sqlAssoCaracs(Collection* coll)
{
  int rc = FALSE;
  RG* caracs = NULL;
  AssoCarac* assoCarac = NULL;
  AVLNode *node = NULL;
  RG* ring = NULL;
  Category* category = NULL;
  Human* human = NULL;
  Archive* archive = NULL;
  Document* document = NULL;
  int i = 0;

  checkCollection(coll);
  logEmit(LOG_DEBUG, "%s", "sql Caracs");

  if (!(caracs = createRing())) goto error;

  // archives
  if (avl_count(coll->archives)) {
    i = 0;
    for(node = coll->archives->head; node; node = node->next) {
      archive = node->item;
   
      // loop on caracs
      rgRewind(archive->assoCaracs);
      while((assoCarac = rgNext(archive->assoCaracs)) != NULL) {

	if (i == 0) printf("INSERT INTO %s_%s_archive_caracs VALUES",
			   env.confLabel, coll->label);
	printf("%s\n  ", i?",":"");
	printf("  ('%s', '%lli', '%s', '%s')", 
	       archive->hash, archive->size,
	       assoCarac->carac->label, assoCarac->value);
	i = 1;
      }
    }
    if (i == 1) printf(";\n\n");
  }
  
  // categories
  ring = coll->catalogTree->categories;
  if (!isEmptyRing(ring)) {
    i = 0;
    rgRewind(ring);
    while((category = rgNext(ring)) != NULL) {

      // loop on caracs
      rgRewind(category->assoCaracs);
      while((assoCarac = rgNext(category->assoCaracs)) != NULL) {
	if (i == 0) printf("INSERT INTO %s_%s_category_caracs VALUES",
			   env.confLabel, coll->label);
	printf("%s\n  ", i?",":"");
	printf("  ('%s', '%s', '%s')", 
	       category->label,
	       assoCarac->carac->label, assoCarac->value);
	i = 1;
      }
    }
    if (i == 1) printf(";\n\n");
  }
  
  // humans
  if (avl_count(coll->catalogTree->humans)) {
    i = 0;
    for(node = coll->catalogTree->humans->head; node; node = node->next) {
      human = node->item;
      
      // loop on caracs
      rgRewind(human->assoCaracs);
      while((assoCarac = rgNext(human->assoCaracs)) != NULL) {
	if (i == 0) printf("INSERT INTO %s_%s_human_caracs VALUES",
			   env.confLabel, coll->label);
	printf("%s\n  ", i?",":"");
	printf("  ('%s', '%s', '%s', '%s')", 
	       human->firstName, human->secondName,
	       assoCarac->carac->label, assoCarac->value);
	i = 1;
      }
    }
    if (i == 1) printf(";\n\n");
  }

  // documents
  if (avl_count(coll->catalogTree->documents)) {
    i = 0;
    for(node=coll->catalogTree->documents->head; node; node=node->next){
      document = node->item;
      
      // loop on caracs
      rgRewind(document->assoCaracs);
      while((assoCarac = rgNext(document->assoCaracs)) != NULL) {
	if (i == 0) printf("INSERT INTO %s_%s_document_caracs VALUES",
			   env.confLabel, coll->label);
	printf("%s\n  ", i?",":"");
	printf("  ('%s', '%s', '%s')", 
	       document->label,
	       assoCarac->carac->label, assoCarac->value);
	i = 1;
      }
    }
    if (i == 1) printf(";\n\n");
  }

  rc = TRUE;
 error:
  if (!rc) {
    logEmit(LOG_ERR, "%s", "cannot sql all caracs");
  }
  caracs = destroyOnlyRing(caracs);
  return(rc);
}

/*=======================================================================
 * Function   : usage
 * Description: Print the usage.
 * Synopsis   : static void usage(char* programName)
 * Input      : programName = the name of the program; usually
 *                                  argv[0].
 * Output     : N/A
 =======================================================================*/
static void 
usage(char* programName)
{
  mdtxUsage(programName);
  fprintf(stderr, "\n\t\tlabel");
  mdtxOptions();
  fprintf(stderr, "  ---\n  label:\t\tcollection's label");
  return;
}


/*=======================================================================
 * Function   : main 
 * Author     : Nicolas ROCHE
 * modif      : 2010/12/10
 * Description: Entry point for mdtx wrapper
 * Synopsis   : ./mdtx
 * Input      : stdin
 * Output     : rtfm
 =======================================================================*/
int 
main(int argc, char** argv)
{
  //int uid = getuid();
  Collection* coll = NULL;
  // ---
  int rc = 0;
  int cOption = EOF;
  char* programName = *argv;
  char* options = MDTX_SHORT_OPTIONS;
  struct option longOptions[] = {
    MDTX_LONG_OPTIONS,
    {0, 0, 0, 0}
  };

  setlocale (LC_ALL, "");
  setlocale(LC_NUMERIC, "C"); // so as printf do not write comma in float

  // import mdtx environment
  env.allocLimit = 256;
  getEnv(&env);

  // parse the command line
  while((cOption = getopt_long(argc, argv, options, longOptions, NULL)) 
	!= EOF) {
    switch(cOption) {
      
      GET_MDTX_OPTIONS; // generic options
    }
    if (rc) goto optError;
  }

  // export mdtx environment
  env.allocDiseaseCallBack = clientDiseaseAll;
  if (!setEnv(programName, &env)) goto optError;
 
  /************************************************************************/
  if (argc == optind) {
    usage(argv[0]); // expect a collection label as first parameter
    goto optError;
  }

  logEmit(LOG_INFO, "** txt2sql: %s **", argv[optind]);

  if (!(coll = mdtxGetCollection(argv[optind]))) goto error;
  if (!loadCollection(coll, SERV|CTLG|EXTR)) goto error;
  sqlCreateTables(coll);
  if (!sqlExtractTree(coll)) goto error;
  if (!sqlServerTree(coll)) goto error;
  if (!sqlCatalogTree(coll)) goto error;
  if (!sqlCaracs(coll)) goto error;
  if (!sqlAssoCaracs(coll)) goto error;
  if (!releaseCollection(coll, SERV|CTLG|EXTR)) goto error;
  /************************************************************************/

  freeConfiguration();
  rc = TRUE;
 error:
  logEmit(LOG_INFO, "** exit on %s **", rc?"success":"error");
  DefaultLog = logClose(DefaultLog);
  rc=!rc;
 optError:
  exit(rc);
}

/* Local Variables: */
/* mode: c */
/* mode: font-lock */
/* mode: auto-fill */
/* End: */
