// A mathematical game
// Copyright (C) 2004-2005 by Christian von Schultz <schultz@linux.nu>

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "theme.h"

#include <wx/log.h>
#include <wx/image.h>

#include <config.h>
#include "checksetup.h"

#include "application.h"

Theme::Theme(const wxString& functor) throw(Exception)
{
  const wxExpr *expr = GetApp()->GetRoomList()->GetExpr(functor);
  Init(expr);
}

Theme::~Theme() throw()
{
  if(m_tlc_image != NULL)
    delete m_tlc_image;
  if(m_trc_image != NULL)
    delete m_trc_image;
  if(m_blc_image != NULL)
    delete m_blc_image;
  if(m_brc_image != NULL)
    delete m_brc_image;

  if(m_lb_image != NULL)
    delete m_lb_image;
  if(m_tb_image != NULL)
    delete m_tb_image;
  if(m_rb_image != NULL)
    delete m_rb_image;
  if(m_bb_image != NULL)
    delete m_bb_image;

  if(m_big_image != NULL)
    delete m_big_image;
  if(m_image != NULL)
    delete m_image;
}

inline void get_filename(const wxExpr *expr, const wxString &name, wxString &filename)
{
  try
  {
    GetLocalizedAttribute(expr, name, filename);
    filename = GetApp()->config->GetModuleDataFile(filename);
  }
  catch(...)
  {
    filename = wxEmptyString;
  }
}

inline wxBitmap *create_bitmap(const wxString &filename)
{
  if(filename != wxEmptyString)
  {
    wxLogTrace("image", "Creating a new bitmap from file \"%s\".",
	       filename.c_str());
    return new wxBitmap(filename, wxBITMAP_TYPE_ANY);
  }
  else
  {
    return NULL;
  }
}

void Theme::Init(const wxExpr *expr) throw(Exception)
{
  InitNumbers(expr);
  InitBorders(expr);
  InitColour(expr);

  wxString image_file;
  get_filename(expr, "image", image_file);
  m_image = create_bitmap(image_file);

  get_filename(expr, "big_image", image_file);
  m_big_image = create_bitmap(image_file);

  wxString placement = "tiled";
  expr->GetAttributeValue("placement", placement);

  m_placement = (placement == "centered") ? centered : tiled;
}

void Theme::InitNumbers(const wxExpr *expr) throw()
{
  // Some reasonable defaults:
  m_tb_height = 0;
  m_bb_height = 0;
  m_lb_width = 0;
  m_rb_width = 0;

  // Now get the real values, if any:
  expr->GetAttributeValue("TB_height", m_tb_height);
  expr->GetAttributeValue("BB_height", m_bb_height);
  expr->GetAttributeValue("LB_width", m_lb_width);
  expr->GetAttributeValue("RB_width", m_rb_width);
}


void Theme::InitBorders(const wxExpr *expr) throw()
{
  // Borders and corners
  wxString tlc_filename;   get_filename(expr, "TLC_image", tlc_filename);
  wxString trc_filename;   get_filename(expr, "TRC_image", trc_filename);
  wxString blc_filename;   get_filename(expr, "BLC_image", blc_filename);
  wxString brc_filename;   get_filename(expr, "BRC_image", brc_filename);

  wxString lb_filename;    get_filename(expr, "LB_image", lb_filename);
  wxString tb_filename;    get_filename(expr, "TB_image", tb_filename);
  wxString rb_filename;    get_filename(expr, "RB_image", rb_filename);
  wxString bb_filename;    get_filename(expr, "BB_image", bb_filename);

  if(! wxImage::FindHandler(wxBITMAP_TYPE_PNG))
    wxImage::AddHandler(new wxPNGHandler());

  if(! wxImage::FindHandler(wxBITMAP_TYPE_JPEG))
    wxImage::AddHandler(new wxJPEGHandler());

  m_tlc_image = create_bitmap(tlc_filename);
  m_trc_image = create_bitmap(trc_filename);
  m_blc_image = create_bitmap(blc_filename);
  m_brc_image = create_bitmap(brc_filename);

  m_lb_image = create_bitmap(lb_filename);
  m_tb_image = create_bitmap(tb_filename);
  m_rb_image = create_bitmap(rb_filename);
  m_bb_image = create_bitmap(bb_filename);
}

void Theme::InitColour(const wxExpr *expr) throw()
{
  m_background = GetColourAttribute(expr, "background");
  m_text_background = GetColourAttribute(expr, "text_background",
					 m_background.Red(),
					 m_background.Green(),
					 m_background.Blue());
  m_text_foreground = GetColourAttribute(expr, "text_foreground", 255, 255, 255);
}


void Theme::Draw(wxDC& dc, int width, int height) const throw()
{
  wxLogTrace("DC", "Now the DC is drawing to a window of %dx%d", width, height);

  wxBitmap *bitmap = NULL;

  if(m_big_image != NULL)
  {
    bitmap = (((width >= m_big_image->GetWidth() || height >= m_big_image->GetHeight())
	       && *m_big_image != wxNullBitmap)
	      ? m_big_image : m_image);
  }
  else
  {
    bitmap = m_image;
  }

  if(bitmap != NULL)
    if(*bitmap == wxNullBitmap)
      bitmap == NULL;

  dc.BeginDrawing();
  {
    wxLogTrace("DC", "Painting the window using the current background colour.");
    dc.SetBrush(*wxTheBrushList->FindOrCreateBrush(m_background, wxSOLID));
    dc.DrawRectangle(0, 0, width, height);
    
    if(bitmap != NULL)
      DrawMainBitmap(bitmap, dc, width, height);

    DrawBorderImages(dc, width, height);

    if(m_tlc_image != NULL)
    {
      if(*m_tlc_image != wxNullBitmap)
      {
	dc.DrawBitmap(*m_tlc_image, 0, 0);
      }
    }

    if(m_trc_image != NULL)
    {
      if(*m_trc_image != wxNullBitmap)
      {
	dc.DrawBitmap(*m_trc_image, width - m_trc_image->GetWidth(), 0);
      }
    }

    if(m_blc_image != NULL)
    {
      if(*m_blc_image != wxNullBitmap)
      {
	dc.DrawBitmap(*m_blc_image, 0, height - m_blc_image->GetHeight());
      }
    }

    if(m_brc_image != NULL)
    {
      if(*m_brc_image != wxNullBitmap)
      {
	dc.DrawBitmap(*m_brc_image,
		      width - m_brc_image->GetWidth(),
		      height - m_brc_image->GetHeight());
      }
    }
  }
  dc.EndDrawing();
  wxLogTrace("DC", "End of DC drawing operations.");

}

void Theme::DrawMainBitmap(wxBitmap *bitmap, wxDC &dc,
			   int width, int height) const throw()
{
  wxASSERT(bitmap != NULL);

  if(*bitmap != wxNullBitmap)
  {
    if(m_placement == centered)
    {
      wxLogTrace("DC", "Drawing an image, centered.");
      dc.DrawBitmap(*bitmap,
		    width/2 - bitmap->GetWidth()/2,
		    height/2 - bitmap->GetHeight()/2);
    }
    else
    {
      wxLogTrace("DC", "Drawing an image, tiled.");
	  
      for(int x = m_lb_width;
	  x <= width - m_rb_width;
	  x += bitmap->GetWidth())
      {
	for(int y = m_tb_height;
	    y <= height - m_bb_height;
	    y += bitmap->GetHeight())
	{
	  dc.DrawBitmap(*bitmap, x, y);
	}
      }
    }
  }
}

void Theme::DrawBorderImages(wxDC &dc, int width, int height) const throw()
{
  if(m_tb_image != NULL)
  {
    if(*m_tb_image != wxNullBitmap)
    {
      for(int x = m_lb_width;
	  x <= width - m_rb_width;
	  x += m_tb_image->GetWidth())
      {
	dc.DrawBitmap(*m_tb_image, x, 0);
      }
    }
  }
  if(m_bb_image != NULL)
  {
    if(*m_bb_image != wxNullBitmap)
    {
      for(int x = m_lb_width;
	  x <= width - m_rb_width;
	  x += m_bb_image->GetWidth())
      {
	dc.DrawBitmap(*m_bb_image, x, height - m_bb_image->GetHeight());
      }
    }
  }
  if(m_lb_image != NULL)
  {
    if(*m_lb_image != wxNullBitmap)
    {
      for(int y = m_tb_height;
	  y <= height - m_bb_height;
	  y += m_lb_image->GetHeight())
      {
	dc.DrawBitmap(*m_lb_image, 0, y);
      }
    }
  }
  if(m_rb_image != NULL)
  {
    if(*m_rb_image != wxNullBitmap)
    {
      for(int y = m_tb_height;
	  y <= height - m_bb_height;
	  y += m_rb_image->GetHeight())
      {
	dc.DrawBitmap(*m_rb_image, width - m_rb_image->GetWidth(), y);
      }
    }
  }
}
