// A mathematical game
// Copyright (C) 2004 by Christian von Schultz <schultz@linux.nu>

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


#include <wx/object.h>
#include <wx/wxexpr.h>
#include <wx/log.h>
#include <wx/panel.h>
#include <wx/sizer.h>
#include <wx/stattext.h>
#include <wx/textctrl.h>
#include <wx/layout.h>
#include <wx/intl.h>

#include <config.h>
#include "../checksetup.h"

#include "../windowcontents.h"
#include "../application.h"
#include "multsim.h"
#include "multsimpanel.h"
#include "selfcorr.h"

MultSimPanel::MultSimPanel(MultSim *ms, const wxExpr *expr)
                           throw(Exception) try:
  m_ms(ms),
  wxPanel(ms->main_panel, -1,
	  wxDefaultPosition, ms->main_panel->GetClientSize(),
	  wxTAB_TRAVERSAL)
{
  wxLogTrace("entering", "MultSimPanel Constructor...");

  // wxWS_EX_TRANSIENT: This can be used to prevent a window from
  // being used as an implicit parent for the dialogs which were
  // created without a parent. It is useful for the windows which can
  // disappear at any moment as creating children of such windows
  // results in fatal problems.
  SetExtraStyle(wxWS_EX_TRANSIENT);

  Start();

}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

MultSimPanel::~MultSimPanel()
{
  wxLogTrace("entering", "MultSimPanel destructor");
}


void MultSimPanel::Start() throw()
try
{
  wxLogTrace("entering", "MultSimPanel::Start()");
  wxFlexGridSizer *sizer =
    new wxFlexGridSizer(m_ms->m_data->GetRows(),
			2 * m_ms->m_data->GetColumns(),
			10, 10); // 10 is space between widgets.

  SelfCorrecting *current_sc = NULL;

  {
    wxStaticText *st;
    for(int i = 0; i < m_ms->m_data->GetRows() * m_ms->m_data->GetColumns(); i++)
    {
      st = new wxStaticText(this, -1, m_ms->m_data->GetQuestion(i),
			    wxDefaultPosition, wxDefaultSize, wxALIGN_RIGHT);
      if(m_ms->m_theme != NULL)
      {
	st->SetBackgroundColour(m_ms->m_theme->GetTextBackground());
	st->SetForegroundColour(m_ms->m_theme->GetTextForeground());
      }
      sizer->Add(st, 1, wxEXPAND | wxCENTRE);
      
      current_sc = new SelfCorrecting(m_ms, this, m_ms->m_data->GetAnswer(i));
      m_sc.push_back(current_sc);
      sizer->Add(current_sc, 0, wxEXPAND);
    }
  }

  wxBoxSizer *hsizer = new wxBoxSizer(wxHORIZONTAL);
  wxBoxSizer *vsizer = new wxBoxSizer(wxVERTICAL);
  hsizer->Add(sizer, 1, wxCENTRE, 10);
  vsizer->Add(hsizer, 1, wxCENTRE, 10);

  this->SetSizerAndFit(vsizer);

  wxBoxSizer *main_hsizer = new wxBoxSizer(wxHORIZONTAL);
  wxBoxSizer *main_vsizer = new wxBoxSizer(wxVERTICAL);
  main_hsizer->Add(this, 1, wxCENTRE | wxEXPAND, 10);
  main_vsizer->Add(main_hsizer, 1, wxCENTRE | wxEXPAND, 10);
  m_ms->main_panel->SetSizer(main_vsizer);

  m_ms->main_panel->Layout();
  
  LoadData();
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

void MultSimPanel::LoadData() throw()
try
{
  wxExpr *el = GetApp()->cGetData(m_ms->expr->Functor() + "_data");

  if(el->Type() == wxExprList)
  {
    wxExpr *item = el->GetFirst();
    list<SelfCorrecting*>::iterator iter = m_sc.begin();
    for(; item != NULL && iter != m_sc.end(); (item = item->GetNext()), ++iter)
    {
      if(item->Type() == wxExprString)
      {
	if(*iter != NULL)
	{
	  (*iter)->SetValue(item->StringValue());
	}
      }
    }
  }

  delete el;

  // If there was any data, it was probably in a file the user had
  // saved, and should not be used again during this session.
  GetApp()->ForgetData(m_ms->expr->Functor() + "_data");
}
catch(NullException &ex)
{
  wxLogDebug("There is no saved data for this object.");
}

void MultSimPanel::GiveOneFocus() throw()
{
  for(list<SelfCorrecting*>::iterator iter = m_sc.begin();
      iter != m_sc.end();
      ++iter)
  {
    if(*iter != NULL)
    {
      if((*iter)->IsEditable())
      {
	(*iter)->SetFocus();
	break;
      }
    }
  }
}

void MultSimPanel::SaveData(const wxString &title) throw()
try
{
  wxExpr *el = new wxExpr(wxExprList);

  for(list<SelfCorrecting*>::iterator iter = m_sc.begin();
      iter != m_sc.end();
      ++iter)
  {
    if(*iter != NULL)
    {
      el->Append(new wxExpr(wxExprString, (*iter)->GetValue()));
    }
    else
    {
      el->Append(new wxExpr(wxExprString, ""));
    }
  }

  // AddLabelledData() makes a local copy.
  GetApp()->AddLabelledData(title, el);

  delete el;
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

void MultSimPanel::OnPaint(wxPaintEvent &event) throw()
try
{
  wxLogTrace("DC", "Creating a new wxPaintDC: Repainting...");
  wxPaintDC dc(this);
  Draw(dc);
}
catch(Exception &ex)
{
  if(!ex.Close())
    abort();
}

void MultSimPanel::Draw(wxDC &dc) throw()
{
  int width = 0, height = 0;
  GetSize(&width, &height);

  if(m_ms->m_theme != NULL)
    m_ms->m_theme->Draw(dc, width, height);
}


BEGIN_EVENT_TABLE(MultSimPanel, wxPanel)
  EVT_PAINT(          MultSimPanel::OnPaint)
END_EVENT_TABLE()
