// A mathematical game
// Copyright (C) 2004 by Christian von Schultz <schultz@linux.nu>

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


#include <wx/object.h>
#include <wx/wxexpr.h>
#include <wx/log.h>
#include <wx/panel.h>
#include <wx/sizer.h>
#include <wx/stattext.h>
#include <wx/textctrl.h>
#include <wx/layout.h>
#include <wx/intl.h>
#include <wx/statbmp.h>
#include <wx/listctrl.h>

#include <climits>

#include <config.h>
#include "../checksetup.h"

#include "../application.h"
#include "../windowcontents.h"
#include "chooseone.h"
#include "chooseonedata.h"
#include "chooseonepanel.h"


ChooseOnePanel::ChooseOnePanel(ChooseOne *co, ChooseOneData *cod)
                           throw(Exception) try:
  m_co(co),
  m_data(cod),
  wxPanel(co->main_panel, CHOOSEONEPANEL,
	  wxDefaultPosition, co->main_panel->GetClientSize(),
	  wxTAB_TRAVERSAL)
{
  wxLogTrace("entering", "ChooseOnePanel Constructor (this=0x%lx, sizeof(*this)=%d)", 
	     (long)this, sizeof(*this));

  // wxWS_EX_TRANSIENT: This can be used to prevent a window from
  // being used as an implicit parent for the dialogs which were
  // created without a parent. It is useful for the windows which can
  // disappear at any moment as creating children of such windows
  // results in fatal problems.
  SetExtraStyle(wxWS_EX_TRANSIENT);

  Start();
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

// Don't forget to Empty() befor Start()ing, if we are already in the window.
void ChooseOnePanel::Start() throw()
try
{
  wxLogTrace("entering", "ChooseOnePanel::Start()");

  this->SetFocus();

  Screen *current = m_data->cGetScreen();
  
  wxBoxSizer *sizer = new wxBoxSizer(wxVERTICAL);
  
  // First we create a wxListCtrl with the choices, and when we know
  // how big this will be we can decide which image we want.
  
  // Preparing the wxListCtrl lc:
  wxListCtrl *lc = new wxListCtrl(this, CHOOSEONEPANEL_LISTCTRL, wxDefaultPosition,
				  wxDefaultSize, wxLC_REPORT | wxLC_NO_HEADER);
  lc->InsertColumn(0, "Answer"); // The header of the column will never be displayed.
  
  for(Screen::choice_iterator iter = current->choices.begin();
      iter != current->choices.end();
      ++iter)
  {
    lc->InsertItem(static_cast<long>(iter - current->choices.begin()), *iter);
  }
  int lcwidth, lcheight;
  lc->GetSize(&lcwidth, &lcheight);
  if(lc->GetCountPerPage() < current->choices.size())
    lcheight *= lc->GetItemCount() / 
      ((lc->GetCountPerPage() != 0) ? lc->GetCountPerPage() : 1);

  lc->SetClientSize(lc->GetColumnWidth(0), lcheight);

  while(lc->GetCountPerPage() > current->choices.size())
  {
    lcheight -= 2;
    lc->SetClientSize(lc->GetColumnWidth(0), lcheight);
  }
  lcheight += 2;
  
  // Now that we know the height of lc, we can decide which image is best;
  wxBitmap *best_bitmap = NULL;
  int mainwidth = 0, mainheight = 0;
  GetTopWindow()->GetClientSize(&mainwidth, &mainheight);
  mainheight -= lcheight;
  if(m_co->m_theme != NULL)
  {
    mainheight -= m_co->m_theme->GetTopBorder();
    mainheight -= m_co->m_theme->GetBottomBorder();
    mainwidth -= m_co->m_theme->GetLeftBorder();
    mainwidth -= m_co->m_theme->GetRightBorder();
  }
  for(Screen::image_iterator iter = current->images.begin();
      iter != current->images.end();
      ++iter)
  {
    wxLogTrace("ChooseOneBitmap", "Examining a bitmap.");
    if(best_bitmap == NULL)
    {
      wxLogTrace("ChooseOneBitmap", "  Current bitmap is NULL, accepting the new one.");
      best_bitmap = *iter;
    }
    else
    {
      if(*iter == NULL)
      {
	wxLogTrace("ChooseOneBitmap", "  New bitmap is NULL, keeping the old one.");
	continue;
      }
      
      int w1 = best_bitmap->GetWidth();
      int h1 = best_bitmap->GetHeight();
      int w2 = (*iter)->GetWidth();
      int h2 = (*iter)->GetHeight();
      
      wxLogTrace("ChooseOneBitmap", "  Current bitmap is %dx%d, new one is %dx%d, window is "
		 "%dx%d.", w1, h1, w2, h2, mainwidth, mainheight);
      
      if((mainheight < h1) && (h2 <  h1))
      {
	wxLogTrace("ChooseOneBitmap",
		   "    The current bitmap is too high. The new one is better.");
	best_bitmap = *iter;
      }
      else if((mainwidth < w1) && (w2 < w1))
      {
	wxLogTrace("ChooseOneBitmap",
		   "    The current bitmap is too wide. The new one is better.");
	best_bitmap = *iter;
      }
      else if((mainwidth < w2) || (mainheight < h2))
      {
	wxLogTrace("ChooseOneBitmap",
		   "    The new bitmap is too large.");
	continue;
      }
      else
      {
	wxLogTrace("ChooseOneBitmap",
		   "    The current bitmap and the new bitmap are both good (or both bad).");
	
	// If we arrive here, both best_bitmap and iter are
	// acceptable (either that, or both are terrible).
	if(h1 + w1 < h2 + w2)
	{
	  wxLogTrace("ChooseOneBitmap",
		     "      %d + %d < %d + %d  =>  %d < %d", h1, w1, h2, w2, h1+w1, h2+w2);
	  best_bitmap = *iter;
	}
      }
    }
  }
  
  if(best_bitmap == NULL)
    throw NoisyException(_("Error in ChooseData file. Could not find a bitmap."));
  
  
  // Now that we have a good bitmap and a good wxListCtrl, we add them to the sizer.
  wxBoxSizer *bitmap_sizer = new wxBoxSizer(wxHORIZONTAL);

  bitmap_sizer->Add(new wxStaticBitmap(this, -1, *best_bitmap),
		     0, wxCENTRE);
  sizer->Add(bitmap_sizer, 1, wxCENTRE | wxNORTH,
	     ((m_co->m_theme != NULL)? m_co->m_theme->GetTopBorder(): 0));
  

  sizer->Add(lc, 0, wxCENTRE | wxSOUTH,
	     ((m_co->m_theme != NULL)? m_co->m_theme->GetBottomBorder(): 0));
  sizer->SetItemMinSize(lc, lc->GetColumnWidth(0) + 5,
			lcheight);
  
  
  this->SetSizerAndFit(sizer);
  delete current;

  wxBoxSizer *main_hsizer = new wxBoxSizer(wxHORIZONTAL);
  wxBoxSizer *main_vsizer = new wxBoxSizer(wxVERTICAL);
  main_hsizer->Add(this, 1, wxCENTRE | wxEXPAND, 10);
  main_vsizer->Add(main_hsizer, 1, wxCENTRE | wxEXPAND, 10);
  m_co->main_panel->SetSizer(main_vsizer);

  m_co->main_panel->Layout();
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

ChooseOnePanel::~ChooseOnePanel()
{
  wxLogTrace("entering", "ChooseOnePanel destructor (this=0x%lx)",
	     (long)(this));
}

void ChooseOnePanel::OnPaint(wxPaintEvent &event) throw()
try
{
  wxLogTrace("DC", "Creating a new wxPaintDC: Repainting...");
  wxPaintDC dc(this);
  Draw(dc);
}
catch(Exception &ex)
{
  if(!ex.Close())
    abort();
}

void ChooseOnePanel::Draw(wxDC &dc) throw()
{
  int width = 0, height = 0;
  GetSize(&width, &height);

  if(m_co->m_theme != NULL)
  {
    wxLogTrace("ChooseOne", "Drawing theme");
    m_co->m_theme->Draw(dc, width, height);
  }
}

void ChooseOnePanel::OnItemActivated(wxListEvent &event) throw()
{
  wxLogTrace("ChooseOne",
	     "OnItemActivated received list event. Item number %ld (index %ld) selected.",
	     event.GetIndex() + 1, event.GetIndex());

  // GetIndex() returns a long, but there _should_ not be any problems
  // converting it to an int.
  wxASSERT(event.GetIndex() >= 0);
  wxASSERT(event.GetIndex() < INT_MAX);
  
  try
  {
    m_data->SetSelection(static_cast<int>(event.GetIndex()));
    GetTopWindow()->DoSoon(m_co->m_restarter);
  }
  catch(SilentException &ex)
  {
    // Probably means that we go through a door instead.
  }
}

void ChooseOnePanel::OnChar(wxKeyEvent &event) throw()
{
  wxLogTrace("ChooseOne", "OnChar received character.");
  wxWindow *lc = FindWindow(CHOOSEONEPANEL_LISTCTRL);
  if(lc != NULL)
    lc->SetFocus();
}

BEGIN_EVENT_TABLE(ChooseOnePanel, wxPanel)
  EVT_PAINT(          ChooseOnePanel::OnPaint)
  EVT_LIST_ITEM_ACTIVATED(CHOOSEONEPANEL_LISTCTRL, ChooseOnePanel::OnItemActivated)
  EVT_CHAR(ChooseOnePanel::OnChar)
END_EVENT_TABLE()
