// A mathematical game
// Copyright (C) 2004 by Christian von Schultz <schultz@linux.nu>

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <wx/wx.h>
#include <wx/wxexpr.h>
#include <wx/config.h>

#include "checksetup.h"

#include "exceptions.h"
#include "roomlist.h"
#include "application.h"

bool myErrorHandler(int err, char *msg)
{
  if(err == WXEXPR_ERROR_SYNTAX)
  {
    wxLogError(_("Syntax error in room list: %s"), msg);
  }
  else
  {
    wxLogError(_("Error while processing room list: %s"), msg);
  }
  return true;			// Continue anyway
}

RoomList::RoomList() throw(Exception) try:
  m_db(NULL), m_open_file(NULL)
{
  wxLogDebug("Constructing RoomList object...");

  m_db = new wxExprDatabase(myErrorHandler);

  try
  {
    wxString roomlist_file = GetApp()->config->GetRoomListFile();

    if(! m_db->Read(GetApp()->config->GetRoomListFile()))
    {
      throw NoisyException
	(wxString::Format(_("There were errors reading room list file \"%s\"."),
			  GetApp()->config->GetRoomListFile().c_str()));
    }
  }
  catch(Exception &ex)
  {
    delete m_db;
    m_db = NULL;
    throw;
  }
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

RoomList::~RoomList() throw()
{
  wxLogDebug("Destructing RoomList object...");

  if(m_db != NULL)
    delete m_db;

  if(m_open_file != NULL)
    delete m_open_file;
}

void RoomList::OpenFile(const wxString &filename) throw(Exception)
try
{
  if(m_open_file != NULL)
    delete m_open_file;

  m_open_file = new wxExprDatabase(myErrorHandler);

  if(! m_open_file->Read(filename))
    throw NoisyException(_("Could not open the file."));
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

const wxExpr *RoomList::GetRoom(const wxString &room) const throw(NullException)
{
  wxExpr *expr = NULL;
  if(m_open_file != NULL)
  {
    m_open_file->BeginFind();
    expr = m_open_file->FindClauseByFunctor(room);
    if(expr != NULL)
      return expr;
  }

  m_db->BeginFind();
  expr = m_db->FindClauseByFunctor(room);
  if(expr == NULL)
    throw NullException(wxString::Format(_("Room \"%s\" not found."),
					 room.c_str()));
  else
    return expr;
}
