function [ indexes, cardinality, glossary ] = sortable2index( sortable )
% Copyright (C) 2005,2006,2007,2008,2009 Daniele de Rigo
%
% This file is part of Mastrave.
%
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
%
% ---------------------------------------------------------------------------
%
% [ indexes, cardinality, glossary ] = sortable2index( sortable )
%
% Floating point values and cells of strings can be sorted but not used as
% indexes.
% sortable2index(...) maps sortable collections into positive integers
% preserving their order.
% In case of sparse matrices, the mapping replaces only nonzeros.
% Each index of <indexes> represents an element of <sortable> which may occur
% several times in <sortable>. To each occurrence of the same value corresponds
% the same index.
% The set of these correspondences is returned in the vector or cell of strings
% <glossary>, whose length is the <cardinality> of the indices set.
%
%
% Input arguments:
%
% <sortable>               ::sortable::
%                          cell array of strings or numerical matrix
%
%
% Example of usage:
%
%    m{1}          = { 'foobar' , 'bar' , '' , 'quux' , 'foo' , 'bar' };
%    m{2}          = { 'foobar' };
%    m{3}          =   'foobar';
%    m{4}          = ceil( randn(1,15)*5 )/2; 
%    m{4}([2 5])   = nan; 
%    m{4}([4 8])   = inf;
%    m{4}([10])    = 0;
%    nulls_id      = find( 0 == m{4} )
%    rows          = ceil(rand(1,15)*1000)
%    cols          = ceil(rand(1,15)*10)
%    vals          = m{4}; vals(nulls_id)=-1
%    m{5}          = sparse( rows, cols, vals, 1100, 20 ); m{5}(10)=0;
%    m{6}          = { };
%    m{7}          = '';
%    m{8}          = [];
%    m{9}          = sparse(    0 ,  0 );
%    m{10}         = sparse( 1100 , 15 )
%    for i=1:numel(m)
%       fprintf( 1 , '\n\nm{%d}:\n', i );
%       disp( m{i} );
%       [idx, len, g] = sortable2index( m{i} )
%    end
%
%
% version: 0.2.9

where = sprintf(  '(in function %s)'  , mfilename );

% mastrave-kernel: this function is used by check_nargin, therefore the
% test for the allowed number of input arguments is performed without
% invoking check_nargin

if(nargin~=1)
   fprintf( 2,  'Error: expected one argument instead of %d.\n'  , nargin );
   fprintf( 2,  'Usage: sortable2index( sortable )\n'  );
   error(  ' '  );
end

check_is( sortable ,  'sortable'                           ,  ...
          [  '%s the argument <sortable> must be a cell '     ...
             'array of strings or a numerical matrix'     ], where );


if issparse( sortable )
   indexes  = sortable;
else
   indexes  = zeros( size(sortable) );
end
cardinality = 0;
glossary    = repmat(sortable,0,0);

if ~numel( sortable ) return; end

if iscellstr( sortable )
   n        = sortable(:);
elseif issparse( sortable )
   [r,c,nz] = find( sortable );
   siz      = size( sortable );
   n        = nz-0;
else
   n        = sortable(:)-0;
end

if ~numel(n) return; end % skip empty sparse matrices whose size is non null

[revid, sn, sid] = score( n );

if iscellstr( sortable )
   % strings to integer indexes mapping
   first_lid     = strcmp( sn(1:end-1), sn(2:end) );
   first_lid     = [ logical(1) ; ~first_lid(:) ];
else
   first_lid     = [ 1          ;  diff(sn-0)   ] ~= 0;
end

glossary      = sn( first_lid );
in            = cumsum( first_lid );
cardinality   = in(end);    % amount of distinct elements
in            = in(revid);  % un-sort <in>

if issparse( sortable )
   indexes    = sparse( r, c,      ...
                        in,        ...
                        siz(1), siz(2) );
else
   indexes(:) = in;
end


% Local Variables:
% mode:mastrave
% End:

