function [ x_cols , y_cols ] = series_cell2mat_( x_series_cell , y_series_cell , sep )
% Copyright (C) 2006,2007,2008,2009 Daniele de Rigo
%
% This file is part of Mastrave.
%
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
%
% ---------------------------------------------------------------------------
%
% [ x_cols , y_cols ] = series_cell2mat_( x_series_cell , y_series_cell , sep )
%
%    Warning: series_cell2mat_ should not be used in end-user code because
%    it deliberately skips input and output arguments checks and other pre- and
%    post-condition testing.
%    End-user code should use instead the series_cell2mat function (without
%    the ending underscore).
%
% Rearranges a set of data series whose monodimensional domain (independent
% variable) values are collected in <x_series_cell> and whose <n>-dimensional
% codomain (<n> dependent variables) values are collected in <y_series_cell>.
% This utility may help to visualize complex data using the standard 2-D plot
% facilities.  See the example for an intuitive introduction.
% The codomain set of data <y_series_cell> must be passed as a cell array of
% <n_param> matrices each of them representing an <n>-dimensional data series.
% All data series must have the same number of <n_elem> vectors of data (i.e
% each matrix must be composed by <n_elem> rows, each row being a row-vector
% of <n> columns; each column being a dimension of the codomain).
% The domain set of data <x_series_cell> can be passed as a cell array of
% <n_param> matrices each of them representing the domain values associated to
% the corresponding <n>-dimensional data series.
% Alternatively, <x_series_cell> can be passed as a cell array of <n_param>
% column vectors, so implying that each dimension of the data series shares a
% common codomain set of values.  Cell-arrays having only one element (a matrix
% of <n_elem> rows and <n> columns, or a column vector of <n_elem> rows) can be
% passed too.  Values will be replicated to fit the size of <y_series_cell>.
% If the domain or codomain data set is passed as matrix, it is regarded as
% the unique element of a cell-array.
% In case of multi-array elements, all dimensions beyond the second one are
% splitted in bidimensional slices and regarded as data matrices, each of them
% being an element of the cell-array.
%
% <x_series_cell> and <y_series_cell> are rearranged to become matrices of
% <n_elem> * <n_param>  rows and  <n>  columns.  If the optional input argument
% <sep> is passed and its value is the string 'true', then each matrix element
% of the data series will be terminated with a trailing row of nan values.  In
% this case the number of output matrices rows will be
%    <n_elem> * ( <n_param> + 1 ).
%
%
%
% Input arguments:
%
% <x_series_cell>    ::cellnumeric::
%                    matrix or cell array of matrices all having the same size.
%                    It collects the values of the mono-dimensional independent
%                    variable.
%
% <y_series_cell>    ::cellnumeric::
%                    matrix or cell array of matrices all having the same size.
%                    It collects the values of the <n>-dimensional dependent
%                    variables.
%
% <sep>              ::string::
%                    string to enable/disable the separation of each matrix
%                    with a row of nan values.  Available categories are:
%                    <sep> value |      behaviour
%                    ------------+---------------------------------------------
%                      'true'    | Add to each matrix element of the data
%                                | series a trailing row of nan values
%                    ------------+---------------------------------------------
%                      'false'   | Don't add any trailing row
%
%
%
% Example of usage:
%
%     n1 = 4;       n2 = 3;
%     p1 = [1:n1]'; p2 = [1:n2]'.^2;
%
%     sc = mat2cell( reshape( p1*p2' , [] , 1 )*[1 10] , ones(n2,1)*n1, 2 )
%
%     [ x_cols , y_cols ] = series_cell2mat_( p1 , sc , 'true' )
%     figure(1); plot( x_cols , y_cols )
%
%     ma = zeros(5,2,3); ma(:) = 1:numel(ma)
%     [ x_cols , y_cols ] = series_cell2mat_( [1:5;7:11]' , {ma} , 'true' )
%     figure(2); plot( x_cols , y_cols )
%
%     [ x_cols , y_cols ] = series_cell2mat_( [1:5;7:11]' , {ma} , 'false' )
%     figure(3); plot( x_cols , y_cols )
%
%
%
% version: 0.3.2


switch sep
case 'true'
   sep_value = nan;
% expected case 'false'
otherwise
   sep_value = [];
end

% if passed a matrix instead of a cell-array, envelop it in a cell-array
if isnumeric( y_series_cell )
   y_series_cell = { y_series_cell };
end
n_param                = numel( y_series_cell );
if ~n_param
   [ x_cols , y_cols ] = deal( [] );
   return
end
siz_   = size( y_series_cell{1} );
n_elem  = siz_(1);
n_dims  = siz_(2);
% in case of multi-array, all dimensions beyond the second one will be splitted
n_param = prod( [n_param siz_(3:end)] );

y_series_cell = mat2cell(                  ...
   multi2mat( [ y_series_cell{:} ] , 1 ) , ...
   n_elem                                , ...
   ones(1,n_param)*n_dims                  ...
);

y_cols = series_cell_compaction(                     ...
   y_series_cell, sep_value, n_elem, n_param, n_dims ...
);

% if passed a matrix instead of a cell-array, envelop it in a cell-array
if isnumeric( x_series_cell )
   x_series_cell = { x_series_cell };
end
x_siz_  = size( x_series_cell{1} );
x_elem  = x_siz_(1);
x_dims  = x_siz_(2);
x_param = prod( [numel( x_series_cell ) x_siz_(3:end)] );

x_cols  = series_cell_compaction(                    ...
   x_series_cell, sep_value, x_elem, x_param, x_dims ...
);

if ~prod( siz_ ) || ~prod( x_siz_ )
   [ x_cols , y_cols ] = deal( [] );
   return
end

%---preconditions
% x_dims    expected to be 1 or <n_dims>
% x_param   expected to be 1 or <n_param>
x_cols = repmat( x_cols , n_param / x_param , n_dims / x_dims );



function cols = series_cell_compaction( series_cell, sep_value, n_elem, n_param, n_dims )

   cols = cell2mat(                                  ...
      mat2cell(                                      ...
         [                                           ...
            series_cell{:} ;                         ...
            repmat( sep_value , 1 , n_dims*n_param ) ...
         ]'                                        , ...
         ones( 1 , n_param ) * n_dims              , ...
         n_elem + numel( sep_value )                 ...
      )'                                             ...
   )';


% Local Variables:
% mode:mastrave
% End:

