function [ screeded_vals , valley_set ] = screed( vals, threshold, method )
% Copyright (C) 2007,2008,2009 Daniele de Rigo 
% 
% This file is part of Mastrave.
% 
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
% 
% ---------------------------------------------------------------------------
% 
% [ screeded_vals , valley_set ] = screed( vals, threshold, method )
% 
% Given a matrix <vals> and a <threshold>, the <vals> elements whose values
% are near the minimum are recognized as being part of the <valley_set>.
% <valley_set> members are detected according to one of the allowed
% <method>s which use <threshold> as a sensitivity criterion to discern
% the degree of proximity to the minimum.
% <valley_set> elements are set to the minimum value.  Values around the
% <valley_set> are scaled according to <method> in order to guarantee
% the continuity (or even the derivativeness, depending on the chosen
% <method>) with the <valley_set> plateau and the remaining values (which
% remain untouched).
% The screeded matrix of <vals> is returned as <screeded_vals>.
%
%
% Input arguments:
%
% <vals>           ::numeric::
%                  scalar, vector or matrix of numeric values
%
% <threshold>      ::nonnegative::
%                  non negative scalar or 2-dimensional vector
%
% <method>         ::cellstring::
%                  method to be used to locate the <valley_set> and to connect
%                  the valley set with the subset Uvals of untouched <vals>.
%                  Follows here a list of the implemented <method>s
%                  (it can be passed in <method> only one method for each set):
%
%           normalization methods | meaning
%        -------------------------+-------------------------------------------
%                   'lin'         | linear mapping where
%                                 |    min(vals(:)) -> 0
%                                 |    max(vals(:)) -> 1
%        -------------------------+-------------------------------------------
%                   'lin-m'       | linear mapping where
%                                 |    min(vals(:)) -> 0
%                                 |    max(vals)    -> 1 (one for each column)
%        -------------------------+-------------------------------------------
%                   'log'         | logarithmic mapping where
%                                 |    min(vals(:)) -> 0
%                                 |    max(vals(:)) -> 1
%        -------------------------+-------------------------------------------
%                   'log-m'       | logarithmic mapping where
%                                 |    min(vals(:)) -> 0
%                                 |    max(vals)    -> 1 (one for each column)
%        -------------------------+-------------------------------------------
%
%           connection methods    | meaning
%        -------------------------+-------------------------------------------
%                   'slide'       | connects <valley_set> with Uvals using a
%                                 | curve having the 1st derivative -> inf
%                                 | near <valley_set> and the same as that of
%                                 | Uvals when the curve is near Uvals
%        -------------------------+-------------------------------------------
%                   'spline1'     | connects <valley_set> with Uvals using a
%                                 | curve that linearly stretches the subset
%                                 | of <vals> between <valley_set> and Uvals
%        -------------------------+-------------------------------------------
%                   'spline3'     | connects <valley_set> with Uvals using a
%                                 | C1 differentiability class curve, i.e. a
%                                 | cubic stretching of the subset of <vals>
%                                 | between <valley_set> and Uvals
%        -------------------------+-------------------------------------------
%                   'spline5'     | connects <valley_set> with Uvals using a
%                                 | C2 differentiability class curve, i.e. a
%                                 | quintic stretching of the subset of <vals>
%                                 | between <valley_set> and Uvals
%        -------------------------+-------------------------------------------
%                   'spline7'     | connects <valley_set> with Uvals using a
%                                 | C3 differentiability class curve, i.e. a
%                                 | septic stretching of the subset of <vals>
%                                 | between <valley_set> and Uvals
%        -------------------------+-------------------------------------------
%                   'spline9'     | connects <valley_set> with Uvals using a
%                                 | C4 differentiability class curve, i.e. a
%                                 | nonic stretching of the subset of <vals>
%                                 | between <valley_set> and Uvals
%        -------------------------+-------------------------------------------
%
%
% Example of usage:
%
%    dx = 0.01;
%    x  = 0:dx:3*pi; y = sin(x);
%    y1 = screed( y , [.2 .4] );
%    figure(1); plot(  x , [y;y1]' )
%    title( 'function and screed function' )
%    xd = mean( [ x(1:end-1) ; x(2:end) ] );
%    figure(2); plot( xd , diff([y;y1]')./dx )
%    title( 'derivative of function and screed function' )
%
%    method = { 'lin' , 'slide' }
%    y1 = screed( y , [.2 .4] , method );
%    figure(3); plot(  x , [y;y1]' )
%    title( 'function and screed function' )
%    xd = mean( [ x(1:end-1) ; x(2:end) ] );
%    figure(4); plot( xd , diff([y;y1]')./dx )
%    title( 'derivative of function and screed function' )
%
%    method = { 'log' , 'spline5' }
%    y1 = screed( y , [.2 .4] , method );
%    figure(5); plot(  x , [y;y1]' )
%    title( 'function and screed function' )
%    xd = mean( [ x(1:end-1) ; x(2:end) ] );
%    figure(6); plot( xd , diff([y;y1]')./dx )
%    title( 'derivative of function and screed function' )
%
%    method = { 'log' , 'spline9' }
%    y1 = screed( y , [.2 .4] , method );
%    figure(6); plot(  x , [y;y1]' )
%    title( 'function and screed function' )
%    xd = mean( [ x(1:end-1) ; x(2:end) ] );
%    figure(7); plot( xd , diff([y;y1]')./dx )
%    title( 'derivative of function and screed function' )
%
%
% version: 0.3.8

where     = sprintf(  '(in function %s)'  , mfilename );
usage_msg = sprintf(                                                      ...
   'Usage:\n   %s\n   %s\n'                                             , ...
   '[ screeded_vals , valley_set ] = screed( vals, threshold )'         , ...
   '[ screeded_vals , valley_set ] = screed( vals, threshold, method )'   ...
);

check_is(                                                        ...
   nargin >= 2                                                 , ...
   'true'                                                      , ...
   '%s not enough input arguments.\n%s'                        , ...
   where                                                       , ...
   usage_msg                                                     ...
);

check_is(                                                        ...
   nargin <= 3                                                 , ...
   'true'                                                      , ...
   '%s too many input arguments.\n%s'                          , ...
   where                                                       , ...
   usage_msg                                                     ...
);

check_is(                                                        ...
   vals                                                        , ...
   'numeric'                                                   , ...
   '%s the first argument <vals> must be numeric.\n'           , ...
   where                                                         ...
);

check_is(                                                        ...
   threshold                                                   , ...
   'nonnegative'                                               , ...
   '%s the second argument <threshold> must be nonnegative.\n' , ...
   where                                                         ...
);
check_is(                                                                ...
   numel( threshold ) > 0 && numel( threshold ) <= 2                   , ...
   'true'                                                              , ...
   '%s the second argument <threshold> must be scalar or 2d vector.\n' , ...
   where                                                                 ...
);
check_is(                                                                ...
   max( threshold(:) ) <= 1                                            , ...
   'true'                                                              , ...
   '%s the second argument <threshold> range must be [0,1].\n'         , ...
   where                                                                 ...
);

if nargin < 3
   method = { 'log' , 'spline7' };
end

check_is(                                                        ...
   method                                                      , ...
   'cellstring'                                                , ...
   [  '%s the third argument <method> must be a string or '      ...
      'a cell of strings.\n'                                ]  , ...
   where                                                         ...
);

if ischar( method )
   method = { method };
end

for i=1:numel( method )
   switch method{i}
   case { 'lin' , 'lin-m' , 'log' , 'log-m' }
      params.norm_method = method{i}
   case { 'slide' , 'spline1' , 'spline3' , 'spline5' , 'spline7' , 'spline9' }
      params.joinfun     = method{i}
   otherwise
      error( 'bad method' )
   end
end 

[ norm_vals, params ]                           = normalize( vals, params );
[ screed_id , join_id , screed_val , join_val ] = shunt( norm_vals , threshold , params );

% plot( [vals ; ones(size(vals))*screed_val ; ones(size(vals))*join_val ]' )
'.'
screeded_vals                                   = vals;
'..'
screeded_vals( screed_id )                      = params.min_vals;
'...'
screeded_vals( join_id   )                      = join_function( vals(join_id) , params , screed_val , join_val );



function [ norm_vals, params ] = normalize( vals, params )
   fprintf( 1, '--- [ norm_vals, params ] = normalize( vals, params ) ---\n' )
   min_vals           = repmat( min(vals,[],1), [size(vals,1) ones(1,ndims(vals)-1)] );
   size(min_vals)

   switch params.norm_method
   case 'lin'
      norm_vals       = vals - min_vals;
      params.maxnorm  = max( norm_vals(:) );
      norm_vals       = norm_vals ./ params.maxnorm;
   case 'lin-m'
      norm_vals       = vals - min_vals;
      params.maxnorm  = repmat( max(norm_vals,[],1), [size(vals,1) ones(1,ndims(vals)-1)] );
      norm_vals       = norm_vals ./ params.maxnorm;
   case 'log'
      norm_vals       = log( vals - min_vals + 1 );
      params.maxnorm  = max( norm_vals(:) );
      norm_vals       = norm_vals ./ params.maxnorm;
   case 'log-m'
      norm_vals       = log( vals - min_vals + 1 );
      params.maxnorm  = repmat( max(norm_vals,[],1), [size(vals,1) ones(1,ndims(vals)-1)] );
      norm_vals       = norm_vals ./ params.maxnorm;
   end
   params.min_vals    = min_vals;
   params.val_size    = size( vals );
   fprintf( 1, 'vals:            %s\n' , sprintf( '%d ' , size(vals) ) )
   fprintf( 1, 'params.min_vals: %s\n' , sprintf( '%d ' , size(params.min_vals) ) )
   fprintf( 1, 'params.maxnorm:  %s\n' , sprintf( '%d ' , size(params.maxnorm) ) )
   fprintf( 1, 'norm_vals:       %s\n' , sprintf( '%d ' , size(norm_vals) ) )



function vals = denorm( norm_vals , params )
   fprintf( 1, '--- vals = denorm( norm_vals , params ) ---\n' )
   fprintf( 1, 'norm_vals:       %s\n' , sprintf( '%d ' , size(norm_vals) ) )
   fprintf( 1, 'params.min_vals: %s\n' , sprintf( '%d ' , size(params.min_vals) ) )
   fprintf( 1, 'params.maxnorm:  %s\n' , sprintf( '%d ' , size(params.maxnorm) ) )
   size( params.min_vals( size(norm_vals,1) , : ) )
   [ size(norm_vals,1) params.val_size(2:end) ]
   min_vals = reshape(                                ...
      params.min_vals( size(norm_vals,1) , : )      , ...
      [ size(norm_vals,1) params.val_size(2:end) ]    ...
   );
   fprintf( 1, 'min_vals:        %s\n' , sprintf( '%d ' , size(min_vals) ) )
   maxnorm  = reshape(                                ...
      params.maxnorm(  size(norm_vals,1) , : )      , ...
      [ size(norm_vals,1) params.val_size(2:end) ]    ...
   );
   fprintf( 1, 'maxnorm:         %s\n' , sprintf( '%d ' , size(maxnorm) ) )
   switch params.norm_method
   case 'lin'
      vals = norm_vals .* params.maxnorm + min_vals;
   case 'lin-m'
      vals = norm_vals .* params.maxnorm + min_vals;
   case 'log'
      vals = exp( norm_vals .* params.maxnorm ) + min_vals - 1;
   case 'log-m'
      vals = exp( norm_vals .* params.maxnorm ) + min_vals - 1;
   end
   fprintf( 1, 'vals:            %s\n' , sprintf( '%d ' , size(vals) ) )



function [ screed_id , join_id , screed_val , join_val ] = shunt( norm_vals , threshold , params )
   fprintf( 1, '--- [ screed_id , join_id , screed_val , join_val ] = shunt( norm_vals , threshold , params ) ---\n' )
   screed_id  = norm_vals < threshold(1);
   fprintf( 'screed_id size: %d\n' , sum(screed_id) )
   join_id    = norm_vals < threshold(2) & ~screed_id;
   fprintf( 'join_id   size: %d\n' , sum(join_id) )
   thres_vals = denorm(                                     ...
      repmat( threshold(:) , [1 params.val_size(2:end)] ) , ...
      params                                                ...
   );
   %screed_val = thres_vals(1);
   %join_val   = thres_vals(2);
   screed_val = repmat(                                        ...
      reshape( thres_vals(1,:) , [1 params.val_size(2:end)] ), ...
      [ params.val_size(1) ones(1,numel(params.val_size)-1) ]  ...
   );
   join_val   = repmat(                                        ...
      reshape( thres_vals(2,:) , [1 params.val_size(2:end)] ), ...
      [ params.val_size(1) ones(1,numel(params.val_size)-1) ]  ...
   );



function stretched_vals = join_function( vals , params , screed_val , join_val )
   fprintf( 1, '--- stretched_vals = join_function( vals , params , screed_val , join_val ) ---\n' )
%                                                                              .
%      |         \                                   /                         .
%      |          \                                 /                          .
%      |           \                               /...................        .
%      |            '.,                           /             |     |        .
%      | join_val _____''., ____________________ /__ ______     |     |        .
%      |  |                '.                   /   |      |    |     |        .
%      |  |                 \                  /    |dY    |    |Y(i) |        .
%      |  |                  \                /     |      |Y1  |     |vals(i) .
%      |  | screed_val ______ \ ___________ .'______|_     |    |     |        .
%      |  |  |                 \          .'          |Y0  |    |     |        .
%      |  |  | minY ____________'.___,.--'____________|____|____|     |        .
%      |  |  |  |                                                     |        .
%      |__|__|__|_____________________________________________________|___     .
%                                                                              .
%      | screeded Y                         ,-'                                .
%      |                                 ,-'                                   .
%      |                              ,:'   bisector line                      .
%      |                           ,-!' |   untouched Y                        .
%      |                        ,-','   |                                      .
%      |                     ,-'  /     |                                      .
%      |                  ,-'    /      |                                      .
%      |               ,-'      /       |                                      .
%      |            ,-'|  join_function |                                      .
%      |         ,-'   |  stretched_Y   |                                      .
%      |      ,-'      |     /          |                                      .
%      |   ,-' screed  |   ,'           |                                      .
%      |__'____________|_.'_____________|__________________ Y                  .
%                     Y0                 Y1                                    .
%                                                                              .

   %    minY   = min( vals )
   %    Y      = vals       - minY
   %    Y0     = screed_val - minY
   %    Y1     = join_val   - minY
   %    dY     = Y1 - Y0
   Y0     = screed_val - params.min_vals;
   Y1     = join_val   - params.min_vals;
   dY     = Y1 - Y0;
   switch params.joinfun
   case 'slide'
      %    stretched_vals  =  f(Y) + minY = b*( Y - Y0 ).^a + minY
      % where
      %    f'(dY) = 1                ->  a*b*dY.^(a-1) = 1
      %    f'(0)  = inf
      %    f(dY)  = Y1               ->  b*dY.^a       = Y1
      %    f(0)   = 0
      % so that
      %    b = Y1 * dY.^-a
      %    1 = a*( Y1 * dY.^-a )*dY.^(a-1)
      %      = a*Y1 * dY^-1
      %    a = dY/Y1

      a      = dY/Y1
      b      = Y1 * dY.^-a
      1/(a*dY.^(a-1))

      [dY b*(dY)^a Y1]
      [dY a*b*(dY)^(a-1)]
      stretched_vals = b*( vals - params.min_vals - Y0 ).^a + params.min_vals;

   case 'spline1'
      %    stretched_vals  =  f(Y) + minY
      %                    =  a*( Y - Y0 ) +
      %                       b            + minY
      % where
      %    f(0)   = 0                ->  b    = 0
      %    f(dY)  = Y1               ->  a*dY = Y1
      % so that
      %    a = Y1/dY
      a     = Y1 ./ dY
      stretched_vals = ...
         a*( vals - params.min_vals - Y0 ) + ...
         params.min_vals;

   case 'spline3'
      %    stretched_vals  =  f(Y) + minY
      %                    =  a*( Y - Y0 ).^3 +
      %                       b*( Y - Y0 ).^2 +
      %                       c*( Y - Y0 )    +
      %                       d               + minY
      % where
      %    f(0)   = 0                ->  d = 0
      %    f'(0)  = 0                ->  c = 0
      %    f(dY)  = Y1               ->    a*dY^3 +   b*dY^2 = Y1
      %    f'(dY) = 1                ->  3*a*dY^2 + 2*b*dY   = 1
      % so that
      %    b = Y1/dY^2 - a*dY = 1/(2*dY) -3/2 * dY * a
      %    1 = 3*a*dY^2 + 2*(Y1/dY^2 - a*dY)*dY
      %      = dY^2*a*(3 -2) + 2*Y1/dY
      %    a = 1/(dY^2) -2*Y1/dY^3
      a     = 1/(dY^2) -2 * Y1 ./ dY.^3
      b     = Y1/dY^2 - a*dY
              1/(2*dY) -3/2 * dY .* a

      [dY a*(dY)^3+b*(dY)^2 Y1]
      [dY 3*a*(dY)^2+2*b*(dY)]
      stretched_vals = ...
         a*( vals - params.min_vals - Y0 ).^3 + ...
         b*( vals - params.min_vals - Y0 ).^2 + ...
         params.min_vals;

   case 'spline5'
      %    stretched_vals  =  f(Y) + minY
      %                    =  a*( Y - Y0 ).^5 +
      %                       b*( Y - Y0 ).^4 +
      %                       c*( Y - Y0 ).^3 +
      %                       d*( Y - Y0 ).^2 +
      %                       e*( Y - Y0 )    +
      %                       f               + minY
      % where
      %    f(0)   = 0                ->  f = 0
      %    f'(0)  = 0                ->  e = 0
      %    f"(0)  = 0                ->  d = 0
      %    f(dY)  = Y1               ->     a*dY^5 +    b*dY^4 +   c*dY^3 = Y1
      %    f'(dY) = 1                ->   5*a*dY^4 +  4*b*dY^3 + 3*c*dY^2 = 1
      %    f"(dY) = 0                ->  20*a*dY^3 + 12*b*dY^2 + 6*c*dY   = 0
      % so that
      %    solve {d^2*{1,1,1},d*{5,4,3},{5*4,4*3,3*2}} . {{x_1*d^3},{x_2*d^2},{x_3*d}} = {{k},{1},{0}}
      a     = -3*(  dY - 2*Y1) ./ dY.^5
      b     =    (7*dY -15*Y1) ./ dY.^4
      c     = -2*(2*dY - 5*Y1) ./ dY.^3

      [dY    a*(dY)^5+b*(dY)^4+c*(dY)^3 Y1]
      [dY  5*a*(dY)^4+4*b*(dY)^3+3*c*(dY)^2]
      [dY 20*a*(dY)^3+12*b*(dY)^2+6*c*(dY)]
      stretched_vals = ...
         a*( vals - params.min_vals - Y0 ).^5 + ...
         b*( vals - params.min_vals - Y0 ).^4 + ...
         c*( vals - params.min_vals - Y0 ).^3 + ...
         params.min_vals;

   case 'spline7'
      %    stretched_vals  =  f(Y) + minY
      %                    =  a*( Y - Y0 ).^7 +
      %                       b*( Y - Y0 ).^6 +
      %                       c*( Y - Y0 ).^5 +
      %                       d*( Y - Y0 ).^4 +
      %                       e*( Y - Y0 ).^3 +
      %                       f*( Y - Y0 ).^2 +
      %                       g*( Y - Y0 )    +
      %                       h               + minY
      % where
      %    f(0)   = 0   ->  h = 0
      %    f'(0)  = 0   ->  g = 0
      %    f"(0)  = 0   ->  f = 0
      %    f"'(0) = 0   ->  e = 0
      %    f(dY)  = Y1  ->     a*dY^7 +     b*dY^6 +    c*dY^5 +    d*dY^4 = Y1
      %    f'(dY) = 1   ->   7*a*dY^6 +   6*b*dY^5 +  5*c*dY^4 +  4*d*dY^3 = 1
      %    f"(dY) = 0   ->  42*a*dY^5 +  30*b*dY^4 + 20*c*dY^3 + 12*d*dY^2 = 0
      %    f"'(dY)= 0   -> 210*a*dY^4 + 120*b*dY^3 + 60*c*dY^2 + 24*d*dY   = 0
      % so that
      %    solve {d^3*{1,1,1,1},d^2*{7,6,5,4},d*{7*6,6*5,5*4,4*3},{7*6*5,6*5*4,5*4*3,4*3*2}} . {{x_1*d^4},{x_2*d^3},{x_3*d^2},{x_4*d}} = {{k},{1},{0},{0}}
      a     =  10*(   dY - 2*Y1) ./ dY.^7
      b     = - 2*(17*dY -35*Y1) ./ dY.^6
      c     =   3*(13*dY -28*Y1) ./ dY.^5
      d     = - 5*( 3*dY - 7*Y1) ./ dY.^4

      [dY    a*(dY)^7+b*(dY)^6+c*(dY)^5+d*(dY)^4 Y1]
      [dY  5*a*(dY)^4+4*b*(dY)^3+3*c*(dY)^2]
      [dY 20*a*(dY)^3+12*b*(dY)^2+6*c*(dY)]
      stretched_vals = ...
         a*( vals - params.min_vals - Y0 ).^7 + ...
         b*( vals - params.min_vals - Y0 ).^6 + ...
         c*( vals - params.min_vals - Y0 ).^5 + ...
         d*( vals - params.min_vals - Y0 ).^4 + ...
         params.min_vals;


   case 'spline9'
      %    stretched_vals  =  f(Y) + minY
      %                    =  a*( Y - Y0 ).^9 +
      %                       b*( Y - Y0 ).^8 +
      %                       c*( Y - Y0 ).^7 +
      %                       d*( Y - Y0 ).^6 +
      %                       e*( Y - Y0 ).^5 +
      %                       f*( Y - Y0 ).^4 +
      %                       g*( Y - Y0 ).^3 +
      %                       h*( Y - Y0 ).^2 +
      %                       i*( Y - Y0 )    +
      %                       j               + minY
      % where
      %    f(0)   = 0   ->  j = 0
      %    f'(0)  = 0   ->  i = 0
      %    f"(0)  = 0   ->  h = 0
      %    f"'(0) = 0   ->  g = 0
      %    f""(0) = 0   ->  f = 0
      %    f(dY)  = Y1  ->      a*dY^9 +      b*dY^8 +     c*dY^7 +     d*dY^6 +     e*dY^5 = Y1
      %    f'(dY) = 1   ->    9*a*dY^8 +    8*b*dY^7 +   7*c*dY^6 +   6*d*dY^5 +   5*e*dY^4 = 1
      %    f"(dY) = 0   ->   72*a*dY^7 +   56*b*dY^6 +  42*c*dY^5 +  30*d*dY^4 +  20*e*dY^3 = 0
      %    f"'(dY)= 0   ->  504*a*dY^6 +  336*b*dY^5 + 210*c*dY^4 + 120*d*dY^3 +  60*e*dY^2 = 0
      %    f""(dY)= 0   -> 3024*a*dY^5 + 1680*b*dY^4 + 840*c*dY^3 + 360*d*dY^2 + 120*e*dY   = 0
      % so that
      %    solve {d^4,d^3,d^2,d,1}*{{1,1,1,1,1},{9,8,7,6,5},{72,56,42,30,20},{504,336,210,120,60},{3024,1680,840,360,120}}.({{x_1},{x_2},{x_3},{x_4},{x_5}}*{d^5,d^4,d^3,d^2,d})={{k},{1},{0},{0},{0}}
      a     = -35*(   dY - 2*Y1) ./ dY.^9
      b     =   5*(31*dY -63*Y1) ./ dY.^8
      c     = -20*(13*dY -27*Y1) ./ dY.^7
      d     =  28*( 7*dY -15*Y1) ./ dY.^6
      e     = -14*( 4*dY - 9*Y1) ./ dY.^5

      stretched_vals = ...
         a*( vals - params.min_vals - Y0 ).^9 + ...
         b*( vals - params.min_vals - Y0 ).^8 + ...
         c*( vals - params.min_vals - Y0 ).^7 + ...
         d*( vals - params.min_vals - Y0 ).^6 + ...
         e*( vals - params.min_vals - Y0 ).^5 + ...
         params.min_vals;

   otherwise

   end



% Local Variables:
% mode:mastrave
% End:


