function answer = mstrrep( txt , search , replace , overlap )
% Copyright (C) 2005,2006,2007 Daniele de Rigo
% 
% This file is part of Mastrave.
% 
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
% 
% ---------------------------------------------------------------------------
% 
% answer = mstrrep( txt , search , replace , overlap )
% 
% Returns the string <answer> obtained modifying a given string <txt> by 
% replacing all the occurrences of the substring <search> with the 
% content of <replace>.
%
% Input arguments:
% 
% <txt>                     ::row_string::
%                           text to be transformed
%
% <search>                  ::row_string::
%                           substring to search for
%
% <replace>                 ::row_string::
%                           substring to replace with
%
% <overlap>                 ::natural::
%                           flag indicating if overlapped occurrences of the
%                           <search> string are allowed
%
% This function extends and disambiguates the strrep(.) function of the
% standard library, enabling the explicit choice between the two behaviours 
% that are currently implemented in different computing environments: 
% if the <overlap> flag is set to a true value, even overlapped occurrences 
% of the <search> string are searched for and replaced. 
% Otherwise, overlapped matches are ignored.
% The <overlap> input argument has not a default value in order to avoid 
% unexpected behaviours in some computing environments. Please prefer this
% function instead of the pre-defined strrep(.), when the portability cost is
% higher than the computational cost.
%
% 
% Example of usage:
%
%    strrep(   'abbbbba'  ,  'bb'  ,  '.'  )
%    mstrrep(  'abbbbba'  ,  'bb'  ,  '.'  , 0 )
%    mstrrep(  'abbbbba'  ,  'bb'  ,  '.'  , 1 )
%
%    strrep(   'abcbcbcba'  ,  'bcb'  ,  '.'  )
%    mstrrep(  'abcbcbcba'  ,  'bcb'  ,  '.'  , 0 )
%    mstrrep(  'abcbcbcba'  ,  'bcb'  ,  '.'  , 1 )
%
%    strrep(   'abcbcbcba'  ,  'bcb'  ,  '_.-*'  )
%    mstrrep(  'abcbcbcba'  ,  'bcb'  ,  '_.-*'  , 0 )
%    mstrrep(  'abcbcbcba'  ,  'bcb'  ,  '_.-*'  , 1 )
%
%
%
% version: 0.1.5

where = sprintf(  '(in function %s)'  , mfilename );

% mastrave-kernel: this function is used by check_nargin, therefore the
% test for the allowed number of input arguments is performed without
% invoking check_nargin

if(nargin~=4)
   fprintf( 2,  'Error: expected 4 arguments instead of %d.\n'  , nargin );
   fprintf( 2,  'Usage: mstrrep( txt , search , replace , overlap )\n'  );
   error(  ' '  );
end;

check_is( txt ,  'row_string'  ,...
          '%s the firts argument <txt> must be a row string'      , where );
check_is( search ,  'row_string'  ,...
          '%s the second argument <search> must be a row string'  , where );
check_is( replace ,  'row_string'  ,...
          '%s the third argument <replace> must be a row string'  , where );
check_is( overlap ,  'natural'  ,...
          '%s the %s argument <overlap> must be a logical value'  , ...
          where , int2ordinal(4) );
          
% finds all the starting positions of the <search> string occurrences in <txt>
if overlap
   idx = strfind( txt , search );
else
   if exist(  'OCTAVE_VERSION'  )
      idx = findstr( txt, search , 0 );
   else
      src = regexprep( search,  '([\]\[\}\{\)\(\?\*\.\\])'  ,  '\\$1'  );
      idx = regexp( txt, src );
   end
end

% indexes of all the characters of the <search> string occurrences in <txt>
[iid,ssg] = meshgrid( 1:numel(search), idx(:) );
iidx      = iid+ssg-1;

% amount of new characters to allocate for each old <txt> character 
nchars       = ones(size(txt));
nchars(iidx) = 0;
% characters to not be replaced but only to be translated
externid     = find(nchars); 
nchars(idx)  = numel(replace);
endpos       = cumsum(nchars);
answer       = ones(1,endpos(end));
answer(endpos(externid)) = 0;  % manage first the <search> -> <replace> text
answer( answer == 1 )    = repmat(replace,1,numel(idx));
answer(endpos(externid)) = txt(externid); % then insert the unreplaced text

answer = char(answer);




% Local Variables:
% mode:mastrave
% End:

