function [cell_array, start_id, end_id] = msplit( matrix, separator, option )
% Copyright (C) 2005,2006,2007 Daniele de Rigo
% 
% This file is part of Mastrave.
% 
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
% 
% ---------------------------------------------------------------------------
% 
% [cell_array, start_id, end_id] = msplit( matrix, separator, option )
%
% Splits a string or a numerical matrix into a cell array, using <separator>
% as delimiter among tokens. The string is transposed before the splitting,
% in order to preserve the coorect order for multiline strings (please take 
% into account of this convention, when passing a non-string matrix).
% <start_id> and <end_id> show the original start and end positions of each
% substring returned in <cell_array>.
% A sequence of more than one <separator>s is processed in different manners
% depending on the value of the <option> argument. 
% If <option> is omitted or has value 'single', each <separator> in the 
% sequence splits an empty token; if <option> has value 'multiple', the
% sequence of <separator>s is considered as a single separator.
%
% Input arguments:
% 
% <matrix>          ::numstring::
%                   matrix of numbers or of characters
%
% <separator>       ::numstring::
%                   element used as token-separator
%
% <option>          ::string::
%                   set the behaviour in case of sequences of <separator>s.
%                   Valid options are:
%
%                          option    |      meaning
%                       -------------+--------------------------------
%                         'single'   | each separator of the sequence
%                                    | splits an empty token (default)
%                       -------------+--------------------------------
%                         'multiple' | consider the sequence as a
%                                    | single separator
%
% 
% Example of usage:
%    t =  '[cell_array,start_id, end_id]=   msplit(  matrix,separator, option )'  ;
%    msplit( t ,  ' '  )
%    msplit( t ,  'ar'  )
%    msplit( t ,  'ara'  )
%    
%    elems = msplit( t ,  '='  )
%    id    = mfind( elems{2} ,  ' (,)'  );
%    elems{2}(id) =  ' '  ;
%    
%            msplit( elems{2} ,  ' '  )
%    token = msplit( elems{2} ,  ' '  ,  'multiple'  )
% 
%    m = [ 7 8 3 5 ; 2 4 8 3 ; 0 1 3 8 ]
%    msplit( m , 6 )
%    msplit( m , 8 )
%    msplit( m , [8 3] )
%    msplit( m , [8 3 0] )
%
%    tt =  '1234 6   01      8'
%    [c,in,fin] = msplit( tt,  ' '  )
%    for i=1:numel(c) fprintf(1,'"%s"\t"%s"\n',c{i},tt(in(i):fin(i))); end
%    [c,in,fin] = msplit( tt,  '  '  )
%    for i=1:numel(c) fprintf(1,'"%s"\t"%s"\n',c{i},tt(in(i):fin(i))); end
%    [c,in,fin] = msplit( tt,  '   '  )
%    for i=1:numel(c) fprintf(1,'"%s"\t"%s"\n',c{i},tt(in(i):fin(i))); end
%
%    [c,in,fin] = msplit( tt,  ' '  ,  'multiple'  )
%    for i=1:numel(c) fprintf(1,'"%s"\t"%s"\n',c{i},tt(in(i):fin(i))); end
%    [c,in,fin] = msplit( tt,  '  '  ,  'multiple'  )
%    for i=1:numel(c) fprintf(1,'"%s"\t"%s"\n',c{i},tt(in(i):fin(i))); end
%    [c,in,fin] = msplit( tt,  '   '  ,  'multiple'  )
%    for i=1:numel(c) fprintf(1,'"%s"\t"%s"\n',c{i},tt(in(i):fin(i))); end
% 
% version: 0.3.5

where = sprintf(  '(in function %s)'  , mfilename );

% mastrave-kernel: this function is used by check_nargin, therefore the
% test for the allowed number of input arguments is performed without
% invoking check_nargin

if(nargin<2) | (nargin>3)
   fprintf( 2,  'Error: expected 2 or 3 arguments instead of %d.\n'  , nargin );
   fprintf( 2,  'Usage: msplit( matrix, separator )\n'  );
   fprintf( 2,  '       msplit( matrix, separator, option )\n'  );
   error(  ' '  );
end;
if nargin < 3 
   option =  'single'  ;
end

check_is( matrix ,  'numstring'                      ,  ...
          [  '%s the first argument <matrix> must be '  ...
             'numeric or a string'  ], where );
check_is( separator,  'numstring'  ,...
          '%s the second argument <separator> must be scalar'  , where );
check_is( numel(separator),  'true'  ,...
          '%s the second argument <separator> must be non empty'  , where );
check_is( option,  'string'  ,...
          '%s the third argument <option> must be a string'  , where );

len        = numel(separator);

matrix     = [separator matrix(:)' separator];
[ matrix , separator , vals, c , isstring ] = compress( matrix , separator );

%matrix     = [separator matrix(:)' separator]; 
lid        = matrix == separator;
id         = find( lid );
start_id   = id(1:end-1);
end_id     = id(2:end)-2;
pos        = find(diff(id)~= 1);
if len>1
   rest       = ones(1,numel(start_id));
   % rest = length(separator)*(#chars per elem) - 1 
   %        (1 is the length of  '.'  that replaced each separator 
   %         after the compression)
   rest(pos)  = ( ( c+1 )*len - 1 );
   % rest is the cumulated sum of each shift
   rest       = cumsum([0 rest(1:end-1)]);
   start_id   = ( start_id(:) + rest(:) +1 ) / ( c+1 );
   end_id     = (   end_id(:) + rest(:)    ) / ( c+1 );
end

if strcmp( option ,  'multiple'  )
   matrix( id(find(diff(id)== 1)) ) = [];
%   pos      = find(diff(id)~= 1)
   start_id = start_id(pos);
   end_id   = end_id(  pos);
%   start_id = find(diff(lid)==-1);
%   end_id   = find(diff(lid)== 1)-1;
   id       = find( matrix == separator );
else
%   start_id = id(1:end-1);
%   end_id   = id(2:end)-2;
end
len_id     = (end_id-start_id+1)/(c+1);
spl_id     = ones(size(matrix)); spl_id(id)=0; 
spl        = matrix(logical(spl_id));

spl        = expand( spl , vals, c , isstring );

id         = find( matrix == separator );
cell_array = mat2cell( spl, 1, (diff(id)-1)/(c+1) );


function  [ m , s , vals, c , isstring ] = compress( matrix , separator )
   isstring = ischar( matrix );
   if numel(separator) > 1
      [ idx , n , vals ] = real2index( {matrix' , separator'} );
      m    = dec2base( [n;idx{1}(:)] , 36 );
      c    = size(m,2);
      m    = [ m(2:end,:) repmat(  ' '  ,numel(matrix),1) ]';
      m    = m(:)';
      s    = dec2base( [n;idx{2}(:)] , 36 );
      s    = [ s(2:end,:) repmat(  ' '  ,numel(separator),1) ]';
      s    = s(:)';
      m    = mstrrep( m , s ,  '.'  , 1 );
      s    =  '.'  ;
   else
      m    = matrix';
      s    = separator';
      vals = [];
      c    = 0;
   end
   

function matrix = expand( m , vals, c , isstring )
   if c
      m = reshape(  m(:) , c+1 , [] )';
      m = vals(base2dec( m(:,1:c) , 36 ));
   end
   if isstring
      matrix = char(m);
   else
      matrix = m;
   end
   matrix = matrix(:)';



% Local Variables:
% mode:mastrave
% End:

