function groups = mat2groups( matrix , groups_length , mode )
% Copyright (C) 2005,2006,2007,2008 Daniele de Rigo
% 
% This file is part of Mastrave.
% 
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
% 
% ---------------------------------------------------------------------------
% 
% groups = mat2groups( matrix, groups_length, mode )
% 
% Returns a sparse matrix <groups> whose nonzeros elements are the elements
% of the input argument <matrix>. <groups> has an amount of columns equal
% to the number of elements of the vector of non-negative integers
% <groups_length>. Each column of <groups> has an amount of rows equal
% to the corresponding element of <groups_length>.
% Depending on the value of <mode>, the elements of each <groups> column
% may be in the same order of the corresponding elements of <matrix> (default 
% behaviour), or may be ordered in ascending or descending order.
%
%
% Input arguments:
% 
% <matrix>           ::numeric::
%                    matrix of numbers
%
% <groups_length>    ::natural::
%                    vector containing the desired length of each group in
%                    which <matrix> has to be splitted
%
% <mode>             ::string::
%                    direction in which each column of the cell-array <groups>
%                    will be sorted. Valid modes are:
%
%                         mode    |      meaning
%                    -------------+---------------------------------------
%                      'none'     | leave the elements of <groups> in
%                      (default)  | the same order of those of <matrix>:
%                                 | nonzeros(<groups>) == <matrix>(:)
%                    -------------+---------------------------------------
%                      'ascend'   | sort in  ascending order
%                    -------------+---------------------------------------
%                      'descend'  | sort in descending order
% 
%
% Example of usage:
%
%    n_elems = [2 0 3 4 0 0 5 3 ]
%    data    = randn( 1, sum(n_elems) )
%    sp_data = mat2groups( data, n_elems );
%    full( sp_data )
%    all( nonzeros(sp_data) == data(:) )
%    sp_data = mat2groups( data, n_elems, 'descend' );
%    full( sp_data )
%    sp_data = mat2groups( data, n_elems, 'ascend' );
%    full( sp_data )
% 
% % difference between  mat2groups  and  score
%    [rev_id, sorted_data, sort_id] = score( sp_data , 'ascend' );
%    full( sorted_data )
% 
%
%
% version: 0.4.4

where     = sprintf(  '(in function %s)'  , mfilename );

% mastrave-kernel: this function is indirectly used by check_nargin, 
% therefore the test for the allowed number of input arguments is performed 
% without invoking check_nargin

usage_msg = sprintf( [ ...
            'Usage: groups = mat2groups( matrix, groups_length, mode )\n'  ...
            '       groups = mat2groups( matrix, groups_length )\n'        ...
                     ] );

if nargin < 2
   fprintf( 2,  'Error: not enough input arguments\n'  );
   fprintf( 2,  usage_msg  );
   error(  ' '  );
end
if nargin > 3
   fprintf( 2,  'Error: too many input arguments\n'  );
   fprintf( 2,  usage_msg  );
   error(  ' '  );
end
if nargin < 3
   mode =  'none'  ;
end


check_is( matrix ,  'numeric'  ,...
          '%s the first argument <matrix> must be numeric'  , where );
check_is( groups_length ,  'natural'  ,...
          [  '%s the second argument <groups_length> must be a vector '  ...
             'of non-negative integers'  ], where );
check_is( mode ,  'char'  ,...
          '%s the third argument <mode> must be a string'  , where );
check_is( any( strcmp( mode, {  'none'  ,  'ascend'  ,  'descend'  }))  ,...
          'true'                                                        ,...
          '%s passed as third argument <mode> the invalid option "%s"'  ,...
          where, mode ); 


groups_length = groups_length(:)';      % ensure row vector
n             = sum( groups_length );

check_is( n == numel(matrix) ,  'true'  ,...
          [  '%s the sum (%d) of the second argument <groups_length> '  ...
             'must be equal to the number of elements (%d) of the '     ...
             'first argument <matrix>'  ], where, n, numel(matrix) );


col_index = find( groups_length >  0 ); % work only with non-zero groups
len       = groups_length( col_index ); % length of each group
c         = zeros( 1, n );
r         = c+1;
id        = [0 cumsum(len(1:end-1))]+1; % start position of each group
c(id)     = 1;                          % next column when a new group starts
r(id)     = [0 -len(1:end-1)]+1;        % first row   when a new group starts
c         = col_index( cumsum(c) );     % insert empty groups adding empty cols
r         = cumsum(r);

values    = matrix(:)';
if any(strcmp(mode,{  'ascend'  ,  'descend'  }))
   [scr, maxlen] = real2index( values );
   if strcmp( mode ,  'ascend' )
      [ans, scr] = sort( scr + c*maxlen ,  'ascend'  );
   else % descend
      [ans, scr] = sort( scr - c*maxlen ,  'descend'  );
   end
   values = values(scr);
end
groups    = sparse( r, c, values, max(r), numel(groups_length) );




% Local Variables:
% mode:mastrave
% End:

