function [obj, answer, errmsg] = get_cell_elem( cell_array, pos, default_obj, obj_type, msg, varargin )
% Copyright (C) 2005,2006,2007,2008,2009 Daniele de Rigo
% 
% This file is part of Mastrave.
% 
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
% 
% ---------------------------------------------------------------------------
% 
% [obj, answer, errmsg] = get_cell_elem( cell_array , pos , default_obj ,
%                                        obj_type   , msg , ...         )
%
% Given a <cell_array> of arbitrary size (for example given the cell array
% varargin declared inside a function, representing an arbitrary number of
% input arguments), returns its element at position <pos> as output <obj>.
% If the <cell_array> has less than <pos> elements, then returns a default
% value ( <default_obj> ) for that element, similary assigning it to <obj>.
% Before returning to the caller, checks if <obj> belongs to the category
% <obj_type> by invoking the function
%
%     [ <answer>, <errmsg> ] = check_is( <obj>, <obj_type>, <msg>, ... )
%
% For more details on <obj_type>, <msg> and the optional arguments, see
% the function  check_is.
% If <answer>, <errmsg> are not requested as output variables, in case of check
% failure an error is thrown whose text being <errmsg>.
%
%
% Input arguments:
%
% <cell_array>              ::cell::
%                           cell array whose element type are to be checked
%
% <pos>                     ::scalar_index::
%                           position of the element which type has to
%                           be checked
%
% <default_obj>             ::generic::
%                           default value of the output argument <obj> when
%                           <cell_array> has less than <pos> elements
%
% <obj_type>                ::generic|row_string::
%                           constraint to be checked. See the help of
%                           check_is function for details.
%
% <msg>                     ::string::
%                           string that contains the message to emit in case of
%                           check-failure. It can optionally contain C-language
%                           format tags that are replaced by the values
%                           specified in subsequent additional arguments.
%                           See check_is and fprintf help for more details.
%
%
% Example of usage:
%     [obj, answer, errmsg] = get_cell_elem( { 'foo' 1234 } , 1 , 'bar' ,...
%                                            'string' , 'must be a string!' )
%     [obj, answer, errmsg] = get_cell_elem( { 'foo' 1234 } , 2 , 'bar' ,...
%                                            'string' , 'must be a string!' )
%     [obj, answer, errmsg] = get_cell_elem( { 'foo' 1234 } , 3 , 'bar' ,...
%                                            'string' , 'must be a string!' )
%
% version: 0.2.3



where = sprintf(  '(in function %s)'  , mfilename );

obj   = [];

% mastrave-kernel: this function is used by function_decompose that is used by
% check_nargin, therefore the test for the allowed number of input arguments
% is performed without invoking check_nargin

usage_msg = sprintf( [                                          ...
   'Usage: [obj, answer, errmsg] = get_cell_elem(\n   '         ...
   '       cell_array, pos, default_obj, obj_type, msg, ...\n'  ...
   '       )\n'                                                 ...
] );

if nargin<5
   fprintf( 2,  'Error: not enough input arguments\n'  );
   fprintf( 2,  usage_msg  );
   error(  ' '  );
end

[answer, errmsg] = check_is(                             ...
   cell_array                                          , ...
   'cell'                                              , ...
   '%s The argument <cell_array> must be a cell array' , ...
   where                                                 ...
);
if ~answer  error( errmsg );  end

[answer, errmsg] = check_is(                             ...
   pos                                                 , ...
   'scalar_natural_nonzero'                            , ...
   '%s The argument <pos> must be a positive integer'  , ...
   where                                                 ...
);
if ~answer  error( errmsg );  end

[answer, errmsg] = check_is(                             ...
   msg                                                 , ...
   'row_string'                                        , ...
   '%s The argument <msg> must be a one-row string'    , ...
   where                                                 ...
);
if ~answer  error( errmsg );  end

if numel(cell_array) < pos
   obj = default_obj;
else
   obj = cell_array{ pos };
end

[answer, errmsg] = check_is( obj, obj_type, msg, varargin{:} );

if ~answer && nargout<2
   error( errmsg )
end

% Local Variables:
% mode:mastrave
% End:

