function [answer, msg, must] = check_is( obj , obj_type , msg , varargin )
% Copyright (C) 2005,2006,2007,2008,2009,2010 Daniele de Rigo
%
% This file is part of Mastrave.
%
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
%
% ---------------------------------------------------------------------------
%
% [answer, msg, must] = check_is( obj , obj_type , msg , ... )
%
% Checks if the variable <obj> belongs to the category <obj_type>.
%
% The behaviour of the function depends on the number of output arguments:
%    if nargout==0   if the check ends correctly returns to the caller, else
%                    throws an error showing the message whose format is <msg>
%    if nargout==1   returns the boolean value <answer> that is true if the
%                    check ends correctly, false otherwise
%    if nargout==2   returns the boolean <answer> as previously seen, and
%                    the error message whose format is <msg> (an empty string
%                    if the check ends correctly).
%    if nargout==3   returns <answer> and <msg> as previously seen, and
%                    the struct <must> whose fields contain the constraints to
%                    be satisfied.
%
% If other arguments are passed after <msg>, they are managed like the
% optional arguments in the function "sprintf" (see the help of "sprintf"
% for more details about the format of <msg> and the optional arguments).
%
% Input arguments:
%
% <obj>                     ::generic::
%                           object on which to check the <obj_type> constraint
%
% <obj_type>                ::generic|row_string::
%                           constraint to be checked. Follows here a list of
%                           the implemented categories for <obj_type>:
%
%    <obj_type> value         |   constraint to verify
%   --------------------------+------------------------------------------------
%    <model>                  | class( <obj> ) must be the same as
%                             | class( <model> ) and if <model> is a cell-array
%                             | or a struct, each element of <model> must
%                             | recursively have the same class of the
%                             | corresponding element of <obj>.
%                             | <model> cannot be a string: if you need to
%                             | check if <obj> is a string, use instead
%                             | something like:
%                             |        check_is(  <obj>  ,  'string'  , ...
%                             |                   'must be a string!'       )
%   --------------------------+------------------------------------------------
%    'generic',               | synonym of:  logical(     1   )
%    'anything'               |
%   --------------------------+------------------------------------------------
%    'nothing'                | synonym of:  logical(     0   )
%   --------------------------+------------------------------------------------
%    'true'                   | synonym of:  logical(   <obj> )
%   --------------------------+------------------------------------------------
%    'false'                  | synonym of:  ~logical(  <obj> )
%   --------------------------+------------------------------------------------
%    'logical'                | synonym of:  islogical( <obj> )
%   --------------------------+------------------------------------------------
%    'binary'                 | synonym of:  islogical( <obj> ) |   ( ...
%                             |              isnumeric( <obj> ) &     ...
%                             |              all(       <obj>(:) ==   ...
%                             |                logical( <obj>(:)) ) )
%   --------------------------+------------------------------------------------
%    'numeric'                | synonym of:  isnumeric( <obj> ) | ...
%                             |              islogical( <obj> )
%   --------------------------+------------------------------------------------
%    'real'                   | synonym of:  isnumeric( <obj> ) | ...
%                             |              islogical( <obj> ) & ...
%                             |              ~any(imag( <obj>(:)) )
%   --------------------------+------------------------------------------------
%    'numstring'              | synonym of:  ischar(    <obj> ) | ...
%                             |              isnumeric( <obj> ) | ...
%                             |              islogical( <obj> )
%   --------------------------+------------------------------------------------
%    'realstring'             | synonym of:  ischar(    <obj> ) |   (...
%                             |              isnumeric( <obj> ) |    ...
%                             |              islogical( <obj> ) &    ...
%                             |              ~any(imag( <obj>(:)) ) )
%   --------------------------+------------------------------------------------
%    'char' , 'string'        | synonym of:  ischar(    <obj> )
%   --------------------------+------------------------------------------------
%    'row_string'             | <obj> must be a one-row string
%   --------------------------+------------------------------------------------
%    'varname'                | synonym of:  ischar(    <obj> ) & ...
%                             |              isvarname( <obj> ) & ...
%                             |              <obj>(1) ~= '_'
%   --------------------------+------------------------------------------------
%    'cell'                   | synonym of:  iscell(    <obj> )
%   --------------------------+------------------------------------------------
%    'struct'                 | synonym of:  isstruct(  <obj> )
%   --------------------------+------------------------------------------------
%    'sparse'                 | synonym of:  issparse(  <obj> )
%   --------------------------+------------------------------------------------
%    'int8'                   | synonym of:  isa( <obj> , 'int8'   )
%   --------------------------+------------------------------------------------
%    'uint8'                  | synonym of:  isa( <obj> , 'uint8'  )
%   --------------------------+------------------------------------------------
%    'int16'                  | synonym of:  isa( <obj> , 'int16'  )
%   --------------------------+------------------------------------------------
%    'uint16'                 | synonym of:  isa( <obj> , 'uint16' )
%   --------------------------+------------------------------------------------
%    'int32'                  | synonym of:  isa( <obj> , 'int32'  )
%   --------------------------+------------------------------------------------
%    'uint32'                 | synonym of:  isa( <obj> , 'uint32' )
%   --------------------------+------------------------------------------------
%    'numcellstring',         | <obj> must be 'numeric' or a 'char' or a cell
%    'sortable'               | array whose elements <obj_type> must be 'char'
%   --------------------------+------------------------------------------------
%    'cellstring'             | <obj> must be a 'char' or a cell array whose
%                             | elements <obj_type> must be 'char'
%   --------------------------+------------------------------------------------
%    'cellstring-r'           | <obj> must be a 'char' or a cell array whose
%                             | elements <obj_type> must be 'cellstring-r'
%   --------------------------+------------------------------------------------
%    'cellstring-1'           | <obj> must be a cell array of 'char'
%   --------------------------+------------------------------------------------
%    'cellstring-2'           | <obj> must be a cell array of cell arrays of
%                             | 'char'
%   --------------------------+------------------------------------------------
%    'cellnumeric'            | <obj> must be 'numeric' or a cell array whose
%                             | elements <obj_type> must be 'numeric'
%   --------------------------+------------------------------------------------
%    'cellnumeric-r'          | <obj> must be 'numeric' or a cell array whose
%                             | elements <obj_type> must be 'cellnumeric-r'
%   --------------------------+------------------------------------------------
%    'cellnumeric-1'          | <obj> must be a cell array of 'numeric'
%   --------------------------+------------------------------------------------
%    'cellnumeric-2'          | <obj> must be a cell array of cell arrays of
%                             | 'numeric'
%   --------------------------+------------------------------------------------
%    'cellnumstring'          | <obj> must be 'numstring' or a cell array whose
%                             | elements <obj_type> must be 'numstring'
%   --------------------------+------------------------------------------------
%    'cellnumstring-r'        | <obj> must be 'numstring' or a cell array whose
%                             | elements <obj_type> must be 'cellnumstring-r'
%   --------------------------+------------------------------------------------
%    'cellnumstring-1'        | <obj> must be a cell array of 'numstring'
%   --------------------------+------------------------------------------------
%    'cellnumstring-2'        | <obj> must be a cell array of cell arrays of
%                             | 'numstring'
%   --------------------------+------------------------------------------------
%    'same_rows'              | <obj> must be a cell array of objects, each of
%                             | them having the same number of rows
%   --------------------------+------------------------------------------------
%    'same_columns'           | <obj> must be a cell array of objects, each of
%                             | them having the same number of columns
%   --------------------------+------------------------------------------------
%    'same_length'            | <obj> must be a cell array of objects, each of
%                             | them having the same length
%   --------------------------+------------------------------------------------
%    'same_size'              | <obj> must be a cell array of objects, each of
%                             | them having the same size
%   --------------------------+------------------------------------------------
%    'same_ndims'             | <obj> must be a cell array of objects, each of
%                             | them having the same number of dimensions
%   --------------------------+------------------------------------------------
%    'same_numel'             | <obj> must be a cell array of objects, each of
%                             | them having the same number of elements
%   --------------------------+------------------------------------------------
%    'comparable'             | <obj> must be a cell array of objects, each of
%                             | them must be comparable with the other using
%                             | the operator ==
%   --------------------------+------------------------------------------------
%    'positive'               | <obj> must be numeric and each of its elements
%                             | must be greater than zero
%   --------------------------+------------------------------------------------
%    'nonpositive'            | <obj> must be numeric and each of its elements
%                             | must be less or equal to zero
%   --------------------------+------------------------------------------------
%    'negative'               | <obj> must be numeric and each of its elements
%                             | must be less than zero
%   --------------------------+------------------------------------------------
%    'nonnegative'            | <obj> must be numeric and each of its elements
%                             | must be greater or equal to zero
%   --------------------------+------------------------------------------------
%    'integer'                | <obj> must be numeric and each of its elements
%                             | must be convertible to an integer without loss
%                             | of information
%   --------------------------+------------------------------------------------
%    'natural',               | <obj> must be numeric and each of its elements
%    'numel'                  | must be convertible to a non-negative integer
%                             | without loss of information
%   --------------------------+------------------------------------------------
%    'natural_nonzero',       | <obj> must be numeric and each of its elements
%    'index'                  | must be convertible to a positive integer
%                             | without loss of information
%   --------------------------+------------------------------------------------
%    'empty'                  | synonim of:  length( <obj> )==0
%   --------------------------+------------------------------------------------
%    'scalar'                 | synonym of:  length( <obj> )==1
%   --------------------------+------------------------------------------------
%    'scalar_numeric'         | synonym of:  length( <obj> )==1 & ...
%                             |              isnumeric( <obj> ) | ...
%                             |              islogical( <obj> )
%   --------------------------+------------------------------------------------
%    'scalar_real'            | synonym of:  length( <obj> )==1 & ...
%                             |              isnumeric( <obj> ) | ...
%                             |              islogical( <obj> ) & ...
%                             |              ~any(imag( <obj>(:)) )
%   --------------------------+------------------------------------------------
%    'scalar_numstring'       | synonym of:  length( <obj> )==1 & ...
%                             |              ischar(    <obj> ) | ...
%                             |              isnumeric( <obj> ) | ...
%                             |              islogical( <obj> )
%   --------------------------+------------------------------------------------
%    'scalar_realstring'      | synonym of:  length( <obj> )==1 & ...
%                             |              ischar(    <obj> ) |   (...
%                             |              isnumeric( <obj> ) |    ...
%                             |              islogical( <obj> ) &    ...
%                             |              ~any(imag( <obj>(:)) ) )
%   --------------------------+------------------------------------------------
%    'scalar_positive'        | <obj> must be scalar, numeric, greater than
%                             | zero
%   --------------------------+------------------------------------------------
%    'scalar_nonpositive'     | <obj> must be scalar, numeric, less or equal to
%                             | zero
%   --------------------------+------------------------------------------------
%    'scalar_negative'        | <obj> must be scalar, numeric, less than zero
%   --------------------------+------------------------------------------------
%    'scalar_nonnegative'     | <obj> must be scalar, numeric, greater or equal
%                             | to zero
%   --------------------------+------------------------------------------------
%    'scalar_integer'         | <obj> must be scalar, numeric and convertible
%                             | to an integer without loss of information
%   --------------------------+------------------------------------------------
%    'scalar_natural',        | <obj> must be scalar, numeric and convertible
%    'scalar_numel'           | to a non-negative integer without loss of
%                             | information
%   --------------------------+------------------------------------------------
%    'scalar_natural_nonzero',| <obj> must be scalar, numeric and convertible
%    'scalar_index'           | to a positive integer without loss of
%                             | information
%   --------------------------+------------------------------------------------
%    'vector'                 | <obj> must be a row or column vector
%   --------------------------+------------------------------------------------
%    'row_vector'             | <obj> must be a row vector
%   --------------------------+------------------------------------------------
%    'col_vector'             | <obj> must be a column vector
%   --------------------------+------------------------------------------------
%    'matrix'                 | synonym of:  ndims(<obj>) <= 2
%   --------------------------+------------------------------------------------
%    '3-array'                | synonym of:  ndims(<obj>) == 3
%   --------------------------+------------------------------------------------
%
% <msg>                     ::string::
%                           string that contains the message to emit in case of
%                           check-failure. It can optionally contain C-language
%                           format tags that are replaced by the values
%                           specified in subsequent additional arguments.
%                           See fprintf help for more details.
%
%
% Examples of usage:
%
%    s={ 'cell'         , 'string'        , 'cellstring'    , ...
%        'cellstring-r' , 'cellstring-1'  , 'cellstring-2'  };
%    obj1   = {},         obj2 =   { 'a'  'bc'  },         obj3 =     'a'    ,
%    obj4   = { 'a' {} }, obj5 = { { 'a'  'bc'  }  {}  },  obj6 = {{{ 'a' }}},
%    objlist = { obj1 , obj2 , obj3 , obj4 , obj5 , obj6 };
%    for i=1:numel(s)
%       for j=1:numel(objlist)
%          obj = objlist{j};
%          [answer, msg] = check_is( obj , s{i}, 'oops..');
%          fprintf('is obj%d a %s ?: %d (%s)\n', j, s{i}, answer, msg );
%       end
%    end
%
%    s={ 'same_rows'   , 'same_columns' , 'same_numel' , ...
%        'same_length' , 'same_size'    , 'same_ndims' , 'comparable' };
%    obj1   = rand(4,4), obj2 = rand(4,4), obj3 = rand(4,2,2),
%    obj4   = rand(3,4), obj5 = rand(5,4),
%    for i=1:length(s)
%       [answer, msg] = check_is( { obj1, obj2, obj3, obj4, obj5 }, ...
%                                 s{i},  'oops..'   );
%       fprintf('%s : %d (%s)\n', s{i}, answer, msg );
%    end
%
%
% version: 0.10.8

where     = sprintf(  '(in function %s)'  , mfilename );
usage_msg = sprintf(  'Usage: check_is( obj , obj_type , msg , ... )\n'   );


if nargin<3
   fprintf( 2,  'error: not enough input arguments\n'  );
   fprintf( 2,  usage_msg  );
   error(  ' '  );
end

% if <obj_type> is not a string, then it is considered the <model>
if ~ischar(obj_type)
   model    =  obj_type ;
   obj_type =  'model'  ;
end

if ~ischar(msg)
   fprintf( 2,  'error %s: the 3rd argument <msg> must be a string\n'  , ...
      where                                                              ...
   );
   error(  ' '  );
end


err = 0;
obj_type = obj_type(:)';
switch obj_type
case  'model'
   [ answer, bad_elem ] = has_model( obj , model , inputname(1) );
   err = ~answer;
   if length(bad_elem)
      msg = sprintf(  '%s\nat position: %s'  , msg , bad_elem );
   end
case  {  'generic'  ,  'anything'  }
   must.be = 'anything (always satisfied constraint)' ;
   err = ~logical(1);
case  'nothing'
   must.be = 'nothing (impossible constraint)' ;
   err = ~logical(0);
case  'true'
   must.be = 'true' ;
   err     = ~is_numeric(obj);
   if ~err
      err  = any(~obj(:));
   end
case  'false'
   must.be = 'false' ;
   err     = ~is_numeric(obj);
   if ~err
      err  = any(obj(:));
   end
case  'logical'
   must.be = 'logical' ;
   err     = ~islogical(obj);
case  'binary'
   must.be = 'binary' ;
   err     = ~islogical(obj);
   if err
      err  = ~is_numeric(obj);
      if ~err
         err = ~all( obj(:) == logical( obj(:) ) );
      end
   end
case  'numeric'
   must.be = 'numeric' ;
   err     = ~is_numeric(obj);
case  'real'
   must.be = 'numeric (real set)' ;
   err     = ~is_numeric(obj);
   if ~err
      err = any(imag( obj(:)) );
   end
case  'numstring'
   must.be = 'numeric or a string' ;
   err     = ~( is_numeric(obj) | ischar(obj) );
case  'realstring'
   must.be = 'numeric (real set) or a string' ;
   err     = ~( is_numeric(obj) | ischar(obj) );
   if ~err &is_numeric(obj)
      err  = any(imag( obj(:)) );
   end
case  {  'char'  ,  'string'  }
   must.be = 'a string' ;
   err     = ~ischar(obj);
case  'row_string'
   must.be = 'a one-row string' ;
   err     = ~ischar(obj) | ndims(obj)>2 | size(obj,1)>1;
case  'varname'
   must.be = 'a valid name of variable' ;
   [err, msg] = is_err( ~ischar(obj) , msg ,  'required a string'  );
   if ~err
      [err, msg] = is_err( ndims(obj)>2 | size(obj,1)>1 , msg , ...
         'required a single-row string'  );
   end
   if ~err
      [err, msg] = is_err( ~isvarname(obj) , msg , ...
         'invalid name (does not match the regexp [A-Za-z][A-Za-z0-9_]*)'  );
   end
   if ~err
      [err, msg] = is_err( obj(1) == '_' , msg , ...
         'invalid name (does not match the regexp [A-Za-z][A-Za-z0-9_]*)'  );
   end
case  'sparse'
   must.be = 'a sparse matrix' ;
   err     = ~issparse(obj);
case  {  'cell'   ,  'struct'  ,...
         'int8'   ,  'uint8'   ,...
         'int16'  ,  'uint16'  ,...
         'int32'  ,  'uint32'  }
   must.be = sprintf( 'of type %%s' , obj_type );
   err     = ~isa( obj , obj_type );
case  {  'numcellstring'  ,  'sortable'  }
   must.be = 'sortable' ;
   err     = ~is_numeric(obj);
   if err
      % old version:
      % [ answer, bad_elem ] = is_cell_of( obj ,  'char'    , inputname(1) );
      [ answer, bad_elem ] = is_recursive_cell_of( ...
         obj ,  'char'    , inputname(1) , 0 , 1   ...
      );
      err = ~answer;
      if length(bad_elem)
         msg = sprintf(  '%s\nat position: %s'  , msg , bad_elem );
      end
   end
case  'cellstring'
   % old version:
   % [ answer, bad_elem ] = is_cell_of( obj ,  'char'    , inputname(1) );
   must.be = 'a string or a cell-array of strings' ;
   [ answer, bad_elem ] = is_recursive_cell_of(   ...
      obj ,  'char'      , inputname(1) , 0 , 1   ...
   );
   err = ~answer;
   if length(bad_elem)
      msg = sprintf(  '%s\nat position: %s'  , msg , bad_elem );
   end
case  'cellnumstring'
   % old version:
   % [ answer, bad_elem ] = is_cell_of( obj ,  'numstring'  , inputname(1) );
   [foo,basetype] = check_is( struct , 'numstring' , '' );
   must.be = sprintf(                                  ...
      '%s (numstring) or a cell-array of numstrings' , ...
      basetype                                         ...
   );
   [ answer, bad_elem ] = is_recursive_cell_of(   ...
      obj ,  'numstring'  , inputname(1) , 0 , 1  ...
   );
   err = ~answer;
   if length(bad_elem)
      msg = sprintf(  '%s\nat position: %s'  , msg , bad_elem );
   end
case  'cellnumeric'
   % old version:
   % [ answer, bad_elem ] = is_cell_of( obj ,  'numeric'  , inputname(1) );
   must.be = 'numeric or a cell-array of numerics' ;
   [ answer, bad_elem ] = is_recursive_cell_of(   ...
      obj ,  'numeric'    , inputname(1) , 0 , 1  ...
   );
   err = ~answer;
   if length(bad_elem)
      msg = sprintf(  '%s\nat position: %s'  , msg , bad_elem );
   end
case  'cellstring-r'
   must.be = 'a string or a recursive cell-array whose leaves are strings' ;
   [ answer, bad_elem ] = is_recursive_cell_of( ...
      obj ,  'char'      , inputname(1)         ...
   );
   err = ~answer;
   if length(bad_elem)
      msg = sprintf(  '%s\nat position: %s'  , msg , bad_elem );
   end
case  'cellnumstring-r'
   [foo,basetype] = check_is( struct , 'numstring' , '' );
   must.be        = sprintf(                                      ...
      [ '%s (numstring) or a recursive cell-array whose leaves '  ...
        'are numstrings'                                       ], ...
      basetype                                                    ...
   ) ;
   [ answer, bad_elem ] = is_recursive_cell_of( ...
      obj ,  'numstring'  , inputname(1)        ...
   );
   err = ~answer;
   if length(bad_elem)
      msg = sprintf(  '%s\nat position: %s'  , msg , bad_elem );
   end
case  'cellnumeric-r'
   must.be = 'numeric or a recursive cell-array whose leaves are numerics' ;
   [ answer, bad_elem ] = is_recursive_cell_of( ...
      obj ,  'numeric'    , inputname(1)        ...
   );
   err = ~answer;
   if length(bad_elem)
      msg = sprintf(  '%s\nat position: %s'  , msg , bad_elem );
   end
case  'cellstring-1'
   must.be = 'a cell-array of strings' ;
   [ answer, bad_elem ] = is_recursive_cell_of(   ...
      obj ,  'char'      , inputname(1) , 1 , 1   ...
   );
   err = ~answer;
   if length(bad_elem)
      msg = sprintf(  '%s\nat position: %s'  , msg , bad_elem );
   end
case  'cellnumstring-1'
   [foo,basetype] = check_is( struct , 'numstring' , '' );
   must.be        = sprintf(                     ...
      'a cell-array of elements each being %s' , ...
      basetype                                   ...
   );
   [ answer, bad_elem ] = is_recursive_cell_of(   ...
      obj ,  'numstring'  , inputname(1) , 1 , 1  ...
   );
   err = ~answer;
   if length(bad_elem)
      msg = sprintf(  '%s\nat position: %s'  , msg , bad_elem );
   end
case  'cellnumeric-1'
   must.be = 'a cell-array of numerics' ;
   [ answer, bad_elem ] = is_recursive_cell_of(   ...
      obj ,  'numeric'    , inputname(1) , 1 , 1  ...
   );
   err = ~answer;
   if length(bad_elem)
      msg = sprintf(  '%s\nat position: %s'  , msg , bad_elem );
   end
case  'cellstring-2'
   must.be = 'a cell-array of cell-arrays of strings' ;
   [ answer, bad_elem ] = is_recursive_cell_of(   ...
      obj ,  'char'      , inputname(1) , 2 , 2   ...
   );
   err = ~answer;
   if length(bad_elem)
      msg = sprintf(  '%s\nat position: %s'  , msg , bad_elem );
   end
case  'cellnumstring-2'
   [foo,basetype] = check_is( struct , 'numstring' , '' );
   must.be        = sprintf(                                    ...
      'a cell-array of cell-arrays of elements each being %s' , ...
      basetype                                                  ...
   );
   [ answer, bad_elem ] = is_recursive_cell_of(   ...
      obj ,  'numstring'  , inputname(1) , 2 , 2  ...
   );
   err = ~answer;
   if length(bad_elem)
      msg = sprintf(  '%s\nat position: %s'  , msg , bad_elem );
   end
case  'cellnumeric-2'
   must.be = 'a cell-array of cell-arrays of numerics' ;
   [ answer, bad_elem ] = is_recursive_cell_of(   ...
      obj ,  'numeric'    , inputname(1) , 2 , 2  ...
   );
   err = ~answer;
   if length(bad_elem)
      msg = sprintf(  '%s\nat position: %s'  , msg , bad_elem );
   end
case  'same_columns'
   must.have = 'the same number of columns' ;
   [ answer, bad_elem ] = match_same( obj ,  'columns'  );
   err = ~answer;
   if length(bad_elem)
      msg = sprintf(  '%s\n%s'  , msg , bad_elem );
   end
case  'same_rows'
   must.have = 'the same number of rows' ;
   [ answer, bad_elem ] = match_same( obj ,  'rows'  );
   err = ~answer;
   if length(bad_elem)
      msg = sprintf(  '%s\n%s'  , msg , bad_elem );
   end
case  'same_length'
   must.have = 'the same length' ;
   [ answer, bad_elem ] = match_same( obj ,  'length'  );
   err = ~answer;
   if length(bad_elem)
      msg = sprintf(  '%s\n%s'  , msg , bad_elem );
   end
case  'same_size'
   must.have = 'the same size' ;
   [ answer, bad_elem ] = match_same( obj ,  'size'  );
   err = ~answer;
   if length(bad_elem)
      msg = sprintf(  '%s\n%s'  , msg , bad_elem );
   end
case  'same_ndims'
   must.have = 'the same number of dimensions (ndims)' ;
   [ answer, bad_elem ] = match_same( obj ,  'ndims'  );
   err = ~answer;
   if length(bad_elem)
      msg = sprintf(  '%s\n%s'  , msg , bad_elem );
   end
case  'same_numel'
   must.have = 'the same number of elements' ;
   [ answer, bad_elem ] = match_same( obj ,  'numel'  );
   err = ~answer;
   if length(bad_elem)
      msg = sprintf(  '%s\n%s'  , msg , bad_elem );
   end
case  'comparable'
   must.be = 'comparable' ;
   [ answer, bad_elem ] = match_same( obj ,  'operator=='  );
   err = ~answer;
   if length(bad_elem)
      msg = sprintf(  '%s\n%s'  , msg , bad_elem );
   end
case  'positive'
   must.be = 'numeric (positive set)' ;
   err = ~is_numeric(obj);
   if ~err
      err = any(obj(:)<=0);
   end
case  'nonpositive'
   must.be = 'numeric (non positive set)' ;
   err = ~is_numeric(obj);
   if ~err
      err = any(obj(:)>0);
   end
case  'negative'
   must.be = 'numeric (negative set)' ;
   err = ~is_numeric(obj);
   if ~err
      err = any(obj(:)>=0);
   end
case  'nonnegative'
   must.be = 'numeric (non negative set)' ;
   err = ~is_numeric(obj);
   if ~err
      err = any(obj(:)<0);
   end
case  'integer'
   must.be = 'numeric (integer set)' ;
   err = ~is_numeric(obj);
   if ~err
      err = any(obj(:)~=floor(obj(:)));
   end
case  {  'natural'  ,  'numel'  }
   must.be = 'numeric (natural set)' ;
   err = ~is_numeric(obj);
   if ~err
      err = any( obj(:)~=floor(obj(:)) | any(obj(:)<0) );
   end
case  {  'natural_nonzero'  ,  'index'  }
   must.be = 'numeric (natural set except zero)' ;
   err = ~is_numeric(obj);
   if ~err
      err = any( obj(:)~=floor(obj(:)) | any(obj(:)<=0) );
   end
case  'empty'
   must.be = 'empty' ;
   err = length(obj)~=0;
case  'scalar'
   must.be = 'scalar' ;
   err = length(obj)~=1;
case  'scalar_numeric'
   must.be = 'a numeric scalar' ;
   err = length(obj)~=1 | ~is_numeric(obj);
case  'scalar_real'
   must.be = 'a real scalar number' ;
   err = length(obj)~=1 | ~is_numeric(obj);
   if ~err
      err = any(imag( obj(:)) );
   end
case  'scalar_numstring'
   must.be = 'sigle char or a numeric scalar' ;
   err = length(obj)~=1 | ~( is_numeric(obj) | ischar(obj) );
case  'scalar_realstring'
   must.be = 'sigle char or a real scalar number' ;
   err = length(obj)~=1 | ~( is_numeric(obj) | ischar(obj) );
   if ~err &is_numeric(obj)
      err = any(imag( obj(:)) );
   end
case  'scalar_positive'
   must.be = 'a positive scalar' ;
   err = length(obj)~=1 | ~is_numeric(obj);
   if ~err
      err = obj<=0;
   end
case  'scalar_nonpositive'
   must.be = 'a non positive scalar' ;
   err = length(obj)~=1 | ~is_numeric(obj);
   if ~err
      err = obj>0;
   end
case  'scalar_negative'
   must.be = 'a negative scalar' ;
   err = length(obj)~=1 | ~is_numeric(obj);
   if ~err
      err = obj>=0;
   end
case  'scalar_nonnegative'
   must.be = 'a non negative scalar' ;
   err = length(obj)~=1 | ~is_numeric(obj);
   if ~err
      err = obj<0;
   end
case  'scalar_integer'
   must.be = 'a scalar integer' ;
   err = length(obj)~=1 | ~is_numeric(obj);
   if ~err
      err = obj~=floor(obj);
   end
case  {  'scalar_natural'  ,  'scalar_numel'  }
   must.be = 'a scalar natural' ;
   err = length(obj)~=1 | ~is_numeric(obj);
   if ~err
      err = obj~=floor(obj) | obj<0;
   end
case  {  'scalar_natural_nonzero'  ,  'scalar_index'  }
   must.be = 'a scalar natural except zero' ;
   err = length(obj)~=1 | ~is_numeric(obj);
   if ~err
      err = obj~=floor(obj) | obj<=0;
   end
case  'vector'
   must.be = 'a vector' ;
   err = ndims(obj)>2 | min(size(obj))>1;
case  'row_vector'
   must.be = 'a row vector' ;
   err = ndims(obj)>2 | size(obj,1)>1;
case  'col_vector'
   must.be = 'a column vector' ;
   err = ndims(obj)>2 | size(obj,2)>1;
case  'matrix'
   must.be = 'a matrix' ;
   err = ndims(obj)>2;
case  '3-array'
   must.be = 'a 3 dimensional array' ;
   err = ndims(obj)~=3;
otherwise
   fprintf( 2,  'error %s: check for type <%s> not implemented\n'  , ...
      where, obj_type                                                ...
   );
   error(  ' '  );
end


answer    = ~err;

if err
   msg    = txt_justify( sprintf( msg, varargin{:} ), 80 ,  'left'  );
   if ~nargout
      throw_error(msg);
   else
      msg = char(msg);
   end
else
   msg    = [];
end



function [ answer, msg ] = has_model( obj , model , obj_name )
   msg    =  ''  ;
   answer = strcmp( class(obj) , class(model) );
   if answer
      switch class(model)
      case  'struct'
         %    [sorted, idx] = sort( my_string_cell ) is not portable.
         % Use instead:
         %    [sorted, idx] = sortrows( char( my_string_cell ) );
         %     sorted       = cellstr( sorted );
         [obj_fn, obj_id] = sortrows( char(fieldnames(  obj  )) );
         [mod_fn, mod_id] = sortrows( char(fieldnames( model )) );
         obj_fn   = cellstr( obj_fn );
         mod_fn   = cellstr( mod_fn );
         answer   = isequal( obj_fn , mod_fn );
         if answer
            obj   = struct2cell( obj );    obj = {  obj{obj_id}};
            model = struct2cell(model);  model = {model{mod_id}};
            for i=1:length(obj)
               [ answer, msg ] = has_model( obj{i} , model{i} , obj_fn{i} );
               if ~answer
                  msg = sprintf(  '.%s%s'  ,obj_fn{i}, msg );
                  return
               end
            end
         else
            msg =  ' (struct fields don''t match the required fields)'  ;
         end
      case  'cell'
         answer   = isequal( size(obj), size(model) );
         if answer
            for i=1:length(obj)
               [ answer, msg ] = has_model( obj{i} , model{i} ,  ''  );
               if ~answer
                  msg = sprintf(  '{%d}%s'  , i , msg );
                  return
               end
            end
         else
            msg =  ' (cell-array has not the required size)'  ;
         end
      end
   end



function [ answer, msg ] = match_same( obj_cell , criterion )
   [answer, msg] = check_is( obj_cell ,  'cell'  ,                          ...
              [  'wrong argument: this option requires as first argument '  ...
                 'a cell-array containing the objects to check'           ]  ) ;
   if( ~answer ) return; end
   % both matlab and octave make A a row cell-array with: A={A{:}}
   % (notice that if A is a matrix, then A=A(:) makes A a COLUMN vector...)
   obj_cell  = {obj_cell{:}};
   [answer, msg] = check_is( numel(obj_cell)>1 ,  'true'  ,                 ...
              [  'wrong argument: this option requires as first argument '  ...
                 'a cell-array containing at least two objects to check'  ]  ) ;
   if( ~answer ) return; end
   switch( criterion )
   case  'rows'
      descr  =  'number of rows'            ;
      crit   = cellfun(  'size'             , obj_cell, 1 );
   case  'columns'
      descr  =  'number of columns'         ;
      crit   = cellfun(  'size'             , obj_cell, 2 );
   case  'length'
      descr  =  'length'                    ;
      crit   = cellfun(  'length'           , obj_cell );
   case  'numel'
      descr  =  'number of elements'        ;
      crit   = cellfun(  'prodofsize'       , obj_cell );
   case  'ndims'
      descr  =  'number of dimensions'      ;
      crit   = cellfun(  'ndims'            , obj_cell );
   case  'size'
      descr  =  'size'                      ;
      crit   = zeros( max( cellfun( 'ndims' , obj_cell ) ), ...
                      length( obj_cell ) );
      for i=1:size(crit,1)
         crit(i,:) = cellfun(  'size'   , obj_cell, i );
      end
   case  'operator=='
      descr  =  'binary operator ''=='' conformant set'  ;
      crit   = ones( max( cellfun( 'ndims' , obj_cell ) ), ...
                      length( obj_cell ) );
      for i=1:size(crit,1)
         crit(i,:) = cellfun(  'size'   , obj_cell, i );
      end
      id  =  prod(crit) == 1; % scalar objects
      nid = find(~id);
      if numel(nid) & any(id)
         crit(:,id) = crit(:,nid(1));
      end
   otherwise
      answer = 0;
      msg    = sprintf([  'wrong criterion: sorry, the criterion "%s" '  ...
                          'is not supported'  ], criterion );
      return
   end
   check  = all(~diff(crit')',1);
   answer = all( check );
   if ~answer
      bad_idx = find( ~check );
      switch( criterion )
      case  'size'
         val = { [  '[ '  ...
                    sprintf(  '%d '  , size( obj_cell{bad_idx(1)  } ) ) ...
                    ']'  ], ...
                 [  '[ '  ...
                    sprintf(  '%d '  , size( obj_cell{bad_idx(1)+1} ) ) ...
                    ']'  ]  ...
               };
      case  'operator=='
         val = { [  'size [ '  ...
                    sprintf(  '%d '  , size( obj_cell{bad_idx(1)  } ) ) ...
                    ']'  ], ...
                 [  'size [ '  ...
                    sprintf(  '%d '  , size( obj_cell{bad_idx(1)+1} ) ) ...
                    ']'  ]  ...
               };
      otherwise
         val = { sprintf(  '%d'  , crit( 1, bad_idx(1)   ) ), ...
                 sprintf(  '%d'  , crit( 1, bad_idx(1)+1 ) )  ...
               };
      end
      msg     = sprintf([  'the %s of the %s object (%s) differs from '  ...
                           'that of the %s one (%s) %s'  ],              ...
                           descr                          ,              ...
                           int2ordinal(bad_idx(1))        , val{1}      ,...
                           int2ordinal(bad_idx(1)+1)      , val{2}      ,...
                           msg );
   end



% <obj>        ::generic::
function answer = is_numeric( obj )
   answer = islogical( obj ) | isnumeric( obj );



% <obj>        ::generic::
% <obj_type>   ::string::
function answer = is_a( obj , obj_type )
   where  = sprintf(  '(in subfunction is_a of function %s)'  , mfilename );
   answer = logical(0);
   switch obj_type
   case  {  'char'  ,  'string'  }
      answer = ischar( obj );
   case  'numeric'
      answer = is_numeric( obj );
   case  'numstring'
      answer = ischar( obj ) | is_numeric( obj );
   otherwise
      fprintf( 2,  'error %s:\n check for type <%s> not implemented\n'  , ...
         where, obj_type                                                  ...
      );
      error(  ' '  );
   end



% <obj>        ::cell::
% <obj_type>   ::string::
function answer = are_cell_elements_a( obj , obj_type )
   where  = sprintf( ...
            '(in subfunction are_cell_elements_a of function %s)'  , ...
            mfilename );
   answer = logical(0);
   switch obj_type
   case  {  'char'  ,  'string'  }
      answer = cellfun(  'isclass'  , obj ,  'char'  );
   case  'numeric'
      answer = cellfun(  'isclass'  , obj ,  'double'   ) |  ...
               cellfun(  'isclass'  , obj ,  'sparse'   ) |  ... % octave
               cellfun(  'isclass'  , obj ,  'float'    ) |  ...
               cellfun(  'isclass'  , obj ,  'numeric'  ) |  ...
               cellfun(  'isclass'  , obj ,  'int8'     ) |  ...
               cellfun(  'isclass'  , obj ,  'uint8'    ) |  ...
               cellfun(  'isclass'  , obj ,  'int16'    ) |  ...
               cellfun(  'isclass'  , obj ,  'uint16'   ) |  ...
               cellfun(  'isclass'  , obj ,  'int32'    ) |  ...
               cellfun(  'isclass'  , obj ,  'uint32'   ) |  ...
               cellfun(  'isclass'  , obj ,  'logical'  ) ;
   case  'numstring'
      answer = cellfun(  'isclass'  , obj ,  'char'     ) |  ...
               cellfun(  'isclass'  , obj ,  'double'   ) |  ...
               cellfun(  'isclass'  , obj ,  'sparse'   ) |  ... % octave
               cellfun(  'isclass'  , obj ,  'float'    ) |  ...
               cellfun(  'isclass'  , obj ,  'numeric'  ) |  ...
               cellfun(  'isclass'  , obj ,  'int8'     ) |  ...
               cellfun(  'isclass'  , obj ,  'uint8'    ) |  ...
               cellfun(  'isclass'  , obj ,  'int16'    ) |  ...
               cellfun(  'isclass'  , obj ,  'uint16'   ) |  ...
               cellfun(  'isclass'  , obj ,  'int32'    ) |  ...
               cellfun(  'isclass'  , obj ,  'uint32'   ) |  ...
               cellfun(  'isclass'  , obj ,  'logical'  ) ;
   otherwise
      fprintf( 2,  'error %s:\n check for type <%s> not implemented\n'  , ...
         where, obj_type                                                  ...
      );
      error(  ' '  );
   end



function [ answer, msg ] = is_cell_of( obj , obj_type , obj_name )
   msg    =  ''  ;
   %fprintf(1,'\nanswer = is_a( obj, ''%s'' ) :' , obj_type );
   answer = is_a( obj, obj_type );
   if ~answer
      %fprintf(1,'\nanswer = iscell( obj ) :');
      answer = iscell( obj );
      if answer
         %are_types = cellfun(  'isclass'  , obj ,  obj_type  );
         are_types = are_cell_elements_a( obj , obj_type );
         %fprintf(1,'\nanswer = iscell( obj ) :');
         answer  = all( are_types );
         if ~numel( are_types )  % if empty cell-array, always match <obj_type>
            answer = logical(1);
         end
         if ~answer
            bad_idx = find( ~are_types );
            msg     = sprintf(  '%s{%d}%s'  , obj_name , bad_idx(1) , msg );
         end
      end
   end



function [ answer, msg ] = is_recursive_cell_of( obj, obj_type, obj_name, from, to )
   if nargin < 5  to    = inf;  end
   if nargin < 4  from  = 0  ;  end

   too_nested              = to   <  0;
   must_be_a_type          = to   == 0;
   can_be_a_cell           = to   >  0;
   cannot_have_subcells    = to   <= 1;
   must_be_a_cell          = from >  0;
   must_be_a_cell_of_cells = from >  1;
   % fprintf([ 'is_recursive_cell_of( obj:"%s", obj_type:"%s", ' ... %debug
   %           'obj_name:"%s", from:"%d", to:"%d" )\n' ]       , ... %debug
   %           class(obj), obj_type, obj_name, from, to )            %debug
   msg    =  ''  ;
   answer = is_a( obj, obj_type );
   if must_be_a_cell | too_nested
      answer = logical(0);
      if must_be_a_cell  & ~iscell( obj )
         msg = sprintf(  'object of type <%s> while expected a cell array'  ,...
                         class(obj)  );
         return
      end
      if   too_nested
         msg = sprintf(  'cell array while expected an object of type <%s>'  ,...
                         obj_type  );
         return
      end
   end
   if ~answer
      answer = iscell( obj ) & can_be_a_cell;
      if answer
         %are_types = cellfun(  'isclass'  , obj ,  obj_type  );
         are_types  = are_cell_elements_a( obj , obj_type );
         if cannot_have_subcells & any( ~are_types )
            answer  = logical(0);
            bad_idx = find( ~are_types );
            msg = sprintf(  '%s{%d} whose type is <%s> instead of <%s> %s'  ,...
                             obj_name , bad_idx(1) ,   ...
                             class(obj{bad_idx(1)}), obj_type, msg  );
            return
         end
         are_cells = cellfun(  'isclass'  , obj ,  'cell'    );
         if must_be_a_cell_of_cells & any( ~are_cells )
            answer  = logical(0);
            bad_idx = find( ~are_cells );
            msg = sprintf(  '%s{%d} whose type is <%s> while expected a cell array %s'  ,...
                             obj_name , bad_idx(1) ,   ...
                             class(obj{bad_idx(1)}), msg  );
            return
         end
         cell_idx=find(are_cells);
         if numel(cell_idx)
            for i=find(are_cells)
               [ are_types(i), msg ] = is_recursive_cell_of( ...
                  obj{i} , obj_type ,  ''  , from-1 , to-1   ...
               );
            end
         end
         answer  = all( are_types );
         if ~numel( are_types )  % if empty cell-array, always match <obj_type>
            answer = logical(1);
         end
         if ~answer
            bad_idx = find( ~are_types );
            msg     = sprintf(  '%s{%d}%s'  , obj_name , bad_idx(1) , msg );
         end
      end
   end



function [err, msg] = is_err( err , msg , err_msg )
   if nargin < 3
      err_msg =  ''  ;
   end
   %
   if err
      if length(msg)
         msg = sprintf(  '%s\n%s'  , msg , err_msg );
      else
         msg = err_msg;
      end
   end



function throw_error( msg )
   persistent need_prefix_;
   if ~numel( need_prefix_ )
      % try
      %    error( 'foo' )
      % catch
      %    err    = lasterr;
      %    if ~ischar( err )
      %       err = err.message;
      %    end
      % end
      [ env_name , env_PID, env_path ] = computing_env_info;
      if numel(env_path)
         % simulate an error by invoking this computing environment
         % as a sub-process: <answer> contains the actual string
         % which is used by the computing environment as error message
         [ status, answer ] = unix(         ...
            [                               ...
               'echo "error(''foo'');" | '  ...
               env_path                     ...
               ' 2>&1 | grep -E foo'        ...
            ]                               ...
         );
         need_prefix_ = ~numel( regexp( answer , '[Ee]rror:' ) );
      else % suboptimal: use lasterr
         try
            error( 'foo' )
         catch
            err    = lasterr;
            if ~ischar( err )
               err = err.message;
            end
         end
         need_prefix_ = ~numel( regexp( err , '^[Ee]rror:' ) );
      end
   end
   if need_prefix_
      error( [ 'error: ' msg ] );
   else
      error( msg );
   end




% Local Variables:
% mode:mastrave
% End:

