function [set_centered_Y, set_X] = central_value_nosparse_( X, Y , statistic , T )
% Copyright (C) 2005,2006,2007,2008 Daniele de Rigo 
% 
% This file is part of Mastrave.
% 
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
% 
% ---------------------------------------------------------------------------
% 
% [set_centered_Y, set_X] = central_value_nosparse_( X, Y , statistic , T )
% 
% central_value_nosparse_ implements a widely compatible version of the
% central_value algorithm, optimized for minimal memory usace without 
% requiring sparse matrices.
%
% WARNING: no pre- and post- condition checks are performed to ensure a fast 
% computation. This function should be used only inside code that esures the
% satisfaction of the constraints described below for the input arguments.
%
% Given an <X> vector, clusterizes <X> by extracting its unique
% value set called <set_X>.
% For each element   i  of <set_X> it will be selected a sub-set containig 
% only the elements of <X> equal to <set_X(i)>.
% The corresponding subsets from the matrix of the signal <Y> are then
% processed (subset by subset) according to the chosen <statistic>, and
% the  i-th  result is stored into the i-th <set_centered_Y> element.
% 
% If <X> is a logical vector, then the sequences of contiguous true values
% are numbered and those numbers are regarded as the set <set_X>, so excluding
% from the <statistic> computation all the <Y> elements corresponding to the
% false values of <X>.
%
% All the computations that implement the <statistic> are strictly 
% vectorial, without any use of interpreted loops.
%
%
% Input arguments:
%
% <X>                ::vector,[logical|numeric]::
%                    the independent variable samples (it must be a
%                    single vector, or at least a scalar)
%
% <Y>                ::matrix::
%                    the dependent variable samples (it must have
%                    the same nuber of rows as the <X> length: it can be
%                    a matrix too, that is R->R^n functions are supported)
%
% <statistic>        ::string::
%                    name of the statistic to be applied to each cluster of
%                    the <Y> matrix. Valid statistics are:
%
%                      statistic  |    meaning
%                    -------------+---------------------------------------
%                      'mean'     |    mean
%                      'median'   |    median
%                      'min'      |    min
%                      'max'      |    max
%                      'count'    |    number of elements
%                      'sum'      |    sum
%                      'sumsq'    |    sum of squares 
%                      'prod'     |    product
%                      'var'      |    sample variance
%                      'std'      |    sample standard deviation
%                      'var_p'    |    population variance
%                      'std_p'    |    population standard deviation
%
% <T>                ::scalar_positive::
%                    the optional cyclostationarity period of the
%                    independent variable. If passed, it will be considered
%                       X(i) + T == X(i) 
%                    for each i-th element of <X>. If omitted, it is
%                    considered <T> = infty (no periodicity)
% 
%   
% Example 1: 
% 
%   n=20;N=40;
%   x=rand(1,n);x=x(ceil(rand(1,N)*n));
%   y=[sin(x*2*pi)+10;sin(x*2*pi)]+randn(2,N)*.3;
%   [cy cx]=central_value_nosparse_(x,y,'min')
%   plot(x,y,'o',cx,cy); pause;
%   [cy cx]=central_value(x,y,'std')
%   plot(x,y,'o',cx,cy); pause;
% 
% 
% Example 2:
% 
%   T=365.25; n=floor(T); N=80*T; k=repmat([1.7 3]',1,N);
%   xx=rand(1,n)*4*T; xx=xx(ceil(rand(1,N)*n));
%   yy=exp(randn(2,N).*k)+[sin(xx/T*2*pi)*2+3; cos(xx/T*2*pi)*20+130];
% 
%   t_central_value=cputime;                           % start speed checking
%      [mean_yy, cxx]=central_value_nosparse_(xx,yy,'mean',T);
%      median_yy=central_value_nosparse_(xx,yy,'median',T);
%   t_central_value=cputime-t_central_value;           % end speed checking
%  
%   semilogy(xx,yy,'.',cxx,[mean_yy;median_yy])
% 
% comparison with the classical approach:
%    
%   t_classical_approach=cputime;                      % start speed checking
%      mxx        = mod(xx,T);
%      cxx2       = sort(mxx); 
%      cxx2       = cxx2(find([1 diff(cxx2)]));
%      mean2_yy   = zeros(size(yy,1),size(cxx2,2));
%      median2_yy = zeros(size(yy,1),size(cxx2,2)); 
%      for i=1:size(cxx2,2) 
%         for j=1:size(yy,1) 
%            mean2_yy(  j,i) = mean(  yy(j, mxx==cxx2(i))); 
%            median2_yy(j,i) = median(yy(j, mxx==cxx2(i))); 
%         end; 
%      end;
%   t_classical_approach=cputime-t_classical_approach; % end speed checking
% 
%   all(all( mean2_yy   == mean_yy   ))
%   all(all( median2_yy == median_yy ))
% 
%   ratio = t_classical_approach/t_central_value;
%   fprintf(1, '\n\tThe central_value approach is %4.2g faster\n\n', ratio )
% 
% 
% Example 3 (speed test):
% 
%   T=200; n0=floor(T); N=20*T;
%   n=n0:n0:N; samples=10;
%   t_central_value      = zeros(samples,size(n,2));
%   t_classical_approach = zeros(samples,size(n,2));
%   for i=1:length(n)
%      fprintf(1,'\nduplication ratio: %g  ', n(i)/N )
%      for s=1:samples
%         xx=rand(1,n(i))*4*T; xx=xx(ceil( rand(1,N)*n(i) ));
%         yy=[sin(xx/T*2*pi)+10;sin(xx/T*2*pi)]+randn(2,N)*.3;
%      
%         fprintf(1,' .' )
%         t_central_value(s,i) = cputime;                               % start speed checking
%            [median_yy, cxx]=central_value_nosparse_(xx,yy,'median',T);
%         t_central_value(s,i) = cputime-t_central_value(s,i);          % end speed checking
%    
%         t_classical_approach(s,i) =cputime;                           % start speed checking
%            mxx        = mod(xx,T);
%            cxx2       = sort(mxx); 
%            cxx2       = cxx2(find([1 diff(cxx2)]));
%            median2_yy = zeros(size(yy,1),size(cxx2,2)); 
%            for j=1:size(cxx2,2) 
%               for k=1:size(yy,1)  
%                  median2_yy(k,j) = median(yy(k, mxx==cxx2(j))); 
%               end; 
%            end;
%         t_classical_approach(s,i) =cputime-t_classical_approach(s,i); % end speed checking
%      end;
%   end;
%   plot(n/N,[t_classical_approach./t_central_value],'ob')
%   xlabel('non duplicated data ratio')
%   ylabel('speed ratio: (classical approach)/(central value)')
% 
% version: 0.2.5

set_centered_Y = [];
set_X          = [];

X = X(:)';

if islogical(X)
   if all(~X) return; end
   id_beg   = find( diff([0 X 0]) ==  1 );
   id_end   = find( diff([0 X 0]) == -1 );
   len      = id_end - id_beg;
   n        = length(len);
   set_X    = [1:n]';
else
   [X, sid] = sort(X);
   Y        = Y(sid,:);
   id_sep   = find( [1 diff(X) 1] ~= 0 ); 
   n_sep    = length(id_sep);
   id_beg   = id_sep(1:n_sep-1);
   id_end   = id_sep(2:n_sep);
   len      = id_end - id_beg;
   n        = length(len);
   set_X    = X(id_beg)';
end
id       = ones(1,sum(len));
id( cumsum([1 len(1:n-1)]) ) = ...
    diff([0 id_beg]) - [1 len(1:n-1)] +1;
id       = cumsum(id)

col_id   = zeros(1,sum(len));
col_id( cumsum([1 len(1:n-1)]) ) = 1;
col_id   = cumsum(col_id)

row_id   = ones(1,sum(len));
row_id( cumsum(len(1:n-1))+1 ) = 1-len(1:n-1);
row_id   = cumsum(row_id)

if size(Y,2) == 1
   Up    = zeros( 1, max(len)*n );
   Up( (col_id-1)*max(len) + row_id ) = Y( id );
   Up    = reshape( Up, max(len), n )
   sum(Up)
end
U        = Y( id, : )
   
last   = cumsum(len);
first  = cumsum([1 len(1:length(len)-1)]);
   
switch statistic

case  'mean'
   Uc             = cumsum(U)
   set_centered_Y = [ Uc(last(1),:) ;     ...
                      Uc(last(2:n)  ,:) - ...
                      Uc(last(1:n-1),:)   ]./len(ones(1,size(Y,2)),:)';
case  'count'                      
   set_centered_Y = len(ones(1,size(Y,2)),:)';
case  'sum'
   Uc             = cumsum(U)
   set_centered_Y = [ Uc(last(1),:) ;     ...
                      Uc(last(2:n)  ,:) - ...
                      Uc(last(1:n-1),:)   ];
case  'sumsq'
   Uc             = cumsum(U.^2)
   set_centered_Y = [ Uc(last(1),:) ;     ...
                      Uc(last(2:n)  ,:) - ...
                      Uc(last(1:n-1),:)   ];
case  'prod'
   Uc             = cumprod(U)
   set_centered_Y = [ Uc(last(1),:) ;     ...
                      Uc(last(2:n)  ,:)./ ...
                      Uc(last(1:n-1),:)   ];
end   


% Local Variables:
% mode:mastrave
% End:

