function [ indexes, cardinality, glossary ] = cellstr2index( cellstring )
% Copyright (C) 2005,2006,2007,2008,2009 Daniele de Rigo
%
% This file is part of Mastrave.
%
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
%
% ---------------------------------------------------------------------------
%
% [ indexes, cardinality, glossary ] = cellstr2index( cellstring )
%
% Strings cannot be used as indexes.
% cellstr2index(...) maps strings into positive integers preserving their
% order.
% If <cellstring> is a cell-array of cellstrings then <indexes> is returned as a
% cell-array whose elements are the corresponding matrices of indices.
% Each index of <indexes> represents a value of (the elements of) <cellstring>
% which may occur several times in <cellstring>. To each occurrence of the same
% value corresponds the same index.
% The set of these correspondences is returned in the cell of strings
% <glossary>, whose length is the <cardinality> of the indices set.
%
%
% Input arguments:
%
% <cellstring>             ::cellstring-1|cellstring-2::
%                          cell array of strings or of string cell arrays
%
%
% Example of usage:
%
%    m1            =  { 'foo' , 'bar', '' , 'baz' }
%    m2            =  { 'foobar' , 'bar' , 'quux' , 'foo' , 'bar' }
%    m3            =  { }
%    m4            =  { '' ; 'foo' ; 'foo' ; '' }
%    idx           = cellstr2index( m1 )
%    idx           = cellstr2index( m2 )
%    [idx, len]    = cellstr2index( { m1 , m2 } )
%    [idx, len]    = cellstr2index( { m1 , m2 , m3 } )
%    [idx, len]    = cellstr2index( { m1 , m2 , m4 } )
%    [idx, len, g] = cellstr2index( { m1 , m2 , m3 , m4 } )
%    [idx, len, g] = cellstr2index( { m1 , m2 , m3 , 'foo' , m4 , '' } )
%    [idx, len, g] = cellstr2index( 'foo' )
%    [idx, len, g] = cellstr2index( { } )
%    [idx, len, g] = cellstr2index( { {} , {} } )
%
%
% version: 0.3.6

where = sprintf(  '(in function %s)'  , mfilename );

% mastrave-kernel: this function is used by check_nargin, therefore the
% test for the allowed number of input arguments is performed without
% invoking check_nargin

if(nargin~=1)
   fprintf( 2,  'Error: expected one argument instead of %d.\n'  , nargin );
   fprintf( 2,  'Usage: cellstr2index( cellstring )\n'  );
   error(  ' '  );
end

check_is( cellstring ,  'cellstring-r'                       ,  ...
          [  '%s the argument <cellstring> must be a cell '     ...
             'array of strings or of string cell arrays'     ], where );

[ indexes, cardinality, glossary ] = deal( {} , 0 , {} );

if ischar( cellstring )
   cellstring = { cellstring };
end
if ~numel( cellstring ) return; end
is_cell_of_cellstrings = any( cellfun( 'isclass' , cellstring , 'cell' ) );
if is_cell_of_cellstrings
   [ len , str_lid ] = get_numel( cellstring );
   clen = [ 0 ; cumsum( len(:) ) ];
   n    = cell( clen(end), 1 );
   for i=1:numel(cellstring)
      if str_lid(i)
         n( clen(i)+1 )            = cellstring(i);
      else
         n( clen(i) + [1:len(i)] ) = cellstring{i};
      end
   end
else
   n = cellstring;
end


[revid, sn, sid] = score( n );
% strings to integer indexes mapping
if numel(sn)
   first_lid     = strcmp( sn(1:end-1), sn(2:end) );
   first_lid     = [ logical(1) ; ~first_lid(:) ];
   glossary      = sn( first_lid );
   in            = cumsum( first_lid );
   cardinality   = in(end);    % amount of distinct elements
   in            = in(revid);  % un-sort <in>
else
   first_lid     = logical( [] );
   glossary      = {};
   in            = [];
end

% replace <cellstring> values with their corresponding indexes
indexes = cellstring;
if is_cell_of_cellstrings
   for i=1:numel(cellstring)
      if str_lid(i)
         indexes{i}    = in( clen(i)+1 );
      else
%         strcell 2 numcell mapping
%           indexes{i}(:) = mat2cell(       ...
%              in( clen(i) + [1:len(i)] ) , ...
%              ones(1,len(i)) , 1           ...
%           );
         indexes{i} = reshape( in( clen(i) + [1:len(i)] ) , size(indexes{i}) );
      end
   end
else
%  strcell 2 numcell mapping
%     indexes(:) = mat2cell( in , ones(1,numel(in)) , 1 );
   indexes = reshape( in , size(indexes) );
end



function [len , str_lid] = get_numel( cellstring )
   len     = cellfun(  'prodofsize'  , cellstring );
   str_lid = ~cellfun( 'isclass' , cellstring , 'cell' );
   if iscell(str_lid)  str_lid = logical([str_lid{:}]); end
   len( str_lid ) = 1;



% Local Variables:
% mode:mastrave
% End:

