/* marray/marray_int.h
 *
 * Copyright (C) 2003, 2004, 2007 Jordi Burguet-Castell
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   Free Software Foundation, Inc.
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */

/*
 * This file contains the basic declarations for a marray.
 */
#ifndef __MARRAY_int_H__
#define __MARRAY_int_H__

#include <stdlib.h>


#include <gsl/gsl_types.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_check_range.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_vector.h>

#include "marray_utilities.h"

#undef __BEGIN_DECLS
#undef __END_DECLS
#ifdef __cplusplus
# define __BEGIN_DECLS extern "C" {
# define __END_DECLS }
#else
# define __BEGIN_DECLS /* empty */
# define __END_DECLS /* empty */
#endif

__BEGIN_DECLS


/*
 * A marray is a struct with the rank of the marray (number of
 * indices it has), an array with the number of elements available for
 * each index, and an array to store the n1*n2*...*nm values.
 *
 * For the moment, there is no tda, as opossed to matrices, because it
 * would complicate quite a bit the algorithms and probably it is not
 * worth it.
 */
typedef struct
{
  unsigned int rank;
  size_t * dimension;
  size_t size;
  int * data;
} marray_int;


/*
 * There is not such a thing as "marray views", in contrast with the
 * case for gsl_matrix.
 */

/* Allocation */

marray_int *
marray_int_alloc(const unsigned int rank, const size_t * dimension);

marray_int *
marray_int_calloc(const unsigned int rank, const size_t * dimension);

marray_int *
marray_int_copy(marray_int * t);

void marray_int_free(marray_int * t);


/* Views */

/*
 * There are no views.
 */


/* Conversions */

gsl_matrix_int * marray_int_2matrix(marray_int * t);
gsl_vector_int * marray_int_2vector(marray_int * t);


/* Operations */

int marray_int_get(const marray_int * t, const size_t * indices);
void marray_int_set(marray_int * t, const size_t * indices, const int x);


int * marray_int_ptr(marray_int * t, const size_t * indices);
const int * marray_int_const_ptr(const marray_int * t, const size_t * indices);

void marray_int_set_zero(marray_int * t);
void marray_int_set_all(marray_int * t, int x);

int marray_int_fread(FILE * stream, marray_int * t);
int marray_int_fwrite(FILE * stream, const marray_int * t);
int marray_int_fscanf(FILE * stream, marray_int * t);
int marray_int_fprintf(FILE * stream, const marray_int * t, const char * format);

int marray_int_memcpy(marray_int * dest, const marray_int * src);
int marray_int_swap(marray_int * t1, marray_int * t2);

int marray_int_max(const marray_int * t);
int marray_int_min(const marray_int * t);
void marray_int_minmax(const marray_int * t, int * min_out, int * max_out);

void marray_int_max_index(const marray_int * t, size_t * indices);
void marray_int_min_index(const marray_int * t, size_t * indices);
void marray_int_minmax_index(const marray_int * t, size_t * imin, size_t * imax);

int marray_int_isnull(const marray_int * t);

int marray_int_add(marray_int * a, const marray_int * b);
int marray_int_sub(marray_int * a, const marray_int * b);
int marray_int_mul_elements(marray_int * a, const marray_int * b);
int marray_int_div_elements(marray_int * a, const marray_int * b);
int marray_int_scale(marray_int * a, const double x);
int marray_int_add_constant(marray_int * a, const double x);


/* inline functions if you are using GCC */

#ifdef HAVE_INLINE
extern inline
size_t
marray_int_position(const size_t * indices, const marray_int * t)
{
  size_t shift, position;
  unsigned int i;

  shift = t->size;
  position = 0;
  for (i = 0; i < t->rank; i++)
    {
#if GSL_RANGE_CHECK
      if (indices[i] >= t->dimension[i])
        return t->size;
#endif
      shift /= t->dimension[i];
      position += shift * indices[i];
    }

  return position;
}


extern inline 
int
marray_int_get(const marray_int * t, const size_t * indices)
{
  size_t position;

  position = marray_int_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VAL("index out of range", GSL_EINVAL, 0);
#endif

  return t->data[position];
}


extern inline
void
marray_int_set(marray_int * t, const size_t * indices, const int x)
{
  size_t position;
  
  position = marray_int_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VOID("index out of range", GSL_EINVAL);
#endif

  t->data[position] = x;
}


extern inline 
int *
marray_int_ptr(marray_int * t, const size_t * indices)
{
  size_t position;

  position = marray_int_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (int *) (t->data + position);
}


extern inline 
const int *
marray_int_const_ptr(const marray_int * t, const size_t * indices)
{
  size_t position;

  position = marray_int_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (const int *) (t->data + position);
} 

#endif

__END_DECLS

#endif /* __MARRAY_int_H__ */
