/* marray/marray_float.h
 *
 * Copyright (C) 2003, 2004, 2007 Jordi Burguet-Castell
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   Free Software Foundation, Inc.
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */

/*
 * This file contains the basic declarations for a marray.
 */
#ifndef __MARRAY_float_H__
#define __MARRAY_float_H__

#include <stdlib.h>


#include <gsl/gsl_types.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_check_range.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_vector.h>

#include "marray_utilities.h"

#undef __BEGIN_DECLS
#undef __END_DECLS
#ifdef __cplusplus
# define __BEGIN_DECLS extern "C" {
# define __END_DECLS }
#else
# define __BEGIN_DECLS /* empty */
# define __END_DECLS /* empty */
#endif

__BEGIN_DECLS


/*
 * A marray is a struct with the rank of the marray (number of
 * indices it has), an array with the number of elements available for
 * each index, and an array to store the n1*n2*...*nm values.
 *
 * For the moment, there is no tda, as opossed to matrices, because it
 * would complicate quite a bit the algorithms and probably it is not
 * worth it.
 */
typedef struct
{
  unsigned int rank;
  size_t * dimension;
  size_t size;
  float * data;
} marray_float;


/*
 * There is not such a thing as "marray views", in contrast with the
 * case for gsl_matrix.
 */

/* Allocation */

marray_float *
marray_float_alloc(const unsigned int rank, const size_t * dimension);

marray_float *
marray_float_calloc(const unsigned int rank, const size_t * dimension);

marray_float *
marray_float_copy(marray_float * t);

void marray_float_free(marray_float * t);


/* Views */

/*
 * There are no views.
 */


/* Conversions */

gsl_matrix_float * marray_float_2matrix(marray_float * t);
gsl_vector_float * marray_float_2vector(marray_float * t);


/* Operations */

float marray_float_get(const marray_float * t, const size_t * indices);
void marray_float_set(marray_float * t, const size_t * indices, const float x);


float * marray_float_ptr(marray_float * t, const size_t * indices);
const float * marray_float_const_ptr(const marray_float * t, const size_t * indices);

void marray_float_set_zero(marray_float * t);
void marray_float_set_all(marray_float * t, float x);

int marray_float_fread(FILE * stream, marray_float * t);
int marray_float_fwrite(FILE * stream, const marray_float * t);
int marray_float_fscanf(FILE * stream, marray_float * t);
int marray_float_fprintf(FILE * stream, const marray_float * t, const char * format);

int marray_float_memcpy(marray_float * dest, const marray_float * src);
int marray_float_swap(marray_float * t1, marray_float * t2);

float marray_float_max(const marray_float * t);
float marray_float_min(const marray_float * t);
void marray_float_minmax(const marray_float * t, float * min_out, float * max_out);

void marray_float_max_index(const marray_float * t, size_t * indices);
void marray_float_min_index(const marray_float * t, size_t * indices);
void marray_float_minmax_index(const marray_float * t, size_t * imin, size_t * imax);

int marray_float_isnull(const marray_float * t);

int marray_float_add(marray_float * a, const marray_float * b);
int marray_float_sub(marray_float * a, const marray_float * b);
int marray_float_mul_elements(marray_float * a, const marray_float * b);
int marray_float_div_elements(marray_float * a, const marray_float * b);
int marray_float_scale(marray_float * a, const double x);
int marray_float_add_constant(marray_float * a, const double x);


/* inline functions if you are using GCC */

#ifdef HAVE_INLINE
extern inline
size_t
marray_float_position(const size_t * indices, const marray_float * t)
{
  size_t shift, position;
  unsigned int i;

  shift = t->size;
  position = 0;
  for (i = 0; i < t->rank; i++)
    {
#if GSL_RANGE_CHECK
      if (indices[i] >= t->dimension[i])
        return t->size;
#endif
      shift /= t->dimension[i];
      position += shift * indices[i];
    }

  return position;
}


extern inline 
float
marray_float_get(const marray_float * t, const size_t * indices)
{
  size_t position;

  position = marray_float_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VAL("index out of range", GSL_EINVAL, 0);
#endif

  return t->data[position];
}


extern inline
void
marray_float_set(marray_float * t, const size_t * indices, const float x)
{
  size_t position;
  
  position = marray_float_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VOID("index out of range", GSL_EINVAL);
#endif

  t->data[position] = x;
}


extern inline 
float *
marray_float_ptr(marray_float * t, const size_t * indices)
{
  size_t position;

  position = marray_float_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (float *) (t->data + position);
}


extern inline 
const float *
marray_float_const_ptr(const marray_float * t, const size_t * indices)
{
  size_t position;

  position = marray_float_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (const float *) (t->data + position);
} 

#endif

__END_DECLS

#endif /* __MARRAY_float_H__ */
