/*
 * Make7Libre, free (as in freedom) implementation of the
 * unfamous game 'Make 7 Hexa Puzzle'
 * Copyright 2016-2017 (C) Felicien PILLOT <felicien.pillot@member.fsf.org>
 * 
 * This file is part of Make7Libre.
 * 
 * Make7Libre is free software: you can redistribute it and/or modify
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Make7Libre is distributed in the hope that it will be useful,
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * with Make7Libre.  If not, see <http://www.gnu.org/licenses/>.
 */
/***********
 * game.cc *
 * ~~~~~~~ *
 ***********/

#include <sys/stat.h>		// mkdir () function
#include "../include/game.h"


/*
 * /////////////////////////
 * //   Public functions  //
 * /////////////////////////
 */

/**
 * Here are defined the private variables, the GTK widgets, etc.
 * Maps and lists containing static informations are initialized,
 * and CssProvider is created and set.
 */
Game::Game () :
/* All these widgets are defined in the basic.glade file, and imported with Gtk::Builder */
// Windows of the game :
make7window (), make7scores (), make7message (), make7error_board (),
make7confirm (), make7askname (), make7credits (),
// Main window title bar and menu items :
headerbar (), appmenu (), newgame (), scores (), credits (), quit (),
// Buttons for the interface :
button_cancel (), button_credits (), button_help (),
button_new (), button_next (), button_quit (), button_rotate (),
button_save (), button_scores (), button_erase (),
// Buttons forming the Board :
nnw (), nn (), nne (), nww (), nw (), ne (), nee (), ww (), w (), m (), e (), ee (), sww (),
sw (), se (), see (), ssw (), ss (), sse (),
// Buttons (not clickable) forming the SelectionArea (under the Board) :
mainNw (), mainNe (), mainW (), mainM (), mainE (), mainSw (), mainSe (),
// Other miscellaneous widgets :
current_score (), box_score (), user_entry (),
/* Now these are global (privates) variables, shared between all functions */
Board (), SelectionArea (), BoardSnapshotMap (), Colors (), BoardKeys (), SelectionAreaKeys (), ColorsKeys (),
pos2 (), max (), valHexa1 (), valHexa2 (), score (), nMatch (), oldValHexa1 (), oldValHexa2 (),
CssProvider ()
{
	// Define keys
	BoardKeys = { 13, 14, 15, 22, 23, 24, 25,
			31, 32, 33, 34, 35, 41, 42, 43,
			44, 51, 52, 53 };
	SelectionAreaKeys = { -10, -9, -1, 0,	1, 9, 10 };
	ColorsKeys = {0, 1, 2, 3, 4, 5, 6, 7};
	Colors = {{0, "nocolor"}, {1, "purple"}, {2, "orange"}, {3, "red"},
			{4, "blue"}, {5, "green"}, {6, "yellow"}, {7, "multicolor"}};
	// Define style sheet
	CssProvider = Gtk::CssProvider::create();
	if (! CssProvider->load_from_path
			(Glib::build_filename (UI_DIR, "theme.css")))
	{
		std::cout << "Failed to load CSS file" << std::endl;
		return;
	}
}

void
Game::connect_every_button ()
{
	/* Interface buttons */
	// Cancel action
	if (button_cancel)
		button_cancel->signal_clicked ().connect
		(sigc::mem_fun (*this, &Game::on_cancel_action_activate));
	// Close scores
	if (button_close_scores)
		button_close_scores->signal_clicked ().connect
		(sigc::mem_fun (*this, &Game::on_close_scores_activate));
	// Display help
	if (button_help)
		button_help->signal_clicked ().connect
		(sigc::mem_fun (*this, &Game::on_display_help_activate));
	// Next hexa
	if (button_next)
		button_next->signal_clicked ().connect
		(sigc::mem_fun (*this, &Game::on_next_hexa_activate));
	/* Menu items */
	if (newgame)
		newgame->signal_activate ().connect
		(sigc::mem_fun (*this, &Game::on_new_game_activate));
	if (scores)
		scores->signal_activate ().connect
		(sigc::mem_fun (*this, &Game::on_display_scores_activate));
	if (credits)
		credits->signal_activate ().connect
		(sigc::mem_fun (*this, &Game::on_display_credits_activate));
	if (quit)
		quit->signal_activate ().connect
		(sigc::mem_fun (*this, &Game::on_quit_game_activate));
	/* Score save entry */
	if (user_entry)
		user_entry->signal_editing_done().connect
		(sigc::mem_fun (*this, &Game::on_entry_editing_done));
	/* Game buttons */
	for (int key : BoardKeys)
		Board [key]->signal_clicked ().connect
		(sigc::bind<int> (sigc::mem_fun (*this, &Game::game_button_selected), key) );
	SelectionArea[0]->signal_clicked () .connect
			(sigc::mem_fun (*this, &Game::rotate_hexas_left));
	SelectionArea[0]->signal_button_press_event () .connect
			(sigc::mem_fun (*this, &Game::rotate_hexas_right));
}

void
Game::display (Glib::RefPtr<Gtk::Application> app)
{
	/** Accelerators settings **/
	newgame->add_accelerator ("activate", make7window->get_accel_group (), 'n',
			Gdk::CONTROL_MASK, Gtk::ACCEL_VISIBLE);
	scores->add_accelerator ("activate", make7window->get_accel_group (), 's',
			Gdk::CONTROL_MASK, Gtk::ACCEL_VISIBLE);
	credits->add_accelerator ("activate", make7window->get_accel_group (), 'c',
			Gdk::CONTROL_MASK, Gtk::ACCEL_VISIBLE);
	quit->add_accelerator ("activate", make7window->get_accel_group (), 'q',
			Gdk::CONTROL_MASK, Gtk::ACCEL_VISIBLE);
	button_help->add_accelerator ("activate", make7window->get_accel_group (), 'h',
			Gdk::CONTROL_MASK, Gtk::ACCEL_VISIBLE);
	button_erase->add_accelerator ("activate", make7window->get_accel_group (), 'e',
			Gdk::CONTROL_MASK, Gtk::ACCEL_VISIBLE);
	button_cancel->add_accelerator ("activate", make7window->get_accel_group (), 'z',
			Gdk::CONTROL_MASK, Gtk::ACCEL_VISIBLE);
	button_next->add_accelerator ("activate", make7window->get_accel_group (), ' ',
			Gdk::CONTROL_MASK, Gtk::ACCEL_VISIBLE);

	/** Window settings **/
	Glib::RefPtr<Gdk::Pixbuf> icon =
			Gdk::Pixbuf::create_from_file (ICON_DIR "/32x32/apps/make7libre.png");
	make7window->set_icon (icon);
	make7window->set_has_resize_grip (true);
	Glib::RefPtr<Gtk::StyleContext> WinContext =
			make7window->get_style_context ();
	WinContext->add_provider (CssProvider,
			GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
	make7window->signal_delete_event (). connect
			(sigc::mem_fun (*this, &Game::on_make7window_hide));
	app->run (*make7window);
}

void
Game::import_every_widget ()
{
	Glib::RefPtr<Gtk::Builder> builder = Gtk::Builder::create_from_file
			(UI_DIR "/basic.glade");
	// Get every Make 7 windows
	builder->get_widget ("make7window", make7window);
	make7window->add_accel_group (Gtk::AccelGroup::create ());
	builder->get_widget ("make7scores", make7scores);
	builder->get_widget ("make7credits", make7credits);
	builder->get_widget ("make7error_board", make7error_board);
	builder->get_widget ("make7confirm", make7confirm);
	builder->get_widget ("make7askname", make7askname);
	builder->get_widget ("make7message", make7message);
	// Get titlebar and menu actions
	builder->get_widget ("headerbar", headerbar);
	make7window->set_titlebar (*headerbar);
	builder->get_widget ("newgame", newgame);
	builder->get_widget ("scores", scores);
	builder->get_widget ("credits", credits);
	builder->get_widget ("quit", quit);
	builder->get_widget ("appmenu", appmenu);
	appmenu->set_accel_group (make7window->get_accel_group ());
	// Get every interface buttons
	builder->get_widget ("button_cancel", button_cancel);
	builder->get_widget ("button_close_scores", button_close_scores);
	builder->get_widget ("button_erase", button_erase);
	builder->get_widget ("button_help", button_help);
	builder->get_widget ("button_next", button_next);
	builder->get_widget ("button_apply_save", button_save);
	// Get score label, box and user entry
	builder->get_widget ("current_score", current_score);
	builder->get_widget ("box_score", box_score);
	builder->get_widget ("user_entry", user_entry);
	// Get every game buttons
	// (board)
	builder->get_widget ("nnw", nnw);
	builder->get_widget ("nn", nn);
	builder->get_widget ("nne", nne);
	builder->get_widget ("nww", nww);
	builder->get_widget ("nw", nw);
	builder->get_widget ("ne", ne);
	builder->get_widget ("nee", nee);
	builder->get_widget ("ww", ww);
	builder->get_widget ("w", w);
	builder->get_widget ("m", m);
	builder->get_widget ("e", e);
	builder->get_widget ("ee", ee);
	builder->get_widget ("sww", sww);
	builder->get_widget ("sw", sw);
	builder->get_widget ("se", se);
	builder->get_widget ("see", see);
	builder->get_widget ("ssw", ssw);
	builder->get_widget ("ss", ss);
	builder->get_widget ("sse", sse);
	// (selection area)
	builder->get_widget ("mainNw", mainNw);
	builder->get_widget ("mainNe", mainNe);
	builder->get_widget ("mainW", mainW);
	builder->get_widget ("mainM", mainM);
	builder->get_widget ("mainE", mainE);
	builder->get_widget ("mainSw", mainSw);
	builder->get_widget ("mainSe", mainSe);
	// For much more convenience, all these buttons are stocked in std::maps<int, Glib::ustring>
	Board = { {13, nnw}, {14, nn}, {15, nne}, {22, nww},
			{23, nw}, {24, ne}, {25, nee}, {31, ww},
			{32, w}, {33, m}, {34, e}, {35, ee}, {41, sww},
			{42, sw}, {43, se}, {44, see}, {51, ssw},
			{52, ss}, {53, sse} };
	SelectionArea = { {-10, mainNw}, {-9, mainNe},
			{-1, mainW}, {0, mainM}, {1, mainE},
			{9, mainSw}, {10, mainSe} };
}

void
Game::launch_game ()
{
	// Get score list from file
	get_scores ();
	// Reset the current score
	current_score->set_label ("0 pts");
	// Reset the maximum of values proposed in selection area
	max = 1;
	// Clean board
	reset_board ();
	// Prepare the selection area
	get_random_hexas ();
	refresh_selection_area ();
}


/*
 * /////////////////////////
 * //  Private functions  //
 * /////////////////////////
 */

void
Game::big_badaboom (std::list<int> hexas)
{
	int nb;
	for (int hexa : hexas)
	{
		for (int key : SelectionAreaKeys)
		{
			nb = get_neighbour (key, hexa);
			if (nb != -42 && Board[nb]->get_label () != "")
			{
				Board[nb]->set_label ("");
				update_hexa_color (nb);
			}
		}
	}
}

void
Game::check_end ()
{
	std::map<int, Glib::ustring> tmpBoard = get_board_snapshot ();
	int nFull = 0, nEmpty = 0;
	for (auto const& hexa : tmpBoard)
	{
		if (hexa.second == "")
			nEmpty++;
		else
			nFull++;
	}
	if (nEmpty == 19)
	{
		make7message->set_message ("Congratulation, you win !");
		if (make7message->run ())
		{
			make7message->hide ();
			write_scores ();
			/* TODO
			 * Not a new game, but a new board !!!
			 */
			launch_game ();
		}
	}
	if (nFull == 19)
	{
		make7message->set_message ("You lose, no more place available !\nTry again !");
		make7message->show ();
		if (make7message->run ())
		{
			make7message->hide ();
			write_scores ();
			launch_game ();
		}
	}
}

void
Game::display_error (int i)
{
	Glib::ustring message;
	if (i == 1)
		message = "Error ! This location is not empty !";
	if (i == 2)
		message = "Error ! One of the hexa is out of the board !";
	if (i == 3)
		message = "Error ! You have not enough score points !";
	if (i == 4)
		message = "Error ! There is nothing here !";
	if (i == 5)
		message = "Error undefined !";
	make7error_board->set_message (message);
	make7error_board->show ();
}

void
Game::define_hexa (int ref, Glib::ustring valref)
{
	// Update the label
	Board[ref]->set_label (valref);
	update_hexa_color (ref);
	// This functions calls every half-second define_hexa(), until it returns false
	Glib::signal_timeout ().connect
			(sigc::bind (sigc::mem_fun (*this, &Game::triggers), ref), 500);
}

void
Game::erase_hexa (int ref)
{
	nMatch = -50;
	if (Board[ref]->get_label () == "")
		display_error (4);
	else
	{
		if (refresh_score ())
		{
			Board[ref]->set_label ("");
			update_hexa_color (ref);
		}
	}
	button_erase->set_active (false);
	// At this point, the game could be finished.
	check_end ();
	return;
}

std::map<int, Glib::ustring>
Game::get_board_snapshot ()
{
	std::map<int, Glib::ustring> map;
	for (int i : BoardKeys)
		map[i] = Board[i]->get_label ();
	return map;
}

void
Game::game_button_selected (int ref)
{
	// Backup the board and the selection area
	BoardSnapshotMap = get_board_snapshot ();
	store_oldvalhexas ();
	button_cancel->set_sensitive(true);

	/* Now the following actions can be triggered : */
	// Are we erasing this hexa ?
	if (button_erase->get_active ())
		erase_hexa (ref);
	// Are we defining a new hexa(s) ?
	else if (Board[ref]->get_label () == "")
	{
		bool there_is_a_second_hexa_to_define = false;
		int nb = -42;
		if (pos2 != -42)
		{
			nb = get_neighbour (pos2, ref);
			if (nb == -42)
			{
				display_error (2);
				return;
			}
			else if (Board[nb]->get_label () != "")
			{
				display_error (1);
				return;
			}
			else
				// If there is a second hexa, and it doesn't override an existing value,
				// neither is out of the board, then put its value on the board
				there_is_a_second_hexa_to_define = true;
		}
		// Put the value of the main hexa on the board it it doesn't override an existing value
		define_hexa (ref, get_main_hexa ());
		if (there_is_a_second_hexa_to_define)
			define_hexa (nb, get_second_hexa ());
		// Then it starts a new turn
		get_random_hexas ();
		refresh_selection_area ();
	}
	else
		display_error (1);
}

Glib::ustring
Game::get_config_file ()
{
	Glib::ustring configdir = Glib::get_user_config_dir () + "/make7libre";
	Glib::ustring filepath = configdir + "/history";
	if (file_test (filepath, Glib::FILE_TEST_IS_REGULAR))
		return filepath;
	else
	{
		// Create directory
		mkdir (configdir.c_str (), 0755);
		Glib::KeyFile *new_config_file = new Glib::KeyFile ();
		// Get current time
		Glib::TimeVal *cur_time = new Glib::TimeVal;
		cur_time->assign_current_time ();
		Glib::ustring top_comment = " This file was firstly created by Make7Libre, the "
				+ cur_time->as_iso8601 ();
		// Fill and create the file
		new_config_file->set_comment (top_comment);
		new_config_file->save_to_file (filepath);
		return filepath;
	}
}


bool
Game::get_confirmation ()
{
	if (make7confirm->run () == -8)
	{
		make7confirm->hide ();
		if (std::stoi (current_score->get_label ()) != 0)
			write_scores ();
		return true;
	}
	// else
	make7confirm->hide ();
	return false;
}

Glib::ustring
Game::get_main_hexa ()
{
	return mainM->get_label ();
}

/** Get the hexa around the given one (ref)
 *
 */
int
Game::get_neighbour (int rel, int ref)
{
	if ((rel == -10 && (ref == 15 || ref == 14 || ref == 13 ||
			ref == 22 || ref == 31))
			||
			(rel == -9 && (ref == 13 || ref == 14 || ref == 15 ||
					ref == 25 || ref == 35))
					||
					(rel == -1 && (ref == 13 || ref == 22 || ref == 31 ||
							ref == 41 || ref == 51))
							||
							(rel == 1 && (ref == 15 || ref == 25 || ref == 35 ||
									ref == 44 || ref == 53))
									||
									(rel == 9 && (ref == 31 || ref == 41 || ref == 51 ||
											ref == 52 || ref == 53))
											||
											(rel == 10 && (ref == 51 || ref == 52 || ref == 53 ||
													ref == 44 || ref == 35)))
		// Forbidden move
		return -42;
	else
		return (rel + ref);
}

int
Game::get_pos2 ()
{
	return pos2;
}

void
Game::get_random_hexas ()
{
	int nHexa = rand () % 2;
	if (nHexa == 1)
	{
		valHexa1 = rand () % max + 1;
		set_pos2 (-42);
	}
	if (nHexa == 0)
	{
		valHexa1 = rand () % max + 1;
		valHexa2 = rand () % max + 1;
		set_pos2 (9);
	}
}

void
Game::get_scores ()
{
	Glib::ustring config_file = get_config_file ();
	Glib::KeyFile *saved_scores = new Glib::KeyFile ();
	saved_scores->load_from_file (config_file);
	// Clear box_score
	std::vector<Gtk::Widget*> old_score_vect = box_score->get_children();
	for (Gtk::Widget* old_score : old_score_vect)
	{
		box_score->remove (*old_score);
	}
	if (saved_scores->has_group ("Scores"))
	{
		Glib::ArrayHandle<Glib::ustring> names =  saved_scores->get_keys ("Scores");
		for (Glib::ustring user : names)
		{
			std::list<int> user_scores = saved_scores->get_integer_list ("Scores", user);
			for (int u_score : user_scores)
			{
				Gtk::Label *user_score_label = new Gtk::Label ();
				Glib::ustring line = user + "\t: " + std::to_string (u_score);
				user_score_label->set_text (line);
				box_score->pack_start (*user_score_label);
			}
		}
		box_score->show_all ();
	}
}

Glib::ustring
Game::get_second_hexa ()
{
	if (pos2 != -42)
		return SelectionArea[pos2]->get_label ();
	return "nosecondhexa";
}

void
Game::merge_hexas (int newHexa, std::list<int> oldHexas)
{
	// Erase old hexas
	for (int i : oldHexas)
	{
		if (i != newHexa)
		{
			Board[i]->set_label ("");
			update_hexa_color (i);
		}
	}
	// Change label
	if (Board[newHexa]->get_label () != "")
	{
		int val = std::stoi (Board[newHexa]->get_label ()) + 1;
		if (val > max && val < 8)
			max = val;
		if (val == 8)
		{
			big_badaboom (oldHexas);
			nMatch = 100;
			refresh_score ();
		}
		else
			Board[newHexa]->set_label (std::to_string (val));
	}
	else
		std::cerr << "No label !" << std::endl;
	update_hexa_color (newHexa);
}

void
Game::on_cancel_action_activate ()
{
	nMatch = -100;
	if (refresh_score ())
		reset_snapshot ();
}

void
Game::on_close_scores_activate ()
{
	make7scores->hide ();
}

void
Game::on_display_credits_activate ()
{
	if (make7credits->run () == -4)
		make7credits->hide ();
}

void
Game::on_display_help_activate ()
{
	// TODO
	// Display help
	system ("yelp");
}

void
Game::on_display_scores_activate ()
{
	make7scores->show ();
}

void
Game::on_entry_editing_done ()
{
	std::cout << "Event enter !!!" << std::endl;
	button_save->clicked ();
}

bool
Game::on_make7window_hide (_GdkEventAny* event)
{
	if (get_confirmation ())
		return false;
	else
		return true;
}

void
Game::on_new_game_activate ()
{
	if (get_confirmation ())
		launch_game ();
}

void
Game::on_next_hexa_activate ()
{
	nMatch = -75;
	if (refresh_score ())
	{
		get_random_hexas ();
		refresh_selection_area ();
	}
}

void
Game::on_quit_game_activate ()
{
	make7window->close ();
}

int
Game::refresh_score ()
{
	int Iscore = std::stoi (current_score->get_label ());
	Iscore += nMatch;
	Glib::ustring Sscore = std::to_string (Iscore) + " pts";
	if (Iscore >= 0)
		current_score->set_label (Sscore);
	else
	{
		display_error (3);
		return 0;
	}
	return 1;
}

void
Game::refresh_selection_area ()
{
	reset_selection_area ();
	set_main_hexa (std::to_string (valHexa1));
	set_second_hexa (std::to_string (valHexa2));
}

void
Game::reset_board ()
{
	for (int key : BoardKeys)
	{
		Board[key]->set_label ("");
		update_hexa_color (key);
	}
}

void
Game::reset_selection_area ()
{
	for (int key : SelectionAreaKeys)
	{
		SelectionArea[key]->set_label ("");
		Glib::PropertyProxy<Gtk::ReliefStyle> prop =
				SelectionArea[key]->property_relief ();
		prop.set_value (Gtk::RELIEF_NONE);
		SelectionArea[key]->set_sensitive (false);
		update_hexa_color (key);
	}
}

void
Game::reset_snapshot ()
{
	for (int i : BoardKeys)
	{
		Board[i]->set_label (BoardSnapshotMap[i]);
		update_hexa_color (i);
	}
	set_pos2 (9);
	valHexa1 = std::stoi (oldValHexa1);
	if (oldValHexa2 != "nosecondhexa")
		valHexa2 = std::stoi (oldValHexa2);
	else
		pos2 = -42;
	reset_selection_area ();
	refresh_selection_area ();
	button_cancel->set_sensitive(false);
}

void
Game::rotate_hexas_left ()
{
	int actualPos2 = get_pos2 ();
	if (actualPos2 == -10)
		set_pos2 (-9);
	if (actualPos2 == -9)
		set_pos2 (1);
	if (actualPos2 == 1)
		set_pos2 (10);
	if (actualPos2 == 10)
		set_pos2 (9);
	if (actualPos2 == 9)
		set_pos2 (-1);
	if (actualPos2 == -1)
		set_pos2 (-10);
	reset_selection_area ();
	refresh_selection_area ();
}

bool
Game::rotate_hexas_right (GdkEventButton* event)
{
	int actualPos2 = get_pos2 ();
	if (event->button == 3)
	{
		if (actualPos2 == -10)
			set_pos2 (-1);
		if (actualPos2 == -9)
			set_pos2 (-10);
		if (actualPos2 == 1)
			set_pos2 (-9);
		if (actualPos2 == 10)
			set_pos2 (1);
		if (actualPos2 == 9)
			set_pos2 (10);
		if (actualPos2 == -1)
			set_pos2 (9);
	}
	reset_selection_area ();
	refresh_selection_area ();
	return true;
}

bool
Game::scan_neighbours (int ref, std::list<int> &oldRefs)
{
	// Scan every neighbours around ref hexa
	for (int key : SelectionAreaKeys)
	{
		// Skip the central position
		if (key == 0)
			continue;
		int Nb = get_neighbour (key, ref);
		// Check if we are not out of the board
		if (Nb == -42)
			continue;
		// Check if we don't match the previous hexa
		bool skip = false;
		for (int i : oldRefs)
			if (Nb == i)
				skip = true;
		if (skip)
			continue;
		// If this matching hexa has the same value as the current one
		if (Board[ref]->get_label () != "" &&
				Board[Nb]->get_label () == Board[ref]->get_label ())
		{
			nMatch++;
			oldRefs.push_back (Nb);
			scan_neighbours (Nb, oldRefs);
		}
	}
	return false;
}

void
Game::set_main_hexa (Glib::ustring Snum)
{
	mainM->set_label (Snum);
	Glib::PropertyProxy<Gtk::ReliefStyle> prop =
			mainM->property_relief ();
	prop.set_value (Gtk::RELIEF_NORMAL);
	mainM->set_sensitive (true);
	update_hexa_color (0);
}

void
Game::set_pos2 (int givenPos2)
{
	pos2 = givenPos2;
}

void
Game::set_second_hexa (Glib::ustring Snum)
{
	if (pos2 != -42)
	{
		SelectionArea[pos2]->set_label (Snum);
		update_hexa_color (pos2);
		Glib::PropertyProxy<Gtk::ReliefStyle> prop =
				SelectionArea[pos2]->property_relief ();
		prop.set_value (Gtk::RELIEF_NORMAL);
		SelectionArea[pos2]->set_sensitive (true);
	}
}

void
Game::store_oldvalhexas ()
{
	oldValHexa1 = get_main_hexa ();
	if (get_second_hexa () != "")
		oldValHexa2 = get_second_hexa ();
	else
		oldValHexa2 = "";
}

bool
Game::triggers (int ref)
{
	std::list<int> matchingNbs;
	matchingNbs = {ref};
	scan_neighbours (ref, matchingNbs);
	// In case there are at least two matching neighbors
	if (matchingNbs.size () > 2)
	{
		// Merge them
		merge_hexas (ref, matchingNbs);
		// and update the score
		nMatch = matchingNbs.size () * 10;
		refresh_score ();
		// Then tell to define_hexas() that it is not finished !
		return true;
	}
	// At this point, the game can be finished.
	check_end ();
	// Stop
	return false;
}

void
Game::update_hexa_color (int hexa)
{
	Glib::RefPtr<Gtk::StyleContext> HexaContext;
	int val = -42;
	if (hexa == -42)
		return;
	if (hexa < 13)
	{
		HexaContext = SelectionArea[hexa]->get_style_context ();
		if (SelectionArea[hexa]->get_label () != "")
			val = std::stoi (SelectionArea[hexa]->get_label ());
	}
	else
	{
		HexaContext = Board[hexa]->get_style_context();
		if (Board[hexa]->get_label () != "")
			val = std::stoi (Board[hexa]->get_label ());
	}
	// Customize this hexa
	HexaContext->add_provider (CssProvider,
			GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
	// Clear old class styles
	for (int key : ColorsKeys)
		HexaContext->remove_class (Colors [key]);
	// Assign new color
	if (val != -42)
		HexaContext->add_class (Colors [val]);
	else
		HexaContext->add_class ("nocolor");
}

void
Game::write_scores ()
{
	// Get config file
	Glib::ustring config_file = get_config_file ();
	Glib::KeyFile *saved_scores;
	saved_scores = new Glib::KeyFile ();
	saved_scores->load_from_file (config_file);
	// If the user wants to save his score
	if (make7askname->run () == 10)
	{
		// Get his name
		Glib::ustring username = user_entry->get_text();
		// Get his old score (eventually)
		std::list<int> Lscore;
		if (saved_scores->has_group("Scores"))
		{
			if (saved_scores->has_key("Scores", username))
				Lscore = saved_scores->get_integer_list ("Scores", username);
		}
		// Add current score
		Lscore.push_back (std::stoi (current_score->get_label ()));
		saved_scores->set_integer_list ("Scores", username, Lscore);
		// Save it
		saved_scores->save_to_file (config_file);
		make7scores->show ();
	}
	make7askname->hide ();
}
