/************************************************************************\
 * Magic Square solves magic squares.                                   *
 * Copyright (C) 2019  Asher Gordon <AsDaGo@posteo.net>                 *
 *                                                                      *
 * This file is part of Magic Square.                                   *
 *                                                                      *
 * Magic Square is free software: you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation, either version 3 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * Magic Square is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with Magic Square.  If not, see                                *
 * <https://www.gnu.org/licenses/>.                                     *
\************************************************************************/

/* debug.c -- print debugging message when we receive a signal that
   indicates a bug */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/resource.h>

#ifdef HAVE_BACKTRACE
# ifndef backtrace_size_t
/* Default to int */
#  define backtrace_size_t int
# endif /* backtrace_size_t */

# ifdef HAVE_EXECINFO_H
#  include <execinfo.h>
# else /* !HAVE_EXECINFO_H */
backtrace_size_t backtrace(void **, backtrace_size_t);
char **backtrace_symbols(void *const *, backtrace_size_t);
void backtrace_symbols_fd(void *const *, backtrace_size_t, int);
# endif /* !HAVE_EXECINFO_H */

/* The maximum backtrace size to print */
# define BACKTRACE_SIZE 100
#endif /* HAVE_BACKTRACE */

#include "debug.h"
#include "sig2str.h"

/* Read until newline or EOF */
#define flush_stdin()				\
  while (!(c == EOF || c == '\n'))		\
    c = getchar()

/* Which signals to print a debug message on. Note that if SIGSYS is
   not defined, the list will end with a trailing comma (assuming at
   least one of the other signals is defined). This has been legal in
   standard C for quite some time. */
const int debug_signals[] =
  {
#ifdef SIGABRT
   SIGABRT,
#endif
#ifdef SIGBUS
   SIGBUS,
#endif
#ifdef SIGFPE
   SIGFPE,
#endif
#ifdef SIGILL
   SIGILL,
#endif
#ifdef SIGSEGV
   SIGSEGV,
#endif
#ifdef SIGSTKFLT
   SIGSTKFLT,
#endif
#ifdef SIGSYS
  SIGSYS
#endif
  };

const size_t debug_signals_entries =
  sizeof(debug_signals) / sizeof(*debug_signals);

/* Static helper functions */
int dump_core(void);

/* Print a debuging message and exit() */
void __attribute__((__noreturn__)) debug_abort (int signum) {
  char *signame;
  struct sigaction action;
#ifdef HAVE_BACKTRACE
  void *bt_buf[BACKTRACE_SIZE];
  char **bt_str;
  backtrace_size_t bt_size;
#endif

  /* Reset SIGABRT to default (core dump) so that we have a way to
     dump core. Do it here rather than in dump_core() so that if we
     recieve SIGABRT again, we dump core right away, no questions
     asked. */
  action.sa_handler = SIG_DFL;
  sigemptyset(&(action.sa_mask));
  action.sa_flags = 0;

  if (sigaction(SIGABRT, &action, NULL))
    exit(-signum);

  if ((signame = sig2str(signum))) {
    fprintf(stderr, "Terminated by SIG%s (%d) %s\n\n",
	    signame, signum, strsignal(signum));
  }
  else {
    fprintf(stderr, "Terminated by unknown signal (%d) %s\n\n",
	    signum, strsignal(signum));
  }

  fputs("You've found a bug in " PACKAGE_NAME "!\n"
	"Please report it to <" PACKAGE_BUGREPORT ">.\n\n",
	stderr);

#ifdef HAVE_BACKTRACE
  /* Get the backtrace */
  bt_size = backtrace(bt_buf, BACKTRACE_SIZE);
  bt_str = backtrace_symbols(bt_buf, bt_size);

  if (!bt_str) {
    fprintf(stderr,
	    "Could not allocate backtrace symbols: %m\n"
	    "This backtrace may not look as nice.\n");
  }

  fprintf(stderr, "Backtrace%s:\n",
	  (bt_size < BACKTRACE_SIZE) ? "" : " (may be truncated)");

  if (bt_str) {
    /* Print the backtrace */
    for (char **cur_str = bt_str; bt_size; bt_size--, cur_str++) {
      /* This needs to hold at most as much as the whole backtrace,
	 and since the whole backtrace can't exceed the size of a
	 `backtrace_size_t', use that. */
      backtrace_size_t skipped = 0;

      fprintf(stderr, "#%d:\t%s\n", bt_size, *cur_str);

      /* Skip duplicate entries, but only if there is more than one
	 duplicate (this is useful since we have recursion). */
      if (bt_size > 2 && !strcmp(cur_str[1], cur_str[2])) {
	while (bt_size > 1 && !strcmp(cur_str[0], cur_str[1])) {
	  skipped++;

	  bt_size--;
	  cur_str++;
	}
      }

      /* Print how many we skipped if any */
      if (skipped)
	fprintf(stderr, "[repeats %d more times]\n", skipped);
    }

    free(bt_str);
  }
  else {
    /* Fall back to backtrace_symbols_fd() */
    backtrace_symbols_fd(bt_buf, bt_size, STDERR_FILENO);
  }
#endif /* HAVE_BACKTRACE */

  putc('\n', stderr);

  if (dump_core())
    fprintf(stderr, "Unable to generate core dump: %m\n");

  exit(-signum);
}

/***************************\
|* Static helper functions *|
\***************************/

/* Dump core. Does not return if core dump is successfully, returns
   nonzero on error (and errno will be set), and returns zero if
   coredump was user-aborted. */
int dump_core(void) {
  struct rlimit corelim;
  int answer, c;

  /* Ask if we should generate a core dump */
  fputs("Dump core? ", stderr);

  c = answer = getchar();
  flush_stdin();

  if (tolower(answer) != 'y')
    return 0;

  /* Get the core size limit */
  if (getrlimit(RLIMIT_CORE, &corelim)) {
    fprintf(stderr, "Unable to get core size limit: %m\n"
	    "Core dump may fail.\n");
  }
  else {
    char set_corelim = 0; /* Whether to set the core size limit */

    if (!corelim.rlim_cur) {
      if (!corelim.rlim_max) {
	/* Both hard and soft limits are zero */

	fputs("Core dumps are hard disabled. Try to enable anyway? ", stderr);

	c = answer = getchar();
	flush_stdin();

	if (tolower(answer) != 'y')
	  return 0;

	corelim.rlim_cur = corelim.rlim_max = RLIM_INFINITY;
	set_corelim = 1;
      }
      else {
	/* Soft limit is zero but hard limit is not */

	fputs("Core dumps are disabled. Enable? ", stderr);

	c = answer = getchar();
	flush_stdin();

	if (tolower(answer) != 'y')
	  return 0;

	if (corelim.rlim_max != RLIM_INFINITY) {
	  /* Hard limit is not zero but is not unlimited */

	  fprintf(stderr, "Core dump size is hard limited to %ldB. "
		  "Try to unlimit? ", corelim.rlim_max);

	  c = answer = getchar();
	  flush_stdin();

	  if (tolower(answer) == 'y') {
	    rlim_t old_max = corelim.rlim_max;

	    corelim.rlim_cur = corelim.rlim_max = RLIM_INFINITY;
	    if (setrlimit(RLIMIT_CORE, &corelim)) {
	      fprintf(stderr, "Unable to unlimit core size: %m\n"
		      "Falling back to hard limit: %ldB\n", old_max);

	      corelim.rlim_cur = corelim.rlim_max = old_max;
	      set_corelim = 1;
	    }
	  }
	  else {
	    corelim.rlim_cur = corelim.rlim_max;
	    set_corelim = 1;
	  }
	}
	else {
	  /* No hard limit */
	  corelim.rlim_cur = corelim.rlim_max;
	  set_corelim = 1;
	}
      }
    }
    else {
      /* Soft limit is not zero */

      if (corelim.rlim_cur != RLIM_INFINITY) {
	/* But it's not unlimited either */

	fprintf(stderr,
		"Core dump size is limited to %ldB. Unlimit? ",
		corelim.rlim_cur);

	c = answer = getchar();
	flush_stdin();

	if (tolower(answer) == 'y') {
	  if (corelim.rlim_max != RLIM_INFINITY) {
	    /* It is hard limited */

	    fputs("It is hard limited. Try to unlimit anyway? ", stderr);

	    c = answer = getchar();
	    flush_stdin();

	    if (tolower(answer) == 'y') {
	      rlim_t old_max = corelim.rlim_max;

	      corelim.rlim_cur = corelim.rlim_max = RLIM_INFINITY;
	      if (setrlimit(RLIMIT_CORE, &corelim)) {
		fprintf(stderr, "Unable to unlimit core size: %m\n"
			"Falling back to hard limit: %ldB\n", old_max);

		corelim.rlim_cur = corelim.rlim_max = old_max;
		set_corelim = 1;
	      }
	    }
	  }
	  else {
	    /* It's not hard limited */
	    corelim.rlim_cur = corelim.rlim_max;
	    set_corelim = 1;
	  }
	}
      }
    }

    if (set_corelim && setrlimit(RLIMIT_CORE, &corelim)) {
      fprintf(stderr, "Unable to set core size limit to %ldB "
	      "(hard limit %ldB): %m\n",
	      corelim.rlim_cur, corelim.rlim_max);

      if (!getrlimit(RLIMIT_CORE, &corelim)) {
	fprintf(stderr, "Soft limit is %ldB and hard limit is %ldB\n",
		corelim.rlim_cur, corelim.rlim_max);
      }
    }
  }

  fprintf(stderr, "\nDumping core.\n\n"
	  "Please note that core dumps may fail under certain conditions. "
	  "The core dump\n"
	  "will probably be called \"core\" or \"core.%d\". See the core(5) "
	  "manpage for more\n"
	  "info.\n", getpid());

  /* Finally dump core (or try to at least) */
  if (raise(SIGABRT))
    return 1;

  /* If we're still here, we failed to dump core */
  return -1;
}
