/************************************************************************\
 * Magic Square solves magic squares.                                   *
 * Copyright (C) 2019  Asher Gordon <AsDaGo@posteo.net>                 *
 *                                                                      *
 * This file is part of Magic Square.                                   *
 *                                                                      *
 * Magic Square is free software: you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation, either version 3 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * Magic Square is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with Magic Square.  If not, see                                *
 * <https://www.gnu.org/licenses/>.                                     *
\************************************************************************/

/* parse.c -- parse human-readable and machine-readable magic square
   files */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <endian.h>
#include <math.h>

#ifndef assert
# include <assert.h>
#endif

#include "square.h"
#include "parse.h"
#include "input.h"

/* Static helper functions */
static int parse_machine_cellval(cellval_t *, FILE *);
static int parse_machine_cell(cell_t *, FILE *);

static clist_t ** parse_human_nodesc(square_t *, FILE *);
static int parse_human_line(square_t *, FILE *, clist_t **, char);
static int parse_human_separator_cell(FILE *, size_t, clist_t **);
static int parse_human_cells(cell_t **, clist_t **, size_t, size_t);
static int parse_human_other_nums(square_t *, FILE *, clist_t **);
static cellval_t parse_human_cellval(char *, char **);

/* Parse a machine readable binary format, returns 0 on success,
   nonzero on error */
int parse_machine(square_t *square, FILE *file) {
  const uint32_t magic_number = MAGIC_NUMBER;
  uint32_t magic_number_check;
  int file_version;
  size_t description_size;

  /* Read the magic number */
  if (!fread(&magic_number_check, sizeof(magic_number_check), 1, file))
    return 1;

  /* Convert it from big endian to host byte order and compare it to
     the actual magic number */
  magic_number_check = be32toh(magic_number_check);
  if (magic_number_check != magic_number) {
    errno = ENOMSG;
    return 1;
  }

  /* Make sure we have the right file version number */
  if ((file_version = getc(file)) == EOF)
    return 1;

  if (file_version != FILE_VERSION) {
    errno = ENOMSG;
    return 1;
  }

  /* Get the sizes of stuff */
  if (!(fread(&(square->size), sizeof(square->size), 1, file) &&
	fread(&(square->nums_size), sizeof(square->nums_size), 1, file) &&
	fread(&description_size, sizeof(description_size),
	      1, file))) {
    return 1;
  }

  /* Convert them to host byte order */
  square->size		= be64toh(square->size);
  square->nums_size	= be64toh(square->nums_size);
  description_size	= be64toh(description_size);

  /* Allocate the cells */
  square->cells = malloc(sizeof(*(square->cells)) * square->size);

  /* Get the cells */
  for (size_t x = 0; x < square->size; x++) {
    square->cells[x] = malloc(sizeof(*(square->cells[x])) * square->size);

    for (size_t y = 0; y < square->size; y++) {
      int ret = parse_machine_cell(&(square->cells[x][y]), file);

      if (ret) {
	/* Free the cells */
	for (size_t i = 0; i <= x; i++)
	  free(square->cells[i]);
	free(square->cells);

	/* Now return an error (`errno' will already have been set, if
	   applicable) */
	return ret;
      }
    }
  }

  if (square->nums_size) {
    /* Get the other valid numbers */
    square->nums = malloc(sizeof(*(square->nums)) * square->nums_size);

    for (size_t i = 0; i < square->nums_size; i++) {
      int ret = parse_machine_cellval(&(square->nums[i]), file);

      if (ret) {
	free(square->nums);

	for (size_t j = 0; j < square->size; j++)
	  free(square->cells[j]);
	free(square->cells);

	return ret;
      }
    }
  }

  if (description_size) {
    /* And finally get the description */
    square->description = malloc(sizeof(*(square->description)) *
				 (description_size + 1));

    if (fread(square->description, sizeof(*(square->description)),
	      description_size, file) != description_size) {
      free(square->description);
      free(square->nums);

      for (size_t i = 0; i < square->size; i++)
	free(square->cells[i]);
      free(square->cells);

      return 1;
    }

    /* And the '\0' */
    square->description[description_size] = '\0';
  }
  else {
    /* There was no description */
    square->description = NULL;
  }

  return 0;
}

/* Parse a human readable format, returns 0 on success, nonzero on
   error */
int parse_human(square_t *square, FILE *file) {
  clist_t *description = malloc(sizeof(*description));
  size_t description_size; /* The size of the description (including
			      the '\0') */

  description->prev = NULL; /* So we know where the beginning is */

  /* Parse the description and the rest of the square */
  while (1) {
    /* This could be the end of the description. We'll start parsing
       and consider it not the end if we fail. */

    /* What to append to the description if this turns out not to be
       the end; `description_append' is the first element of the
       list and `description_append + 1' is the last */
    clist_t **description_append;

    errno = 0;
    description_append = parse_human_nodesc(square, file);

    if (description_append) {
      int c;

      /* We're not done yet; keep parsing, but first add the
	 characters to the description */

      /* Since the last char of `description' will be '\0' */
      if (description->prev) {
	description = description->prev;
	free(description->next);

	description_append[0]->prev = description;
	description->next = description_append[0];
      }
      else {
	free(description);
      }

      /* Now go to the end */
      description = description_append[1];

      /* Note that we are only free()ing the ARRAY (the clist_t **),
	 not the actual LISTS (the clist_t *) */
      free(description_append);

      /* Squares can only start at the beginning of a line, so read to
	 the end of this line. */
      c = description->prev->c;

      while (c != '\n') {
	c = getinput(file, &description);

	if (c == EOF) {
	  errno = ENOMSG;
	  goto error;
	}
      }
    }
    else if (errno) {
      /* There was an error */
      goto error;
    }
    else {
      /* We're done! */
      break;
    }
  }

  /* Remove trailing whitespace */
  while (description->prev && isspace(description->prev->c)) {
    description = description->prev;
    free(description->next);
  }

  /* Set `next' to NULL so we know where the end is */
  description->next = NULL;

  /* Now convert the linked list to a string */

  /* First get the size while also rewinding to the beginning of the
     list */
  for (description_size = 0; description->prev; description_size++) {
    if (description_size == SIZE_MAX - 1 /* -1 since we will be using
                                             `description_size + 1' */) {
      errno = EOVERFLOW;

      while (description->next)
	description = description->next;

      goto error;
    }

    description = description->prev;
  }

  if (description_size) {
    /* Now allocate the string */
    square->description =
      malloc(sizeof(*(square->description)) * (description_size + 1));

    /* Now copy the list to the string */
    for (size_t i = 0; i < description_size; i++) {
      square->description[i] = description->c;
      description = description->next;

      /* We don't need this anymore */
      free(description->prev);
    }
    assert(!description->next);
    /* Don't forget the '\0' */
    square->description[description_size] = '\0';
  }
  else {
    /* No description */
    square->description = NULL;
  }

  free(description);

  /* The rest of `square' will already be set by
     parse_human_nodesc() */

  return 0;

 error:
  /* Free the description */
  while (description->prev) {
    description = description->prev;
    free(description->next);
  }
  free(description);

  return 1;
}

/***************************\
|* Static helper functions *|
\***************************/

/* Parse a `cellval_t' from a machine readable binary file, returning
   0 on success and nonzero on error. */
static int parse_machine_cellval(cellval_t *cellval, FILE *file) {
  int type;

  /* Get the type */
  if ((type = getc(file)) == EOF)
    return 1;

  cellval->type = type;

  switch (cellval->type) {
  case INT:
    if (!fread(&(cellval->i), sizeof(cellval->i), 1, file))
      return 1;

    /* Convert to host byte order */
    cellval->i = be64toh(cellval->i);

    break;
  case FLOAT:
    if (!fread(&(cellval->f), sizeof(cellval->f), 1, file))
      return 1;

    /* TODO: There is no handling for byte order or other
       portability issues for floating point numbers; fix it. */

    break;
  default:
    errno = ENOMSG;
    return 1;
  }

  return 0;
}

/* Parse a cell from a machine readable binary file, returning 0 on
   success and nonzero on error. */
static int parse_machine_cell(cell_t *cell, FILE *file) {
  int tmp;

  /* Parse the value of the cell */
  if ((tmp = parse_machine_cellval(&(cell->val), file)))
    return tmp;

  /* Get the mutability */
  if ((tmp = getc(file)) == EOF)
    return 1;

  cell->mutable = tmp;

  return 0;
}

/* Parse a magic square assuming the description was already parsed,
   returning NULL if successful or we've reached EOF, otherwise
   returning two pointers to a linked list of characters that were
   read, the first pointer at the first position and the second
   pointer at the last. */
static clist_t ** parse_human_nodesc(square_t *square, FILE *file) {
  char *prespace = NULL; /* Whitespace prepended before each line */
  clist_t /* What to return if we fail */
    **return_lists = malloc(sizeof(*return_lists) * 2),
    *return_list_pos; /* A certain position in `return_lists' */
  size_t prespace_size = 1; /* 1 for the '\0' */
  int c, ret;

  return_lists[0] = malloc(sizeof(*(return_lists[0])));

  /* Remember the first position */
  return_lists[0]->prev = NULL; /* So we know where the beginning is */
  return_lists[1] = return_lists[0];

  /* Allow each line to be prepended with spaces and tabs */
  do {
    c = getinput(file, return_lists + 1);
  } while (strchr(" \t", c));

  if (c == EOF) {
    /* EOF to soon! */
    errno = ENOMSG;
    goto error;
  }

  /* Unget `c' so parse_human_line() can parse it */
  ungetinput(c, file, return_lists + 1);

  return_lists[1]->next = NULL; /* So we know where the end is */

  for (return_list_pos = return_lists[1];
       return_list_pos->prev; prespace_size++) {
    if (prespace_size == SIZE_MAX) {
      errno = EOVERFLOW;
      goto error;
    }

    return_list_pos = return_list_pos->prev;
  }

  assert(return_list_pos == return_lists[0]);

  /* Allocate the string */
  prespace = malloc(sizeof(*prespace) * prespace_size);

  /* Copy the list to the string */
  for (size_t i = 0; i < prespace_size - 1; i++) {
    prespace[i] = return_list_pos->c;
    return_list_pos = return_list_pos->next;
  }
  assert(!return_list_pos->next);
  /* Don't forget the '\0' */
  prespace[prespace_size - 1] = '\0';

  /* Parse the line */
  ret = parse_human_line(square, file, return_lists + 1, 0);

  if (ret > 1)
    /* Parse error */
    goto parse_error;

  if (ret < 0)
    /* Some other error; `errno' should be set */
    goto error;

  /* We shouldn't be done parsing on the first line */
  assert(!ret);

  /* Read whitespace up to a newline */
  while ((c = getinput(file, return_lists + 1)) != '\n') {
    if (c == EOF) {
      /* EOF too soon */
      errno = ENOMSG;
      goto error;
    }

    if (!isspace(c))
      /* Invalid char */
      goto parse_error;
  }

  /* Now parse lines until we're done */
  do {
    /* Skip the prepended whitespace */
    for (size_t i = 0; i < prespace_size - 1; i++) {
      c = getinput(file, return_lists + 1);

      if (c == EOF) {
	errno = ENOMSG;
	goto error;
      }

      if (c != prespace[i])
	goto parse_error;
    }

    /* Parse the line */
    ret = parse_human_line(square, file, return_lists + 1, 0);

    if (!ret) {
      /* Read whitespace up to a newline */
      while ((c = getinput(file, return_lists + 1)) != '\n') {
	if (c == EOF) {
	  /* EOF too soon */
	  errno = ENOMSG;
	  goto error;
	}

	if (!isspace(c))
	  /* Invalid char */
	  goto parse_error;
      }
    }
  } while (!ret);

  if (ret > 1)
    /* Parse error */
    goto parse_error;

  if (ret < 0)
    /* Some other error; `errno' should be set */
    goto error;

  /* Now parse the line which has the rest of the valid numbers not
     already in the square */
  ret = parse_human_other_nums(square, file, return_lists + 1);

  if (ret) {
    /* Free the cells in `square' */
    for (size_t i = 0; i < square->size; i++)
      free(square->cells[i]);
    free(square->cells);

    if (ret > 0)
      /* Parse error */
      goto parse_error;

    if (ret < 0)
      /* Some other error; `errno' should be set */
      goto error;
  }

  /* We're done parsing! */
  goto success;

 parse_error:
  if (prespace)
    free(prespace);

  /* Reset parse_human_line()'s internal variables */
  parse_human_line(NULL, NULL, NULL, 1);

  return return_lists;

 error:
  /* `errno' is assumed to be set (otherwise it will look like we
     succeeded in parsing) */
  assert(errno);

 success:
  if (prespace)
    free(prespace);

  /* Free the return lists */
  while (return_lists[1]->prev) {
    return_lists[1] = return_lists[1]->prev;
    free(return_lists[1]->next);
  }

  assert(return_lists[1] == return_lists[0]);

  free(return_lists[1]);
  free(return_lists);

  /* Reset parse_human_line()'s internal variables */
  parse_human_line(NULL, NULL, NULL, 1);

  return NULL;
}

/* Parse a single line of a file; `*parsed_chars' will be the
   characters read from the file. If `reset', reset all internal
   variables. Returns 0 on success, 1 if done parsing, > 1 on parse
   error, and negative (with `errno' set) on some other error. */
static int parse_human_line(square_t *square, FILE *file,
			    clist_t **parsed_chars, char reset) {
  /* Internal static variables */
  static size_t
    cell_width  = 0, cell_height = 0,
    square_size = 0;
  static size_t
    cell_row   = 0, /* The cell row we are currently parsing */
    square_row = 0; /* The row on the square we are currently parsing */
  static char separator = 1; /* Whether the next line will be a separator */
  static clist_t **cell_strings = NULL; /* The strings in the cells
					   for the row we are
					   currently parsing */
  static cell_t **cells = NULL; /* The known cells */

  int c, ret;

  if (reset) {
    /* Reset our internal static variables */
    if (cell_strings) {
      for (size_t i = 0; i < square_size; i++) {
	while (cell_strings[i]->prev) {
	  cell_strings[i] = cell_strings[i]->prev;
	  free(cell_strings[i]->next);
	}

	free(cell_strings[i]);
      }

      free(cell_strings);
      cell_strings = NULL;
    }

    if (cells) {
      for (size_t i = 0; i < square_size; i++)
	free(cells[i]);

      free(cells);
      cells = NULL;
    }

    cell_width  = 0; cell_height = 0;
    square_size = 0;

    cell_row   = 0;
    square_row = 0;

    separator = 1;

    return 0;
  }

  /* If we know the square size, we should know the cell width
     too. And if we don't know the square size, then we don't know the
     cell width either (yet). */
  assert(!square_size == !cell_width);

  /* Get the first input chararter */
  c = getinput(file, parsed_chars);

  if (c == EOF) {
    /* EOF too soon */
    errno = ENOMSG;
    return -1;
  }

  if (!separator && !cell_height && c == JOINT) {
    /* Well, well, well! Whad'ya know? It should have been a separator
       after all! */
    separator = 1;

    /* And now we know the cell_height */
    cell_height = cell_row;

    if (!cell_height) {
      /* We can't have 0 cell height */
      ungetinput(c, file, parsed_chars); /* So we can parse it again */
      return 2;
    }

    /* We should also parse the cells we got */

    /* Allocate `cells' if it's not already */
    if (!cells) {
      cells = malloc(sizeof(*cells) * square_size);

      for (size_t i = 0; i < square_size; i++)
	cells[i] = malloc(sizeof(*(cells[i])) * square_size);
    }

    /* Now parse the cells */
    ret = parse_human_cells(cells, cell_strings, square_size, square_row);

    if (ret)
      return ret;

    /* And we're on the next row */
    square_row++;
  }

  if (separator) {
    /* The first character should be a joint */
    if (c != JOINT)
      return 2;

    if (square_size) {
      /* Reset `cell_row' since we're starting a new row of cells */
      cell_row = 0;

      for (size_t parsed_cells = 0;
	   parsed_cells < square_size; parsed_cells++) {
	ret = parse_human_separator_cell(file, cell_width, parsed_chars);

	if (ret) {
	  if (ret == 1) {
	    if (parsed_cells < square_size -1)
	      /* We shouldn't be done yet */
	      return 2;
	  }
	  else {
	    /* Some other error */
	    return ret;
	  }
	}
      }
    }
    else {
      /* We don't know the square size, and we don't know the cell
	 width yet either */

      /* First find the cell width */
      while ((c = getinput(file, parsed_chars)) == HLINE)
	cell_width++;

      if (c == EOF) {
	/* EOF too soon */
	errno = ENOMSG;
	return -1;
      }

      if (!cell_width)
	/* Cannot have 0 cell width */
	return 2;

      /* The terminating character should be a joint */
      if (c != JOINT)
	return 2;

      square_size = 1;

      /* Check if we're not already done with the line */
      if (!isspace(peekinput(file))) {
	/* Find the square size by parsing the rest of the line */
	while (!(ret =
		 parse_human_separator_cell(file, cell_width, parsed_chars))) {
	  square_size++;
	}

	if (ret != 1)
	  return ret;

	/* Add one for the last cell */
	square_size++;
      }

      /* Now that we know the square size, we can allocate
	 `cell_strings' */
      cell_strings = malloc(sizeof(*cell_strings) * square_size);

      for (size_t i = 0; i < square_size; i++) {
	cell_strings[i] = malloc(sizeof(*(cell_strings[i])));

	/* So we know where the beginning is */
	cell_strings[i]->prev = NULL;
      }
    }

    if (square_row == square_size) {
      /* We're done! */

      /* Commit the stuff we parsed to `square' */
      square->size  = square_size;
      square->cells = cells;

      /* So reset doesn't try to free `cells' (since it is now in
	 `square') */
      cells = NULL;

      return 1;
    }

    /* The next line won't be a separator since this one was */
    separator = 0;
  }
  else {
    /* If it's not a separator, we should know the cell width and
       square size (although not necessarily the cell height) */
    assert(cell_width && square_size);

    /* The first character should be a vertical line */
    if (c != VLINE)
      return 2;

    /* Now parse each of the cells */
    for (size_t i = 0; i < square_size; i++) {
      /* Parse the cell */
      for (size_t j = 0; j < cell_width; j++) {
	c = getinput(file, parsed_chars);

	if (c == EOF) {
	  /* EOF too soon */
	  errno = ENOMSG;
	  return -1;
	}

	if (c != ' ') {
	  /* Add it to the characters for this cell */
	  cell_strings[i]->c = c;
	  cell_strings[i]->next = malloc(sizeof(*(cell_strings[i]->next)));
	  cell_strings[i]->next->prev = cell_strings[i];
	  cell_strings[i] = cell_strings[i]->next;
	}
      }

      /* The terminating character should be a vertical line */
      c = getinput(file, parsed_chars);

      if (c == EOF) {
	/* EOF too soon */
	errno = ENOMSG;
	return -1;
      }

      if (c != VLINE)
	return 2;
    }

    /* Increment the cell row counter and check if it's equal to the
       cell height */
    if (++cell_row == cell_height) {
      /* Parse the strings into integers */
      ret = parse_human_cells(cells, cell_strings, square_size, square_row);

      if (ret)
	return ret;

      /* The next row should be a separator */
      separator = 1;

      /* Increment the row counter for the row on the square we are
	 currently parsing */
      square_row++;
    }
  }

  return 0;
}

/* Parse the separator part of a cell, for example: "---+" */
static int parse_human_separator_cell(FILE *file, size_t cell_width,
				      clist_t **parsed_chars) {
  int c;

  /* Parse the first part of the cell */
  for (size_t parsed_hlines = 0;
       parsed_hlines < cell_width; parsed_hlines++) {
    c = getinput(file, parsed_chars);

    if (c == EOF) {
      /* EOF too soon */
      errno = ENOMSG;
      return -1;
    }

    if (c != HLINE)
      return 2;
  }

  /* And the terminating joint */
  c = getinput(file, parsed_chars);

  if (c == EOF) {
    /* EOF too soon */
    errno = ENOMSG;
    return -1;
  }

  if (c != JOINT)
    return 2;

  return isspace(peekinput(file)) ?
    1 :	/* We're done parsing */
    0;	/* We're done parsing the cell, but not the line */
}

/* Parse cell strings into integers */
static int parse_human_cells(cell_t **cells, clist_t **cell_strings,
		       size_t square_size, size_t row) {
  for (size_t i = 0; i < square_size; i++) {
    char *string;
    size_t size = 0;

    /* Turn the character list into a conventional string */

    /* So we can check where the end is */
    cell_strings[i]->next = NULL;

    /* First get the size and rewind to the beginning */
    while (cell_strings[i]->prev) {
      cell_strings[i] = cell_strings[i]->prev;
      size++;
    }

    if (size) {
      char *endptr, endchar;
      int errno_save;

      /* Now allocate the string */
      string = malloc(sizeof(*string) * (size + 1 /* 1 for the '\0' */));

      /* Now copy the list to the string */
      for (size_t j = 0; j < size; j++) {
	string[j] = cell_strings[i]->c;
	cell_strings[i] = cell_strings[i]->next;
	free(cell_strings[i]->prev); /* We don't need that anymore */
      }

      assert(!cell_strings[i]->next);

      /* So it will can reset properly */
      cell_strings[i]->prev = NULL;

      /* Don't forget the '\0' */
      string[size] = '\0';

      /* Parse the string */
      errno = 0;
      cells[i][row].val = parse_human_cellval(string, &endptr);

      errno_save = errno;
      endchar = *endptr; /* Because `*endptr' will become unreliable after we free(string) */

      free(string);

      if (errno_save || endchar) {
	/* Error */

	       /*------------*\-<----<----<----<----<----<--\
*/	return (!errno_save * 3) - 1;/*			    |
	Sorry, but I can't resist clever little things like /.
	It's equivalent to `errno_save' ? -1 : 2'. */
      }

      /* The value was specified, so mark it as immutable */
      cells[i][row].mutable = 0;
    }
    else {
      /* The value was not specified, so set it to 0 and mark it
	 as mutable (and default to int) */
      cells[i][row].val.i = 0;
      cells[i][row].val.type = INT;
      cells[i][row].mutable = 1;
    }
  }

  return 0;
}

/* Parse the line which has the other valid numbers; return value > 0
   is parse error, < 0 is other error and 0 is success */
static int parse_human_other_nums(square_t *square, FILE *file,
				  clist_t **parsed_chars) {
  int c;
  size_t max_size;
  cellval_t *raw_nums; /* Before we process them */
  size_t raw_nums_size = 0;

  /* Calculate the size of the final array */
  square->nums_size = 0;
  for (size_t x = 0; x < square->size; x++) {
    for (size_t y = 0; y < square->size; y++) {
      if (square->cells[x][y].mutable)
	square->nums_size++;
    }
  }

  /* Calculate the maximum size */
  max_size = square->size * square->size;

  /* Skip whitespace */
  do {
    c = getinput(file, parsed_chars);
  } while (isspace(c));

  if (c == EOF) {
    if (square->nums_size) {
      errno = ENOMSG;
      return -1;
    }
    else {
      /* It's OK since we didn't really want anything anyway */
      return 0;
    }
  }

  /* The first character should be a '(' */
  if (c != '(') {
    if (square->nums_size) {
      return 1;
    }
    else {
      /* It's OK */
      ungetinput(c, file, parsed_chars);
      return 0;
    }
  }

  /* Allocate the maximum we will need */
  raw_nums = malloc(sizeof(*raw_nums) * max_size);

  /* Now parse the numbers */
  while (raw_nums_size < max_size) {
    /* The beginning of where we start parsing the stuff */
    clist_t *beginning;
    char *string, *endptr, endchar;
    size_t string_size = 0;
    int errno_save;

    /* Skip whitespace */
    do {
      c = getinput(file, parsed_chars);
    } while (isspace(c));

    if (c == EOF) {
      errno = ENOMSG;
      free(raw_nums);
      return -1;
    }

    if (!raw_nums_size && c == ')')
      /* It's empty */
      break;

    beginning = (*parsed_chars)->prev;

    /* Parse until whitespace */
    while (!isspace(c) && !strchr("),", c)) {
      c = getinput(file, parsed_chars);
      string_size++;
    }

    /* Now allocate the string and copy the parsed characters to it */
    string = malloc(sizeof(*string) * (string_size + 1));

    for (size_t i = 0; i < string_size; i++) {
      string[i] = beginning->c;
      beginning = beginning->next;
    }

    assert(beginning == (*parsed_chars)->prev);

    /* Don't forget the '\0' */
    string[string_size] = '\0';

    /* And parse the string */
    errno = 0;
    raw_nums[raw_nums_size++] = parse_human_cellval(string, &endptr);

    errno_save = errno;
    endchar = *endptr;

    free(string);

    if (errno_save || endchar) {
      /* Parse error */
      free(raw_nums);
      return 1;
    }

    if (c == ')')
      /* We're done */
      break;

    /* Skip whitespace */
    while (isspace(c)) {
      c = getinput(file, parsed_chars);
    }

    if (c == EOF) {
      errno = ENOMSG;
      free(raw_nums);
      return -1;
    }

    if (c == ')')
      /* We're done */
      break;

    if (c != ',') {
      /* Parse error */
      free(raw_nums);
      return 1;
    }
  }

  if (raw_nums_size == square->nums_size) {
    if (raw_nums_size) {
      /* No need to process them */
      square->nums = raw_nums;
    }
    else {
      /* We needn't even set them since there aren't any */
      free(raw_nums);
    }
  }
  else if (raw_nums_size == max_size) {
    size_t remove_nums_size, i = 0;
    cellval_t *remove_nums;
    /* We need to process them */

    /* First get the size of the numbers to remove */
    remove_nums_size = max_size - square->nums_size;

    /* Now get the numbers to remove */
    remove_nums = malloc(sizeof(*remove_nums) * remove_nums_size);

    for (size_t x = 0; x < square->size; x++) {
      for (size_t y = 0; y < square->size; y++) {
	if (!square->cells[x][y].mutable)
	  remove_nums[i++] = square->cells[x][y].val;
      }
    }

    assert(i == remove_nums_size);

    /* Remove the numbers */
    for (i = 0; i < remove_nums_size; i++) {
      char found = 0;

      /* Search for the number and remove it */
      for (size_t j = 0; j < raw_nums_size; j++) {
	if (cellval_equal(raw_nums[j], remove_nums[i])) {
	  /* Remove it */
	  raw_nums[j] = raw_nums[--raw_nums_size];
	  found = 1;
	  break;
	}
      }

      if (!found) {
	/* We couldn't find it; that's a parse error */
	free(raw_nums);
	return 1;
      }
    }

    free(remove_nums);

    assert(raw_nums_size == square->nums_size);

    /* If there are any numbers to copy over, allocate the array and
       copy them over. Otherwise, we needn't even allocate it. */
    if (square->nums_size) {
      square->nums = malloc(sizeof(*(square->nums)) * square->nums_size);

      for (i = 0; i < square->nums_size; i++) {
	square->nums[i] = raw_nums[i];
      }
    }

    /* Free the original array */
    free(raw_nums);
  }
  else {
    /* Parse error */
    free(raw_nums);
    return 1;
  }

  return 0;
}

/* Parse a long long or a long double */
static cellval_t parse_human_cellval(char *nptr, char **endptr) {
  cellval_t cellval;
  int errno_save = errno; /* So we can reset it later */

  errno = 0;
  cellval.i = strtoll(nptr, endptr, 0);

  if (errno || **endptr) {
    /* Couldn't parse as an int? Try as a float instead. */
    errno = 0;
    cellval.f = strtold(nptr, endptr);

    if (!isfinite(cellval.f))
      /* If it's infinite or NaN, set errno to erange */
      errno = ERANGE;

    /* Set the type to `FLOAT' */
    cellval.type = FLOAT;
  }
  else {
    /* Set the type to `INT' */
    cellval.type = INT;
  }

  if (!errno)
    /* Reset errno */
    errno = errno_save;

  return cellval;
}
