/*  Tarlz - Archiver with multimember lzip compression
    Copyright (C) 2013-2019 Antonio Diaz Diaz.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#define _FILE_OFFSET_BITS 64

#include <cctype>
#include <cerrno>
#include <climits>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <string>
#include <vector>
#include <pthread.h>
#include <stdint.h>
#include <unistd.h>
#include <lzlib.h>

#include "arg_parser.h"
#include "lzip_index.h"
#include "tarlz.h"


namespace {

bool parse_records( const int infd, Extended & extended,
                    const Tar_header header, Resizable_buffer & rbuf,
                    const bool permissive )
  {
  const long long edsize = parse_octal( header + size_o, size_l );
  const long long bufsize = round_up( edsize );
  if( edsize <= 0 || edsize >= 1LL << 33 || bufsize >= INT_MAX )
    return false;				// overflow or no extended data
  if( !rbuf.resize( bufsize ) ) return false;	// extended records buffer
  return ( readblock( infd, (uint8_t *)rbuf(), bufsize ) == bufsize &&
           extended.parse( rbuf(), edsize, permissive ) );
  }

} // end namespace


bool safe_seek( const int fd, const long long pos )
  {
  if( lseek( fd, pos, SEEK_SET ) == pos ) return true;
  show_error( "Seek error", errno ); return false;
  }


int tail_copy( const char * const archive_namep, const Arg_parser & parser,
               std::vector< char > & name_pending,
               const Lzip_index & lzip_index, const long long istream_pos,
               const int infd, const int outfd, int retval )
  {
  const long long rest = lzip_index.file_size() - istream_pos;
  if( istream_pos > 0 && rest > 0 &&
      ( !safe_seek( infd, istream_pos ) ||
        !copy_file( infd, outfd, rest ) ) )
    { show_file_error( archive_namep, "Error during tail copy." );
      return retval ? retval : 1; }
  const long long ostream_pos = lseek( outfd, 0, SEEK_CUR );
  if( ostream_pos < 0 ) { show_error( "Seek error", errno ); retval = 1; }
  else if( ostream_pos > 0 && ostream_pos < lzip_index.file_size() )
    {
    int result;
    do result = ftruncate( outfd, ostream_pos );
      while( result != 0 && errno == EINTR );
    if( result != 0 )
      {
      show_file_error( archive_namep, "Can't truncate archive", errno );
      if( retval < 1 ) retval = 1;
      }
    }

  if( ( close( outfd ) != 0 || close( infd ) != 0 ) && !retval )
    { show_file_error( archive_namep, "Error closing archive", errno );
      retval = 1; }

  if( retval == 0 ) for( int i = 0; i < parser.arguments(); ++i )
    if( !parser.code( i ) && parser.argument( i ).size() && name_pending[i] )
      {
      show_file_error( parser.argument( i ).c_str(), "Not found in archive." );
      retval = 1;
      }
  return retval;
  }


/* Deleting from a corrupt archive must not worsen the corruption. Stop and
   tail-copy as soon as corruption is found. */
int delete_members( const std::string & archive_name, const Arg_parser & parser,
                    const int filenames, const bool missing_crc,
                    const bool permissive )
  {
  if( !filenames )
    { if( verbosity >= 1 ) show_error( "Nothing to delete." ); return 0; }
  if( archive_name.empty() )
    { show_error( "Deleting from stdin not implemented yet." ); return 1; }
  const char * const archive_namep = archive_name.c_str();
  const int infd = open_instream( archive_name );
  if( infd < 0 ) return 1;
  const int outfd = open_outstream( archive_name, false );
  if( outfd < 0 ) { close( infd ); return 1; }

  // mark member names to be deleted
  std::vector< char > name_pending( parser.arguments(), false );
  for( int i = 0; i < parser.arguments(); ++i )
    if( !parser.code( i ) && parser.argument( i ).size() &&
        !Exclude::excluded( parser.argument( i ).c_str() ) )
      name_pending[i] = true;

  const Lzip_index lzip_index( infd, true, false );	// only regular files
  if( lzip_index.retval() == 0 )			// compressed
    return delete_members_lz( archive_namep, parser, name_pending, lzip_index,
                              filenames, infd, outfd, missing_crc, permissive );
  if( lseek( infd, 0, SEEK_SET ) != 0 )
    { show_file_error( archive_namep, "Archive is not seekable." ); return 1; }
  if( lzip_index.file_size() < 3 * header_size )
    { show_file_error( archive_namep, posix_msg ); return 2; }
  // archive is uncompressed seekable, unless compressed corrupt

  Resizable_buffer rbuf;
  long long istream_pos = 0;		// source of next data move
  long long member_begin = 0;		// first pos of current tar member
  Extended extended;			// metadata from extended records
  int retval = 0;
  bool prev_extended = false;		// prev header was extended
  while( true )				// process one tar header per iteration
    {
    if( !prev_extended && ( member_begin = lseek( infd, 0, SEEK_CUR ) ) < 0 )
      { show_error( "Seek error", errno ); retval = 1; break; }
    Tar_header header;
    const int rd = readblock( infd, header, header_size );
    if( rd == 0 && errno == 0 )			// missing EOF blocks
      { show_file_error( archive_namep, end_msg ); retval = 2; break; }
    if( rd != header_size )
      { show_file_error( archive_namep, "Read error", errno );
        retval = 2; break; }
    if( !verify_ustar_chksum( header ) )
      {
      if( block_is_zero( header, header_size ) )		// EOF
        {
        if( prev_extended && !permissive )
          { show_file_error( archive_namep, fv_msg1 ); retval = 2; }
        break;
        }
      show_file_error( archive_namep, "Corrupt header in archive." );
      retval = 2; break;
      }

    const Typeflag typeflag = (Typeflag)header[typeflag_o];
    if( typeflag == tf_global )
      {
      if( prev_extended && !permissive )
        { show_file_error( archive_namep, fv_msg2 ); retval = 2; break; }
      Extended dummy;		// global headers are parsed and ignored
      if( !parse_records( infd, dummy, header, rbuf, true ) )
        { show_file_error( archive_namep, gblrec_msg ); retval = 2; break; }
      continue;
      }
    if( typeflag == tf_extended )
      {
      if( prev_extended && !permissive )
        { show_file_error( archive_namep, fv_msg3 ); retval = 2; break; }
      if( !parse_records( infd, extended, header, rbuf, permissive ) )
        { show_file_error( archive_namep, extrec_msg ); retval = 2; break; }
      else if( !extended.crc_present() && missing_crc )
        { show_file_error( archive_namep, mcrc_msg ); retval = 2; break; }
      prev_extended = true;
      continue;
      }
    prev_extended = false;

    extended.fill_from_ustar( header );		// copy metadata from header

    {							// skip member
    long long rest = extended.file_size();
    const int rem = rest % header_size;
    if( rem ) rest += header_size - rem;		// padding
    if( lseek( infd, rest, SEEK_CUR ) <= 0 )
      { show_file_error( archive_namep, "Seek error", errno );
        retval = 1; break; }
    }

    if( !check_skip_filename( parser, name_pending, extended.path().c_str(),
                              filenames ) )		// delete tar member
      {
      if( !show_member_name( extended, header, 1, rbuf ) )
        { retval = 1; break; }
      const long long pos = lseek( infd, 0, SEEK_CUR );
      if( pos <= 0 || pos <= member_begin || member_begin < istream_pos )
        { show_file_error( archive_namep, "Seek error", errno );
          retval = 1; break; }
      const long long size = member_begin - istream_pos;
      if( size > 0 )	// move pending data each time a member is deleted
        {
        if( istream_pos == 0 )
          { if( !safe_seek( outfd, size ) ) { retval = 1; break; } }
        else if( !safe_seek( infd, istream_pos ) ||
                 !copy_file( infd, outfd, size ) ||
                 !safe_seek( infd, pos ) ) { retval = 1; break; }
        }
      istream_pos = pos;
      }
    extended.reset();
    }

  return tail_copy( archive_namep, parser, name_pending, lzip_index,
                    istream_pos, infd, outfd, retval );
  }
