/*  Lziprecover - Data recovery tool for lzipped files
    Copyright (C) 2009, 2010, 2011, 2012, 2013 Antonio Diaz Diaz.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#define _FILE_OFFSET_BITS 64

#include <algorithm>
#include <cerrno>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <string>
#include <vector>
#include <stdint.h>
#include <unistd.h>
#include <sys/stat.h>

#include "lzip.h"
#include "decoder.h"
#include "file_index.h"


namespace {

// Returns the number of chars read, or 0 if error.
//
int parse_long_long( const char * const ptr, long long & value )
  {
  char * tail;
  errno = 0;
  value = strtoll( ptr, &tail, 0 );
  if( tail == ptr || errno || value < 0 ) return 0;
  int c = tail - ptr;

  if( ptr[c] )
    {
    const int factor = ( ptr[c+1] == 'i' ) ? 1024 : 1000;
    int exponent = 0;
    switch( ptr[c] )
      {
      case 'Y': exponent = 8; break;
      case 'Z': exponent = 7; break;
      case 'E': exponent = 6; break;
      case 'P': exponent = 5; break;
      case 'T': exponent = 4; break;
      case 'G': exponent = 3; break;
      case 'M': exponent = 2; break;
      case 'K': if( factor == 1024 ) exponent = 1; else return 0; break;
      case 'k': if( factor == 1000 ) exponent = 1; else return 0; break;
      }
    if( exponent > 0 )
      {
      ++c;
      if( ptr[c] == 'i' ) { ++c; if( value ) format_num( 0, 0, -1 ); }
      if( ptr[c] == 'B' ) ++c;
      for( int i = 0; i < exponent; ++i )
        {
        if( INT64_MAX / factor >= value ) value *= factor;
        else return 0;
        }
      }
    }
  return c;
  }


// Recognized formats: <begin> <begin>-<end> <begin>,<size>
//
void parse_range( const char * const ptr, Block & range )
  {
  long long value = 0;
  int c = parse_long_long( ptr, value );		// pos
  if( c && value >= 0 && value < INT64_MAX &&
      ( ptr[c] == 0 || ptr[c] == ',' || ptr[c] == '-' ) )
    {
    range.pos( value );
    if( ptr[c] == 0 ) { range.size( INT64_MAX - value ); return; }
    const bool issize = ( ptr[c] == ',' );
    c = parse_long_long( ptr + c + 1, value );		// size
    if( c && value > 0 && ( issize || value > range.pos() ) )
      {
      if( !issize ) value -= range.pos();
      if( INT64_MAX - range.pos() >= value ) { range.size( value ); return; }
      }
    }
  show_error( "Bad decompression range.", 0, true );
  std::exit( 1 );
  }


bool safe_seek( const int fd, const long long pos )
  {
  if( lseek( fd, pos, SEEK_SET ) == pos ) return true;
  show_error( "Seek error", errno ); return false;
  }


int decompress_member( const int infd, const int outfd,
                       const Pretty_print & pp,
                       const unsigned long long mpos,
                       const unsigned long long outskip,
                       const unsigned long long outend )
  {
  try {
    Range_decoder rdec( infd );
    File_header header;
    rdec.read_data( header.data, File_header::size );
    if( rdec.finished() )			// End Of File
      { pp( "Error reading member header" ); return 1; }
    if( !header.verify_magic() )
      { pp( "Bad magic number (file not in lzip format)" ); return 2; }
    if( !header.verify_version() )
      {
      if( pp.verbosity() >= 0 )
        { pp();
          std::fprintf( stderr, "Version %d member format not supported.\n",
                        header.version() ); }
      return 2;
      }
    if( header.dictionary_size() < min_dictionary_size ||
        header.dictionary_size() > max_dictionary_size )
      { pp( "Invalid dictionary size in member header" ); return 2; }

    if( pp.verbosity() >= 2 ) { pp(); show_header( header ); }

    LZ_decoder decoder( header, rdec, outfd, outskip, outend );
    const int result = decoder.decode_member( pp );
    if( result != 0 )
      {
      if( pp.verbosity() >= 0 && result <= 2 )
        {
        pp();
        if( result == 2 )
          std::fprintf( stderr, "File ends unexpectedly at pos %llu\n",
                        mpos + rdec.member_position() );
        else
          std::fprintf( stderr, "Decoder error at pos %llu\n",
                        mpos + rdec.member_position() );
        }
      return 2;
      }
    if( pp.verbosity() >= 2 ) std::fprintf( stderr, "done\n" );
    }
  catch( std::bad_alloc )
    {
    pp( "Not enough memory. Find a machine with more memory" );
    return 1;
    }
  catch( Error e ) { pp(); show_error( e.msg, errno ); return 1; }
  return 0;
  }


int list_file( const std::string & input_filename, const Pretty_print & pp )
  {
  struct stat in_stats;
  const int infd = open_instream( input_filename, &in_stats, true, true );
  if( infd < 0 ) return 1;

  File_index file_index( infd );
  close( infd );
  if( file_index.retval() != 0 )
    { show_error( file_index.error().c_str() ); return file_index.retval(); }

  if( pp.verbosity() >= 0 )
    {
    const unsigned long long data_size = file_index.data_end();
    const unsigned long long file_size = file_index.file_end();
    pp( 0, stdout );
    if( data_size > 0 && file_size > 0 )
      std::printf( "%6.3f:1, %6.3f bits/byte, %5.2f%% saved.  ",
                   (double)data_size / file_size,
                   ( 8.0 * file_size ) / data_size,
                   100.0 * ( 1.0 - ( (double)file_size / data_size ) ) );
    std::printf( "decompressed size %9llu, compressed size %8llu.\n",
                  data_size, file_size );

    if( pp.verbosity() >= 1 && file_index.members() > 1 )
      {
      std::printf( "    Total members in file = %d.\n", file_index.members() );
      if( pp.verbosity() >= 2 )
        for( int i = 0; i < file_index.members(); ++i )
          {
          const Block & db = file_index.dblock( i );
          const Block & mb = file_index.mblock( i );
          std::printf( "    Member %3d   data pos %9llu   data size %7llu   "
                       "member pos %9llu   member size %7llu.\n", i + 1,
                       db.pos(), db.size(), mb.pos(), mb.size() );
          }
      }
    }
  return 0;
  }

} // end namespace


int list_files( const std::vector< std::string > & filenames,
                const int verbosity )
  {
  Pretty_print pp( filenames, verbosity );
  int retval = 0;
  for( unsigned i = 0; i < filenames.size(); ++i )
    {
    pp.set_name( filenames[i] );
    const int tmp = list_file( filenames[i], pp );
    if( tmp > retval ) retval = tmp;
    }
  return retval;
  }


int range_decompress( const std::string & input_filename,
                      const std::string & output_filename,
                      const std::string & range_string, const int verbosity,
                      const bool force, const bool ignore, const bool to_stdout )
  {
  Block range( 0, 0 );
  parse_range( range_string.c_str(), range );
  struct stat in_stats;
  const int infd = open_instream( input_filename, &in_stats, true, true );
  if( infd < 0 ) return 1;

  File_index file_index( infd );
  if( file_index.retval() != 0 )
    { show_error( file_index.error().c_str() ); return file_index.retval(); }

  if( range.end() > file_index.data_end() )
    range.size( std::max( 0LL, file_index.data_end() - range.pos() ) );
  if( range.size() <= 0 )
    { if( verbosity >= 1 ) show_error( "Nothing to do." ); return 0; }

  if( verbosity >= 1 )
    {
    if( verbosity >= 2 )
      std::fprintf( stderr, "Decompressed file size = %sB\n",
                    format_num( file_index.data_end() ) );
    std::fprintf( stderr, "Decompressing range %sB", format_num( range.pos() ) );
    std::fprintf( stderr, " to %sB ", format_num( range.pos() + range.size() ) );
    std::fprintf( stderr, "(%sBytes)\n", format_num( range.size() ) );
    }

  int outfd = -1;
  if( to_stdout || !output_filename.size() )
    outfd = STDOUT_FILENO;
  else
    { outfd = open_outstream_rw( output_filename, force );
      if( outfd < 0 ) return 1; }
  Pretty_print pp( input_filename, verbosity );
  int retval = 0;
  for( int i = 0; i < file_index.members(); ++i )
    {
    const Block & db = file_index.dblock( i );
    if( range.overlaps( db ) )
      {
      if( verbosity >= 3 )
        std::fprintf( stderr, "Decompressing member %3d\n", i + 1 );
      const long long outskip = std::max( 0LL, range.pos() - db.pos() );
      const long long outend = std::min( db.size(), range.end() - db.pos() );
      const long long mpos = file_index.mblock( i ).pos();
      if( !safe_seek( infd, mpos ) ) { retval = 1; break; }
      const int tmp = decompress_member( infd, outfd, pp, mpos, outskip, outend );
      if( tmp && ( tmp != 2 || !ignore ) )
        cleanup_and_fail( output_filename, outfd, tmp );
      if( tmp > retval ) retval = tmp;
      pp.reset();
      }
    }
  close( infd );
  if( close( outfd ) != 0 )
    {
    show_error( "Error closing output file", errno );
    cleanup_and_fail( output_filename, -1, 1 );
    }
  if( verbosity >= 2 && retval == 0 )
    std::fprintf( stderr, "Byte range decompressed successfully.\n" );
  return retval;
  }
