/***************************************************************************
                          main.c  -  Lumiere Main Loop
                             -------------------
    begin                : Mon Oct 13 02:08:47 CET 2003
    copyright            : (C) 2003 by Stphane Konstantaropoulos
    email                : stephanek@brutele.be
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/*
 * CVS block
 *
 * $Author: stephanek $
 * $Revision: 1.7 $
 * $Date: 2005/04/13 01:26:42 $
 *
 */
#include <config.h>

#include "lum-automation.h"
#include "LumAutomation.h"

#include <string.h>
#include <X11/Xlib.h>
#include <libbonoboui.h>
#include <libgnome/gnome-program.h>
#include <libgnomeui/gnome-ui-init.h>
#include <libgnomeui/gnome-window-icon.h>
#include <libgnomevfs/gnome-vfs-init.h>

#ifdef ENABLE_NLS
# include <locale.h>
#include <bonobo/bonobo-i18n.h>
#endif

#define LUM_FACTORY_OAFIID "OAFIID:GNOME_Lum_Automation_Factory"

CORBA_Environment corba_env;

extern LumShell *lum_shell;

static gboolean open_in_existing_window = FALSE;
static gboolean open_in_new_window = FALSE;
static gboolean enqueue = FALSE;
static gboolean close_option = FALSE;
static gboolean quit_option = FALSE;
static gboolean server_mode    = FALSE;
static gboolean open_as_nautilus_view = FALSE;
static gchar *session_filename = NULL;

static gboolean xscr_window = FALSE;
static gboolean xscr_root = FALSE;
static gboolean xscr_mono = FALSE;
static gboolean xscr_install = FALSE;
static gboolean xscr_noinstall = FALSE;
static gint xscr_window_id = 0;
static gint xscr_delay = 0;
static gint xscr_visual = 0;

static gint n_uris; 
static gchar **uri;

/* command line argument parsing structure */
static struct poptOption popt_options[] =
{
	{ NULL, '\0', POPT_ARG_INCLUDE_TABLE, &bonobo_activation_popt_options, 0, NULL,
	  NULL },
	{ "new-window", 'w', POPT_ARG_NONE, &open_in_new_window, 0,
	  N_("Open a new window in an existing Lumiere process"),
	  NULL },
	{ "existing", 'x', POPT_ARG_NONE, &open_in_existing_window, 0,
	  N_("Attepmt to load in an existing Lumiere window"),
	  NULL },
	{ "enqueue", 'e', POPT_ARG_NONE, &enqueue, 0,
	  N_("Attempt to enqueue in an existing Lumiere playlist"),
	  NULL },
	{ "nautilus-view", 'v', POPT_ARG_NONE, &open_as_nautilus_view, 0,
	  N_("Used internally by the nautilus view"),
	  NULL },
	{ "server-mode", 's', POPT_ARG_NONE, &server_mode, 0,
	  N_("Do not open any windows"),
	  NULL },
	{ "quit", 'q', POPT_ARG_NONE, &quit_option, 0,
	  N_("Quit server"),
	  NULL },
	{ "close", 'c', POPT_ARG_NONE, &close_option, 0,
	  N_("close active window keep server mode on"),
	  NULL },
  	{ "load-session", 'l', POPT_ARG_STRING, &session_filename, 0,
	  N_("Load the given session file"),
	  N_("FILE") },
	/* terminator, must be last */
	{ NULL, 0, 0, NULL, 0, NULL, NULL }
};

/* xscreensaver argument parsing structure */
static struct poptOption popt_options_xscrsaver[] =
{
	{ "window", 'w', POPT_ARG_NONE, &xscr_window, 0,
	  N_("XScreenaver: in a window"),
	  NULL },
	{ "root", 'r', POPT_ARG_NONE, &xscr_root, 0,
	  N_("XScreenaver: in the root window"),
	  NULL },
	{ "mono", 'm', POPT_ARG_NONE, &xscr_mono, 0,
	  N_("XScreenaver: mono"),
	  NULL },
	{ "install", 'i', POPT_ARG_NONE, &xscr_install, 0,
	  N_("XScreenaver: install colormap"),
	  NULL },
	{ "noinstall", 'n', POPT_ARG_NONE, &xscr_noinstall, 0,
	  N_("XScreenaver: Do not install colormap"),
	  NULL },
	{ "visual", 'v', POPT_ARG_INT, &xscr_visual, 0,
	  N_("XScreenaver: visual id"),
	  NULL },
	{ "window-id", 'e', POPT_ARG_INT, &xscr_window_id, 0,
	  N_("XScreenaver: window id"),
	  NULL },
  	{ "delay", 'd', POPT_ARG_INT, &xscr_delay, 0,
	  N_("XScreenaver: delay"),
	  NULL },
	{ NULL, 0, 0, NULL, 0, NULL, NULL }
};

/* singleton instance of the shell, use "extern" to declare it */
LumShell *lum_shell;

static void lum_main_start (void);

/* perhaps not necessary */
static int lum_main_translate_uri_arguments(poptContext context, char ***urls)
{
	char buffer[PATH_MAX];
	char **args;
	int i, n;

	if (context == NULL){
		*urls = NULL;
		return 0;
	}

	args = (char **) poptGetArgs (context);
	if (args == NULL){
		poptFreeContext (context);
		*urls = NULL;
		return 0;
	}

	for (n = 0; args[n] != NULL; n++);

	*urls = g_new0 (char *, n + 1);

	for (i = 0; i < n; i++){
		if (realpath (args[i], buffer) != NULL){
			(*urls)[i] = g_strconcat("file://", buffer, NULL);
		} else {
			(*urls)[i] = g_strdup (args[i]);
		}
	}
	poptFreeContext (context);
	(*urls)[i] = NULL;

	return n;
}

/* unref LumShell when all windows are closed
 * lumiere will then exit if not in server mode
 */
static gboolean
unref_when_idle (gpointer user_data)
{
#ifndef NDEBUG
	puts(__PRETTY_FUNCTION__);
#endif
	g_object_unref (user_data);
	return FALSE;
}

int main (int argc, char *argv[])
{
	poptContext context;
	GValue context_as_value = { 0 };
	GnomeProgram *program;
	GNOME_Lum_Automation factory;
	struct poptOption *options;

	if(strstr(argv[0], "lumiere-saver"))
		options = popt_options_xscrsaver;
	else
		options = popt_options;
	/* Standard Gnome initialization stuff */
#ifdef ENABLE_NLS
	/* Initialize the i18n stuff */
	setlocale(LC_ALL, "");
	bindtextdomain(GETTEXT_PACKAGE, DATADIR "/locale");
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
	textdomain(GETTEXT_PACKAGE);
#endif
	XInitThreads();

	if (!gnome_vfs_init ()) {
		printf ("Could not initialize GnomeVFS\n");
		return 1;
	}
	g_set_prgname (PACKAGE);
	g_set_application_name (_("Lumiere"));


	program = gnome_program_init (PACKAGE, VERSION,
				 LIBGNOMEUI_MODULE, argc, argv,
				 GNOME_PARAM_POPT_TABLE, options,
				 GNOME_PARAM_HUMAN_READABLE_NAME, _("Lumiere"),
				GNOME_PROGRAM_STANDARD_PROPERTIES,
				 LIBGNOMEUI_PARAM_DEFAULT_ICON,
				 DATADIR"/pixmaps/gnome-lumiere.png",
				 GNOME_PARAM_NONE);
#ifndef NDEBUG
	bonobo_debug_init();
#endif
	g_object_get_property (G_OBJECT (program),
				 GNOME_PARAM_POPT_CONTEXT,
				 g_value_init (&context_as_value, G_TYPE_POINTER));

	context = g_value_get_pointer (&context_as_value);

	n_uris = lum_main_translate_uri_arguments(context, &uri);


	factory = bonobo_activation_activate_from_id 
			(LUM_FACTORY_OAFIID,
			 Bonobo_ACTIVATION_FLAG_EXISTING_ONLY, 
			 NULL, NULL);

	if (!factory){
	/* first init, launch process */
		LumAutomation *automation = g_object_new(LUM_TYPE_AUTOMATION, NULL);

		if(gnome_vfs_init () == FALSE)
			g_error (_("Could not initialize GnomeVFS!\n"));

		gnome_window_icon_set_default_from_file (DATADIR"/pixmaps/gnome-lumiere.png");

/* END - Standard Gnome initialization stuff */

		lum_shell = lum_shell_new ();
		g_object_add_weak_pointer (G_OBJECT(lum_shell),
			(gpointer *) &lum_shell);
		lum_main_start();
	
		if(open_as_nautilus_view || server_mode)
			lum_automation_complete_initialization(automation, FALSE);
		else
			lum_automation_complete_initialization(automation, TRUE);

		if(open_as_nautilus_view && !server_mode){
			g_object_ref(lum_shell);
			g_timeout_add(10000, unref_when_idle, lum_shell);
		}

		g_idle_add (unref_when_idle, lum_shell);
		bonobo_object_unref(automation);
		bonobo_main ();
#ifndef NDEBUG
		bonobo_debug_shutdown();
#endif	
		gnome_vfs_shutdown ();

	} else {
	/* server already running, connecting */
		lum_main_start ();
		gdk_notify_startup_complete ();
	}
	return EXIT_SUCCESS;
}
		
/** lum_main_start:
 * call LumAutomation CORBA server with command-line options
 */
static void lum_main_start (void)
{
	int i;
	GNOME_Lum_Automation lumserver = NULL;

	CORBA_exception_init (&corba_env);

	lumserver = bonobo_activation_activate_from_id (
				"OAFIID:GNOME_Lum_Automation",
				0, NULL, &corba_env);

	if (corba_env._major != CORBA_NO_EXCEPTION) 
		g_warning ("Error: unable to initialize corba: %s\n", 
			   CORBA_exception_id (&corba_env));
	
	/* default case */
	if (!open_in_existing_window && !open_in_new_window && !enqueue)
		open_in_new_window = TRUE;

	if (lumserver == NULL) {
		GtkWidget *dialog;
		dialog = gtk_message_dialog_new(NULL,
			 GTK_DIALOG_MODAL,
			 GTK_MESSAGE_ERROR,
			 GTK_BUTTONS_OK,
 			 _("Lumiere can't be used now.."));
		char * text = g_strdup(_(
			 "Running the command \"bonobo-slay\" "
			 "from the console may fix the problem. If not, "
			 "you can try logging out of Gnome or "
			 "installing Lumiere again.\n\n"
			 "Bonobo couldn't locate the <tt>\"GNOME_Lum_Automation.server\"<tt> file."));
		gtk_message_dialog_format_secondary_text (
			GTK_MESSAGE_DIALOG(dialog), "%s\n\n%s", text,
			_("Usually this is an indication of an incomplete installation. "
			"See the INSTALL document for more details."));
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy(dialog);
		g_free (text);
		exit(EXIT_FAILURE);
	}

	if (server_mode && !close_option) {
	/* Server mode */
		GNOME_Lum_Automation_setServerMode (lumserver, TRUE, &corba_env);
	}

	/* load the session if requested */
	if (session_filename)
	{
		GNOME_Lum_Automation_loadsession 
			(lumserver, session_filename, &corba_env);

	} else if (close_option || quit_option) {
	/* quit */
		GNOME_Lum_Automation_quit (lumserver,
					 quit_option, &corba_env);

	} else if (n_uris == 0 && 
		 !open_as_nautilus_view && !server_mode) {
		/* open a default window */
		GNOME_Lum_Automation_loaduri 
			(lumserver, "",
			 open_in_existing_window,
			 open_in_new_window,
			 FALSE,
			 &corba_env);
	} else {
		/* open all of the uris */
		for (i = 0; i < n_uris; i++) {
			GNOME_Lum_Automation_loaduri 
				(lumserver, uri[i],
				 open_in_existing_window,
				 open_in_new_window,
				 enqueue,
				 &corba_env);
		}
	}

	if (lumserver) {
		bonobo_object_release_unref (lumserver, &corba_env);
	}
	CORBA_exception_free (&corba_env);
}
