/*
**  Sinek (Media Player)
**  Copyright (c) 2001-2002 Gurer Ozen
**
**  This code is free software; you can redistribute it and/or
**  modify it under the terms of the GNU General Public License.
**
**  screen saver control
*/

#include "config.h"
#include "scrsaver.h"

#include <glib.h>
#include <gdk/gdkx.h>
#include <gdk/gdk.h>
#ifdef HAVE_LIBXTST
#include <X11/extensions/XTest.h>
#endif /* HAVE_LIBXTST */
#include <X11/keysym.h>

#define XSCREENSAVER_MIN_TIMEOUT 60

struct ScreenSaver {
	/* Whether the screensaver is disabled */
	gboolean disabled;
	Display *display;

	/* To save the screensaver info */
	int timeout;
	int interval;
	int prefer_blanking;
	int allow_exposures;

	/* For use with XTest */
	int keycode;
	Bool have_xtest;
};

#ifdef HAVE_LIBXTST
static gboolean
fake_event (ScreenSaver *scr)
{
	if (scr->disabled)
	{
		XLockDisplay (scr->display);
		XTestFakeKeyEvent (scr->display, scr->keycode,
				True, CurrentTime);
		XTestFakeKeyEvent (scr->display, scr->keycode,
				False, CurrentTime);
		XFlush (scr->display);
		XUnlockDisplay (scr->display);
	}

	return TRUE;
}
#endif /* HAVE_LIBXTST */

ScreenSaver
*scrsaver_new (void)
{
	ScreenSaver *scr;
	int a, b, c, d;

	scr = g_new0 (ScreenSaver, 1);
	scr->display = GDK_DISPLAY_XDISPLAY(gdk_display_get_default());

#ifdef HAVE_LIBXTST
	XLockDisplay (scr->display);
	scr->have_xtest = XTestQueryExtension (scr->display, &a, &b, &c, &d);
	if(scr->have_xtest == True)
	{
		scr->keycode = XKeysymToKeycode (scr->display, XK_Shift_L);
	}
	XFlush (scr->display);
	XUnlockDisplay (scr->display);
#endif /* HAVE_LIBXTST */

	return scr;
}

void
scrsaver_disable (ScreenSaver *scr)
{
	g_return_if_fail (scr->disabled == FALSE);

	scr->disabled = TRUE;

#ifdef HAVE_LIBXTST
	if (scr->have_xtest == True)
	{
		XLockDisplay (scr->display);
		XGetScreenSaver(scr->display, &scr->timeout,
				&scr->interval,
				&scr->prefer_blanking,
				&scr->allow_exposures);
		XFlush (scr->display);
		XUnlockDisplay (scr->display);

		if (scr->timeout != 0)
		{
			g_timeout_add (scr->timeout / 2 * 1000,
					(GSourceFunc) fake_event, scr);
		} else {
			g_timeout_add (XSCREENSAVER_MIN_TIMEOUT / 2 * 1000,
					(GSourceFunc) fake_event, scr);
		}

		return;
	}
#endif /* HAVE_LIBXTST */

	XLockDisplay (scr->display);
	XGetScreenSaver(scr->display, &scr->timeout,
			&scr->interval,
			&scr->prefer_blanking,
			&scr->allow_exposures);
	XSetScreenSaver(scr->display, 0, 0,
			DontPreferBlanking, DontAllowExposures);
	XFlush (scr->display);
	XUnlockDisplay (scr->display);
}

void
scrsaver_enable (ScreenSaver *scr)
{
	g_return_if_fail (scr->disabled == TRUE);

	scr->disabled = FALSE;

#ifdef HAVE_LIBXTST
	if (scr->have_xtest == True)
	{
		g_source_remove_by_user_data (scr);
		return;
	}

#endif /* HAVE_LIBXTST */
	XLockDisplay (scr->display);
	XSetScreenSaver (scr->display,
			scr->timeout,
			scr->interval,
			scr->prefer_blanking,
			scr->allow_exposures);
	XFlush (scr->display);
	XUnlockDisplay (scr->display);
}

void scrsaver_free(ScreenSaver *scr)
{
	g_source_remove_by_user_data (scr);
	g_free (scr);
}
