# -*- coding: utf-8 -*-
# Author:   $Author: merkosh $
# Revision: $Rev: 83 $
############################################################################
#    Copyright (C) 2005 by Uwe Mayer                                       #
#    merkosh@hadiko.de                                                     #
#                                                                          #
#    This program is free software; you can redistribute it and/or modify  #
#    it under the terms of the GNU General Public License as published by  #
#    the Free Software Foundation; either version 2 of the License, or     #
#    (at your option) any later version.                                   #
#                                                                          #
#    This program is distributed in the hope that it will be useful,       #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
#    GNU General Public License for more details.                          #
#                                                                          #
#    You should have received a copy of the GNU General Public License     #
#    along with this program; if not, write to the                         #
#    Free Software Foundation, Inc.,                                       #
#    59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             #
############################################################################

#-- imports --------------------------------------------------------------------
#-- Qt imports
from qt import QImage, QPixmap, QLabel, QGridLayout, PYSIGNAL
from qt import QImageDrag, QDragObject, QSizePolicy


#-- Python imports
import logging
import sys
import os

from os.path import exists as fexists, dirname, splitext, join as fjoin


#-- own imports
from PicturePreviewBase import PicturePreviewBase
from PictureDialog import PictureDialog
from DragObject import XMozUrlDrag, URIListDrag
from misc import openResource
from IODeviceBuffer import IODeviceBuffer
from Settings import getPreferences


#-- constants ------------------------------------------------------------------
# defines maximum number of characters which are recognized as a file suffix
# rather than binary image data
MAX_SUFFIX_LENGTH = 20


#-- preferences ----------------------------------------------------------------
pref = getPreferences(__name__)



class DropLabel(QLabel):
    """label with support for dropping images"""
    def __init__(self, parent=None, name=None, fl=0):
        QLabel.__init__(self, parent, name, fl)
        self.setAcceptDrops(True)
        self.setSizePolicy( QSizePolicy(QSizePolicy.Ignored, QSizePolicy.Ignored) )
        
        # create logging object
        self.log = logging.getLogger( self.__class__.__name__ )
        self.log.setLevel(pref['logLevel'])


    def dragEnterEvent(self, e):
        """called when drop is initiated"""
        accept = QImageDrag.canDecode(e) or XMozUrlDrag.canDecode(e) or \
                 URIListDrag.canDecode(e)
        self.log.debug("Accepting source: %s (type=\"%s\")", str(accept), e.format())
        e.accept( accept )


    def dropEvent(self, e):
        """called when image is dropped on the widget"""
        img = QImage()
        if (QImageDrag.decode(e, img) or XMozUrlDrag.decode(e, img) or \
            URIListDrag.decode(e, img)):
            self.log.debug("Accepting dropped image: %dx%d (%d bytes)",
                           img.width(), img.height(), img.numBytes())
            self.emit(PYSIGNAL("imageDropped"), (img,))
        else:
            self.log.warning("Could not decode dropped image")




class PicturePreview(PicturePreviewBase):
    """displays a picture and scales it appropriately"""
    def __init__(self, dirname, parent=None, name=None, fl=0):
        PicturePreviewBase.__init__(self, parent, name, fl)

        # create logging
        self.log = logging.getLogger( self.__class__.__name__ )
        self.log.setLevel( pref['logLevel'] )

        # runtime variables
        self.basedir = dirname          # remember picture base directory
        self.record = None
        self.img = None                 # original image object
        self.pix = QPixmap()            # pixmap
        self.labelImage = DropLabel(self) # label with drop support

        gridLayoutLabel = QGridLayout( self, 1,1 )
        gridLayoutLabel.addWidget( self.labelImage, 0,0 )

        # connect signal to detect dropped images
        self.connect(self.labelImage, PYSIGNAL("imageDropped"), self.onImageDropped)



    #-- event handlers ---------------------------------------------------------
    def resizeEvent(self, e):
        """called when widget is resized"""
        self.resizeImage()
        

    def mousePressEvent(self, e):
        """called when a mouse button is pressed"""
        # only accept left mouse button
        if (e.button() != e.LeftButton):
            e.ignore()
        # spawn unscaled picture
        if (self.img != None):
            dlg = PictureDialog(self.img, modal=False)
            dlg.exec_loop()
        

    def onImageDropped(self, img):
        """called when image dropped"""
        self.img = img
        self.record.modified(True)      # causes record to be saved, because it was
                                        # modified
        self.resizeImage()              # update on screen
        self.getRecord(forceDecode=True) # update internal record
        self.emit(PYSIGNAL("modified"), ()) # notify parent
        
        

    #-- interface --------------------------------------------------------------
    def setRecord(self, rec):
        """sets a record and displays the picture"""
        self.log.debug("Setting record: %d, %s", rec['number'], rec['mediaLabel'])
        # remember record
        self.record = rec

        #-- read record fields and put picture, if available on streen
        if (self.record == None):
            self.img = None
            
        # record has embedded image   
        elif (len(self.record['embeddedImage']) > MAX_SUFFIX_LENGTH):
            self.log.debug("Loading embedded image")
            self.img = QImage()
            ok = self.img.loadFromData(self.record['embeddedImage'])
            if (not ok):
                self.log.error("Could not identify embedded image data")
                self.img = None
                
        # try loading picture from reference
        elif (self.record['picture']):
            self.log.debug("Loading linked image: %s", self.record['picture'])

            pathToPicture = self.basedir # assume relative path
            
            # absolute picture path or linked image
            if ((self.record['picture'][0] in ('\\', '/')) or
                (self.record['picture'][:7] in ('http://',))):
                pathToPicture = ""
                
            # load picture
            f = openResource(fjoin(pathToPicture, self.record['picture']), proxies=pref['proxy'])
            if (f != None):
                self.img = QImage()
                ok = self.img.loadFromData(f.read())
                f.close()
                if (not ok):
                    self.log.error("Could not identify embedded image data")
                    self.img = None
            else:
                self.img = None
                
        # new record: display no picture
        else:
            self.img = None
        
        # update image label
        self.resizeImage()


    def resizeImage(self):
        """puts the image in a label on the screen at the correct size"""
        if (self.img != None):
            size = (self.labelImage.contentsRect().width(), \
                    self.labelImage.contentsRect().height())
            img = self.img.smoothScale( size[0], size[1] , QImage.ScaleMin )
            self.pix.convertFromImage(img)
            self.labelImage.setPixmap(self.pix)

        else:
            self.log.debug("No image set: clearing label")
            self.labelImage.clear()



    def getRecord(self, forceDecode=False):
        """reads the displayed image back into the record and returns it"""
        self.log.debug("Returning record: %d, %s", self.record['number'], self.record['mediaLabel'])

        # create empty data if none set
        if (self.record == None): return None
        # only update picture data if it was modified
        if (not forceDecode): return self.record

        # update record fields
        #-- handle embedded image
        if (pref['embeddedImage']):
            # extract image from QImage 
            data = IODeviceBuffer()
            if ( self.img.save(data, pref['imageFormat']) ):
                self.log.debug("Saving embedded image: %d bytes", data.size())
                self.record['embeddedImage'] = data.readAll()
                self.record['picture'] = ".%s"%pref['imageFormat'].lower()
                data.close()            # free memory
                return self.record
            else:
                self.log.error("Could not save image to record field")
                self.record['embeddedImage'] = ""
                self.record['picture'] = u""
                return self.record
        #-- handle save image as link
        else:
            pathToPicture = self.basedir # assume relative path
            
            # absolute picture path or linked image
            if ((pref['filenamePrefix'][0] in ('\\', '/')) or
                (pref['filenamePrefix'][:7] in ('http://',))):
                pathToPicture = ""

            # build filename from prefix (including suffix)
            try:
                filename = (pref['filenamePrefix']%self.record['number'])+\
                           ".%s"%pref['imageFormat'].lower()
            except TypeError, e:
                self.log.error("Invalid filename prefix: %s", unicode(e))
                self.record['embeddedImage'] = ""
                self.record['picture'] = u""
                return self.record

            # create target directory
            filename = fjoin(pathToPicture, filename)
            filedir = dirname(filename)
            if (not fexists(filedir)):
                self.log.info("Creating target directory %s", filedir)
                try:
                    os.makedirs(filedir)
                except IOError, e:
                    self.log.error("Could not create target directory: %s", unicode(e))
                    self.record['embeddedImage'] = ""
                    self.record['picture'] = u""
                    return self.record

            # test if file exists
            if (fexists(filename)):
                self.log.warning("Target file %s already exists: overwriting", filename)

            # copy data to target file
            data = IODeviceBuffer()
            # copy to target buffer is successfull
            if (self.img.save(data, pref['imageFormat'])):
                self.log.debug("Saving linked image %s: %d bytes", filename, data.size())
                try:
                    f = open(filename, "wb")
                    f.write(data.readAll())
                    f.close
                    data.close()        # free memory
                except IOError, e:
                    self.log.error("Could not write data to disk: %s", unicode(e))
                    self.record['embeddedImage'] = ""
                    self.record['picture'] = u""
                    return self.record
                else:
                    self.record['embeddedImage'] = str(splitext(filename)[1]) # MUST be ASCII string
                    self.record['picture'] = filename
                    return self.record
                
            # could not save data to target buffer
            else:
                self.log.error("Could not save image to intermediate buffer")
                self.record['embeddedImage'] = ""
                self.record['picture'] = u""
                return self.record


#-- testroutine ----------------------------------------------------------------
if (__name__ == "__main__"):
    from qt import QApplication, QObject, SIGNAL, SLOT
    from AMCFile33 import AMCFile33

    a = QApplication(sys.argv)
    QObject.connect(a,SIGNAL("lastWindowClosed()"),a,SLOT("quit()"))

    # need logging handlers for testing
    log = logging.getLogger("")
    hdlr = logging.StreamHandler()
    formatter = logging.Formatter('[%(name)s] %(levelname)s: %(message)s')
    hdlr.setFormatter(formatter)
    log.addHandler(hdlr) 
    log.setLevel(pref['logLevel'])

    # read sample record
    f = AMCFile33("Filme.amc", "rb")
    rec = f.read()[-1]

    w = PicturePreview(".")
    w.setRecord(rec)

    a.setMainWidget(w)
    w.show()

    a.exec_loop()

    tmp = w.getRecord(forceDecode=True)
    print "len embedded: ",len(tmp['embeddedImage'])
    print "picture:  ",tmp['picture']
