# -*- coding: utf-8 -*-
# Author:   $Author: merkosh $
# Revision: $Rev: 162 $
############################################################################
#    Copyright (C) 2005 by Uwe Mayer                                       #
#    merkosh@hadiko.de                                                     #
#                                                                          #
#    This program is free software; you can redistribute it and/or modify  #
#    it under the terms of the GNU General Public License as published by  #
#    the Free Software Foundation; either version 2 of the License, or     #
#    (at your option) any later version.                                   #
#                                                                          #
#    This program is distributed in the hope that it will be useful,       #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
#    GNU General Public License for more details.                          #
#                                                                          #
#    You should have received a copy of the GNU General Public License     #
#    along with this program; if not, write to the                         #
#    Free Software Foundation, Inc.,                                       #
#    59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             #
############################################################################

#-- imports --------------------------------------------------------------------
#-- Qt imports
from qt import QImage, QPixmap, QLabel, QGridLayout, PYSIGNAL
from qt import QImageDrag, QDragObject, QSizePolicy, QPopupMenu, QIconSet
from qt import QCursor, QPixmap, QFileDialog, QMessageBox


#-- Python imports
import logging
import os

from os.path import exists as fexists, dirname, splitext, join as fjoin, \
     basename, splitext


#-- own imports
from PicturePreviewBase import PicturePreviewBase
from PictureDialog import PictureDialog
from DragObject import XMozUrlDrag, URIListDrag
from misc import openResource
from IODeviceBuffer import IODeviceBuffer
from Settings import getPreferences


#-- constants ------------------------------------------------------------------
# defines maximum number of characters which are recognized as a file suffix
# rather than binary image data
MIN_PICTURE_SIZE = 20


recycle_bin_data = \
    "\x89\x50\x4e\x47\x0d\x0a\x1a\x0a\x00\x00\x00\x0d" \
    "\x49\x48\x44\x52\x00\x00\x00\x10\x00\x00\x00\x10" \
    "\x08\x06\x00\x00\x00\x1f\xf3\xff\x61\x00\x00\x01" \
    "\xa7\x49\x44\x41\x54\x38\x8d\x95\x93\x31\x6b\xdb" \
    "\x40\x18\x86\x1f\x99\x0c\x32\x28\xe0\xa1\x14\x79" \
    "\x8b\x41\x53\x68\x86\x40\x07\x4b\x60\x68\xb6\x72" \
    "\x78\xd3\xd4\x8c\xc9\x96\xa4\x53\x7e\x4a\x43\x36" \
    "\x8f\xce\x2f\x30\x37\x66\x09\xba\xd1\xca\x94\xe9" \
    "\xc0\xa3\x85\xa6\x80\x8d\x3c\x7e\x1d\xcc\xa9\x52" \
    "\xad\x96\x56\x70\x1c\xaf\x78\xf5\xdd\xab\x47\xaf" \
    "\x90\x52\x10\x23\x48\x29\x64\x26\xe3\x7f\x75\x7d" \
    "\xb3\xb9\x03\xf2\xfb\xea\xf2\x49\x29\x78\x52\x0a" \
    "\xc6\x1a\x92\x28\xc1\xfb\xe8\x09\xc0\x42\x2f\x28" \
    "\x5e\x0a\xb4\xd5\xa8\x48\x11\x4e\x42\xa6\x6a\x0a" \
    "\x80\x94\xe2\x39\x3f\x16\xbc\xcc\x64\xf5\xc3\x37" \
    "\xdf\x6f\x28\x8a\x02\x80\xf8\x73\x4c\xea\xa7\xcc" \
    "\xb7\x73\xf2\xd7\x1c\x80\x30\x0c\x79\xfc\xf1\xd8" \
    "\x1a\x72\xe4\x26\xb9\xeb\xfe\xeb\x3d\xf9\x36\xe7" \
    "\x6a\x72\xc5\x73\xf1\xcc\xed\x87\x5b\x9e\x82\x27" \
    "\xce\x83\x73\xe6\xaf\xf3\xbd\xc9\x42\x12\x25\x18" \
    "\x6b\x7e\x31\xe8\x7a\xef\x3f\xad\xbf\x32\x78\x98" \
    "\x3e\xf0\x36\x7a\xe3\x7a\x72\xcd\x8a\x15\x23\x46" \
    "\xcc\x5e\x66\x9c\xae\x4e\xb9\x5b\xdc\x1d\x30\xe8" \
    "\x39\x61\xac\x01\x60\x7d\xb6\x66\x18\x0e\xc9\x37" \
    "\x39\xe9\xa7\x94\x7c\x93\x33\x0c\x87\xac\xcf\xd6" \
    "\x00\xb4\xfc\xd1\x9e\x6a\xfd\x5d\x01\xc9\x66\x99" \
    "\xa8\xa9\x92\x6a\x59\xc9\xe5\xb7\x4b\xa9\x96\x95" \
    "\xa8\xa9\x92\x6c\x96\xd5\xf1\x9b\x3d\xe8\xb9\x49" \
    "\x2e\x81\xb6\x9a\xdd\x76\x87\xb6\x9a\xf7\xcd\x7b" \
    "\x4b\xbb\x04\xce\x9f\x44\x09\xad\x52\xb8\x04\xf1" \
    "\x38\x96\x6a\x59\xc9\xc5\x97\x0b\xa9\x96\x95\xc4" \
    "\xe3\xb8\x95\xa0\xd9\xc8\x03\x06\xda\x6a\x7c\xdf" \
    "\x47\x5b\x4d\x3f\xe8\xb7\x74\x17\x83\x9e\xa3\x99" \
    "\x44\x09\x00\x2a\x52\xf4\x83\x3e\x2a\x52\x0c\x8e" \
    "\x07\x2d\x0d\xd0\xf4\x1b\x6b\xba\x19\x0c\x8e\x07" \
    "\xfb\x93\x03\xbf\xa5\xbb\x18\x1c\xb9\x49\xcd\x04" \
    "\x79\x98\xa3\x22\x45\x1a\xa6\xec\x82\x1d\xc5\xb8" \
    "\xe0\x84\x93\x3a\x41\xb3\x07\x07\x7f\xd7\xbf\x36" \
    "\xd0\xed\x3f\x01\x02\x90\x9f\xc2\x40\xd7\xb4\x8f" \
    "\x00\x00\x00\x00\x49\x45\x4e\x44\xae\x42\x60\x82"

copy_data = \
    "\x89\x50\x4e\x47\x0d\x0a\x1a\x0a\x00\x00\x00\x0d" \
    "\x49\x48\x44\x52\x00\x00\x00\x10\x00\x00\x00\x10" \
    "\x08\x06\x00\x00\x00\x1f\xf3\xff\x61\x00\x00\x01" \
    "\xe1\x49\x44\x41\x54\x38\x8d\x7d\x93\xb1\x6b\xdb" \
    "\x40\x18\xc5\x7f\x2e\x59\x0b\xf5\x10\x90\x06\x97" \
    "\x08\xec\x25\x43\xb1\x42\x27\x8f\x99\x8a\xdb\x2e" \
    "\xf1\xd8\x2d\x43\x87\x92\xad\x63\xc6\xa4\x5d\xfa" \
    "\x17\x14\x9a\xad\x53\xb1\x32\x86\x52\x70\x87\x14" \
    "\x69\xb3\x06\x13\xb9\x60\x1b\x1b\x1c\xf0\x15\x1b" \
    "\x3c\xb8\x20\x42\x97\xd7\xe1\x2c\xc5\x36\xc1\x1f" \
    "\x88\xa7\x3b\xee\x3e\xde\xfb\xdd\x5d\xe1\xb4\x7a" \
    "\xaa\x2b\xae\xc8\xaa\x4e\x9d\x6d\xe3\xf8\x47\x5c" \
    "\x60\x00\x94\x21\x1a\x44\xe0\x57\x7d\x49\x92\xd2" \
    "\xd4\xca\xc2\xea\x7c\x3a\xb7\x3a\x99\x48\x92\x26" \
    "\xe3\x89\xfc\xaa\xaf\x30\x0a\xd1\x54\x64\x6a\x1b" \
    "\x2c\x24\xc7\x71\xf2\x4f\x53\xdb\x53\xe3\x75\xf5" \
    "\xab\xbe\x56\x37\x2b\x12\x8f\x00\x78\x6c\xed\x19" \
    "\x63\xe8\x76\xba\xb0\x0b\xe6\xd6\x40\x09\xcc\x68" \
    "\x04\x25\x18\xf5\x47\x80\xb5\x5d\x2b\xd7\xac\xfd" \
    "\x32\xec\xd4\xa9\xc3\xdf\x65\xc0\x19\xec\x3f\xdb" \
    "\xcf\xf3\x9a\x6b\x83\x5b\xf1\xa0\x0f\xde\x9e\x07" \
    "\x40\xad\x5c\x83\x01\xf7\x4d\x32\x06\x59\x66\xc7" \
    "\x71\x24\x49\xc3\xde\xf0\x5e\xff\x49\x49\x37\xd9" \
    "\xc2\x60\x25\xb3\xe3\x38\x52\x6f\x9d\x49\xd2\x4d" \
    "\xa4\xc5\x36\x06\x2b\x99\xa3\xeb\x08\x2a\xcb\x08" \
    "\x1d\x43\x70\x19\xe0\xdd\x79\x04\xdf\x83\x2d\x0c" \
    "\x6e\xc1\x2d\xb9\x36\x6b\xc5\x83\xdf\xe4\x8b\x1b" \
    "\x47\x0d\x56\xab\x71\xd4\x50\xf6\x1f\x5c\x06\x85" \
    "\x9c\x41\x96\x39\xe9\x26\x79\xe6\x34\x4e\x73\x26" \
    "\xf3\x69\x6b\x4d\x97\xc7\xcd\x0e\xc0\xf9\xc7\x73" \
    "\x8a\xbf\x8a\xc4\x6e\xcc\xc5\xe7\x0b\xe2\x9b\x98" \
    "\x83\xf2\x41\x6e\x9b\xd9\x4f\x8a\xbb\x87\x30\xc3" \
    "\xaa\x59\xce\x0f\x80\x30\x0a\xf1\xab\xbe\xd2\x45" \
    "\xaa\xfa\x8b\xba\xda\x71\x5b\x5a\x48\xed\xb8\xfd" \
    "\x80\x83\x54\x93\xf1\x57\x49\xc3\xdc\x41\x4e\xd3" \
    "\xaf\xfa\x3a\xfb\x70\x26\x2d\x24\xc5\x92\xa6\x52" \
    "\xf3\x5b\xd3\x2e\x9c\x34\x25\xcd\xa5\xe5\x66\xf5" \
    "\x3e\xc9\x71\x1c\x85\x51\x48\x21\x8c\xc2\x9c\xea" \
    "\xc9\xbb\x13\xb1\x51\xe6\x8f\xa1\xf6\xfc\x8e\xa3" \
    "\xd7\x2e\x6f\xde\x1e\x63\xfa\xe0\xee\x1d\xe2\x3e" \
    "\x7d\x89\xe9\x98\x02\x9b\xe7\xba\x39\xb6\x56\x5b" \
    "\xfa\xf2\xea\x89\x14\x1f\x4b\x69\x4b\xea\xbc\xb7" \
    "\xce\x22\xc1\xe6\xcd\x7a\x48\x57\x2f\xd5\xc6\xa3" \
    "\xe3\x3f\x01\x68\x0b\xe9\x8a\xed\x68\x05\x00\x00" \
    "\x00\x00\x49\x45\x4e\x44\xae\x42\x60\x82"

paste_data = \
    "\x89\x50\x4e\x47\x0d\x0a\x1a\x0a\x00\x00\x00\x0d" \
    "\x49\x48\x44\x52\x00\x00\x00\x10\x00\x00\x00\x10" \
    "\x08\x06\x00\x00\x00\x1f\xf3\xff\x61\x00\x00\x01" \
    "\xba\x49\x44\x41\x54\x38\x8d\x8d\x92\x3f\x6b\x14" \
    "\x51\x14\xc5\x7f\x09\x29\x26\xb0\xcd\x06\x84\x6c" \
    "\x11\x58\x65\x2d\xec\x66\x4a\x2d\x04\xc1\x42\x26" \
    "\xdf\x40\xb0\xd8\x94\x7e\x83\xb0\xa4\x4a\xb4\xb3" \
    "\x10\xb6\xb1\xd8\x2f\x10\xb7\x9c\xbc\xc2\x32\xb2" \
    "\xd3\xe5\x55\xba\xc2\x66\x98\x40\x84\x41\x36\x90" \
    "\x62\x09\xb3\xe5\xb1\x78\x3b\xff\x56\x23\x3e\x78" \
    "\x1c\x2e\x87\x7b\xcf\xb9\x97\xb3\x31\xf0\x07\x32" \
    "\x18\x8a\x17\x12\xf2\xaf\xda\x7e\xb1\x1b\x24\x40" \
    "\x0f\xe2\x24\x86\xc0\x0f\x24\x49\xca\x73\x07\x0b" \
    "\x87\xb7\xf3\x5b\x87\x59\x26\x49\x1a\x9f\x8e\xe5" \
    "\xb5\x3c\x4d\xe2\x09\x9a\x8b\x02\xdd\x80\x85\x9b" \
    "\x51\xe2\x7c\x85\xd7\x6a\x34\x0f\xfb\x34\x86\x28" \
    "\x56\xe5\x60\x5d\x39\xbb\x76\xca\x59\x9a\x2a\xb7" \
    "\xb9\xbc\x96\x27\xcd\xdf\xae\xb0\xe6\x60\xe0\x0f" \
    "\xee\x55\xd6\x4c\xb5\xe6\x48\xfa\x88\x64\x69\x0c" \
    "\x29\x1d\xac\x2b\xa7\xb3\x4a\x39\xed\xa3\x74\xd5" \
    "\xfc\xfd\x03\x4a\xfb\xd5\x90\x4d\x00\x6e\xa0\xfd" \
    "\xa0\x0d\x3f\xa1\xb3\xd7\x81\x4b\xd8\xbd\xdb\x65" \
    "\xe7\xf9\x0e\xcb\x6c\x89\xf1\x87\x78\x5b\x5d\x46" \
    "\x9f\x9f\xd2\xf6\xba\x18\x7f\xc8\xd2\x2e\xd9\x7e" \
    "\xb4\xad\xd2\x41\x5d\xb9\x7e\xb8\xc8\x44\xca\x6d" \
    "\xae\xc8\x44\x52\xa6\xb2\x2e\x78\x77\x83\xda\xce" \
    "\x92\xa4\x69\x6d\xf7\x99\xf4\xe2\xcd\x81\xba\x87" \
    "\xd2\xeb\x23\xa9\x7b\xe8\xea\x82\xdf\x34\x18\xd8" \
    "\x83\xab\xcb\x2b\x78\x0c\xd3\x1f\x53\xe8\x81\x49" \
    "\x5c\x78\xce\x92\x33\x97\xa0\x84\x06\x16\xfc\x16" \
    "\xc0\xc9\xfb\x13\xda\x5f\xdb\xd8\x8e\x65\xf4\x69" \
    "\x84\xfd\x66\x09\x7b\x21\x00\xfb\xbd\x7d\x8c\xff" \
    "\x8b\xb0\x35\xc5\x24\x86\x87\x2f\x43\x9e\xdc\x3d" \
    "\x2b\x79\x26\xf1\x84\xc0\x0f\x94\x2f\x72\x85\xaf" \
    "\x42\x5d\xd8\x0b\x69\xd1\xbc\x81\xb2\xea\x36\xe9" \
    "\x2c\x95\xb2\x8a\x2f\x13\x15\xf8\x81\x8e\xdf\x1d" \
    "\xbb\x4c\xd8\xe6\x0d\x74\xbe\x96\x8f\xf3\x8a\xa7" \
    "\x9e\xed\xc0\x0f\x54\xff\xff\xed\xa0\x9e\xed\x7a" \
    "\x5d\x38\x18\x9f\x8e\x2b\x27\xab\x3a\x35\xe9\x9f" \
    "\x0e\xfe\x86\x5e\xcb\xd3\x7d\x5f\x73\xf1\x1b\x8d" \
    "\x1e\x6c\xa4\x5a\xd2\x7e\x46\x00\x00\x00\x00\x49" \
    "\x45\x4e\x44\xae\x42\x60\x82"


saveAs_data = \
    "\x89\x50\x4e\x47\x0d\x0a\x1a\x0a\x00\x00\x00\x0d" \
    "\x49\x48\x44\x52\x00\x00\x00\x10\x00\x00\x00\x10" \
    "\x08\x06\x00\x00\x00\x1f\xf3\xff\x61\x00\x00\x01" \
    "\xd9\x49\x44\x41\x54\x38\x8d\x8d\x93\x31\x68\x1a" \
    "\x61\x18\x86\x9f\x0b\x29\x74\x94\x92\xa1\x2d\x1d" \
    "\x72\x83\x43\xe1\x0a\xde\x68\x4a\x87\x64\x2a\x8e" \
    "\x1d\xcf\xc9\xd5\x8e\xba\x88\x19\x93\x29\x64\x08" \
    "\x35\xdd\x43\x73\x0e\x25\xe3\x91\x21\xc5\x16\x84" \
    "\xbb\x4d\x97\x5a\x03\x3a\x08\x0e\x8e\x47\x30\x83" \
    "\x60\x87\xb7\xc3\x7f\xea\xa9\xb4\xf4\xe0\xe7\xe5" \
    "\x83\xff\xfd\xbe\x87\xf7\xff\xce\xd2\x54\x9c\x7c" \
    "\x3a\x51\xf1\x4d\x11\xff\xa7\xcf\xff\xa8\xbd\x67" \
    "\x5b\x38\x10\xf5\x22\x68\x94\x90\x24\xb5\x6a\xeb" \
    "\xaa\xb9\xb6\x34\x2e\x21\x8d\x03\x75\xde\xa3\x30" \
    "\x0a\xd1\x54\x10\x9f\x23\xf5\xcf\x24\x1f\x73\xe9" \
    "\x9c\x95\x69\x20\xb9\x39\x57\x6e\xce\x95\x24\x99" \
    "\x61\x13\x29\x2a\xeb\x26\x87\x14\x89\x9d\x66\x0f" \
    "\xc8\xfc\xe0\x7b\x0f\x78\x02\xcd\x44\x47\xc3\x11" \
    "\x64\x21\xff\x2e\x4f\xab\xd4\x62\x34\x1c\xe1\x01" \
    "\xdc\xfb\xe0\x40\x13\xc0\x81\x1d\xcf\x01\x1e\x0e" \
    "\x39\x72\x80\xdf\xe0\x25\x6a\x67\x6d\x18\x42\xf5" \
    "\xb0\xca\xe5\xc3\x25\x76\xd6\x36\xa6\xd7\x45\xe8" \
    "\x81\x87\xc9\xc0\x10\xbc\x1a\x6d\x11\xcc\x1e\x67" \
    "\x90\x85\xfe\xd3\x3e\xc7\xa5\x63\x66\x8f\x33\x3e" \
    "\x3e\x5f\x27\x38\x70\x0e\x36\x32\xf8\x8a\xe4\x63" \
    "\x82\xf4\x91\x6a\x1b\xf5\x5d\xc1\x64\x70\x57\x58" \
    "\x66\x90\xbc\xc2\x99\x69\xa2\x49\xa2\xff\xa8\xc7" \
    "\x81\x34\x29\xe8\x43\x0e\x6d\xbd\xc2\x5f\x27\xa7" \
    "\xeb\x14\x41\x18\x85\x09\xc1\xb4\xac\x56\x0d\xc5" \
    "\x17\xb1\x1a\x17\x0d\xe9\x4a\xba\xbe\xba\x96\xc6" \
    "\x52\xd8\x0e\xa5\x58\x0a\x6e\x03\x01\xcb\xb3\x4e" \
    "\x10\x95\x25\x1f\x63\xee\x9a\xcb\x1a\x4b\x6a\x9b" \
    "\x05\x5a\x34\x01\xd4\xd1\x2f\x01\x1b\x19\x24\x04" \
    "\x1a\xaf\xcc\x61\x3b\x94\xe6\x52\xa7\xdb\x59\x4e" \
    "\xcd\x9c\xee\xcb\xed\x16\x94\x39\xdd\x5f\xd1\xa4" \
    "\x09\x82\xdb\x40\x8a\x93\xc9\xb1\x31\x6b\xbe\x9a" \
    "\x5c\x19\xd4\x75\xa3\x96\x2a\x83\xba\x26\x8a\x4d" \
    "\x83\x46\x09\x69\x52\x30\x04\x29\x53\x5a\xd3\x04" \
    "\x95\x41\x7d\x9b\xc0\xcd\xb9\x26\xed\xae\xd6\xcc" \
    "\xea\xae\xfe\x09\xc9\x90\x2c\x26\x2f\x32\xd8\xbd" \
    "\xcf\x87\xd6\xd1\xb7\xb7\xaa\x7e\x01\x0f\xcb\xac" \
    "\x2b\x66\xd3\xbc\x44\xd3\xdf\x4b\xeb\x19\x00\x2f" \
    "\x3e\x87\x16\x0e\xfc\x01\x3d\xe2\x45\xd3\xc4\x4a" \
    "\xa5\xaa\x00\x00\x00\x00\x49\x45\x4e\x44\xae\x42" \
    "\x60\x82"

#-- preferences ----------------------------------------------------------------
pref = getPreferences(__name__)



class DropLabel(QLabel):
    """label with support for dropping images"""
    def __init__(self, parent=None, name=None, fl=0):
        QLabel.__init__(self, parent, name, fl)
        self.setAcceptDrops(True)
        self.setSizePolicy( QSizePolicy(QSizePolicy.Ignored, QSizePolicy.Ignored) )
        
        # create logging object
        self.log = logging.getLogger( self.__class__.__name__ )
        self.log.setLevel(pref['logLevel'])


    def dragEnterEvent(self, e):
        """called when drop is initiated"""
        accept = QImageDrag.canDecode(e) or XMozUrlDrag.canDecode(e) or \
                 URIListDrag.canDecode(e)
        self.log.debug("Accepting source: %s (type=\"%s\")", str(accept), e.format())
        e.accept( accept )


    def dropEvent(self, e):
        """called when image is dropped on the widget"""
        img = QImage()
        if (QImageDrag.decode(e, img) or XMozUrlDrag.decode(e, img) or \
            URIListDrag.decode(e, img)):
            self.log.debug("Accepting dropped image: %dx%d (%d bytes)",
                           img.width(), img.height(), img.numBytes())
            self.emit(PYSIGNAL("imageDropped"), (img,))
        else:
            self.log.warning("Could not decode dropped image")



class PicturePreview(PicturePreviewBase):
    """displays a picture and scales it appropriately"""
    def __init__(self, dirname, parent=None, name=None, fl=0):
        PicturePreviewBase.__init__(self, parent, name, fl)

        # create logging
        self.log = logging.getLogger( self.__class__.__name__ )
        self.log.setLevel( pref['logLevel'] )

        # runtime variables
        self.basedir = dirname          # remember picture base directory
        self.record = None
        self.img = None                 # original image object
        self.pix = QPixmap()            # pixmap
        self.labelImage = DropLabel(self) # label with drop support
        self.copyImg = None             # when copying images

        self.recycle_pix = QPixmap()
        self.recycle_pix.loadFromData(recycle_bin_data, "PNG")
        self.copy_pix = QPixmap()
        self.copy_pix.loadFromData(copy_data, "PNG")
        self.paste_pix = QPixmap()
        self.paste_pix.loadFromData(paste_data, "PNG")
        self.saveAs_pix = QPixmap()
        self.saveAs_pix.loadFromData(saveAs_data, "PNG")
        
        gridLayoutLabel = QGridLayout( self, 1,1 )
        gridLayoutLabel.addWidget( self.labelImage, 0,0 )

        # connect signal to detect dropped images
        self.connect(self.labelImage, PYSIGNAL("imageDropped"), self.onImageDropped)



    #-- event handlers ---------------------------------------------------------
    def resizeEvent(self, e):
        """called when widget is resized"""
        self.resizeImage()
        

    def mousePressEvent(self, e):
        """called when a mouse button is pressed"""
        # only accept left mouse button
        if (e.button() == e.RightButton):
            menu = QPopupMenu(self)
            menu.insertItem(self.tr("Picture Menu"), None)
            menu.insertSeparator()
            if (self.img): menu.insertItem(QIconSet(self.copy_pix), self.tr("Copy"), self.copyPicture)
            menu.insertItem(QIconSet(self.paste_pix), self.tr("Paste"), self.pastePicture)
            if (self.img): menu.insertItem(QIconSet(self.saveAs_pix), self.tr("Save As"), self.saveAsPicture)
            if (self.img): menu.insertItem(QIconSet(self.recycle_pix), self.tr("Delete"), self.deletePicture)

            if (self.img):
                if (self.copyImg == None):
                    menu.setItemEnabled(menu.idAt(3), False)
                if (self.img == None):
                    menu.setItemEnabled(menu.idAt(4), False)
            else:
                if (self.copyImg == None):
                    menu.setItemEnabled(menu.idAt(2), False)
                    
            menu.exec_loop(QCursor.pos())
            return
            
        # spawn unscaled picture
        if (self.img != None):
            dlg = PictureDialog(self.img, modal=False)
            dlg.exec_loop()


    def saveAsPicture(self):
        """saves the picture as a file"""
        # skip out when no picture available
        if (not self.img):
            self.log.warning("Not saving picture: no picture available")
            return
        
        # display dialog
        filename = QFileDialog.getSaveFileName(
            pref['defaultPath'], self.tr("Images (*.bmp, *.jpeg, *.pbm, *.png, *.ppm *.xbm, *.xpm)"),
            self, None, self.tr("Save file as"))
        # exit on abort
        if (filename == None): return
        # convert to non QString object
        filename = unicode(filename)

        # construct filename
        filenameTuple = splitext(filename)
        imgType = splitext(filename)[1].upper()[1:]
        if (not imgType in ['BMP', 'JPEG', 'PBM', 'PNG', 'PBM', 'PNG', 'PPM', 'XBM', 'XPM']):
            self.log.warning("Image type not recognised: changing to JPEG")
            imgType = 'JPEG'

        # test if already exists
        if (fexists(filename)):
            ans = QMessageBox.question(self, self.tr("File already exists"),
                                       self.tr("File already exists. Overwrite?"),
                                       QMessageBox.Yes, QMessageBox.No)
            if (ans == QMessageBox.No): return

        # save file
        self.log.info("Saving image as %s", basename(filename))        
        ok = self.img.save(filename, str(imgType))
        if (not ok):
            self.log.warning("Could not save picture to file")
            

    def deletePicture(self):
        """removes the picture of the current record"""
        self.log.info("Deleting picture from record")
        self.img = None                 # remove picture

        # query wether picture should be removed from filesystem (linked image only)
        if (len(self.record['embeddedImage']) < MIN_PICTURE_SIZE):
            ans = QMessageBox.question(self, self.tr("delete from file system"), self.tr("Delete picture from filesystem?"),
                                       QMessageBox.Yes, QMessageBox.No)
            if (ans == QMessageBox.Yes):
                self.log.info("Deleting image from filesystem")
                # remove linked image
                filename = fjoin(self.basedir, self.record['picture'])
                if (self.record['picture'] and fexists(filename)):
                    try:
                        os.unlink(filename)
                    except OSError, e:
                        self.log.error("Could not remove linked file: %s", unicode(e))                        
                else:
                    self.log.warning("Not removing image from filesystem: file not found")

        # decode and display changes
        self.imageDecode()              # decode to empty picture
        self.resizeImage()
        

    def copyPicture(self):
        """remembers an image from a record"""
        if (self.img == None):
            self.log.warning("Not copying picture: no picture available")
        self.copyImg = self.img
        

    def pastePicture(self):
        """pasts a copied image to the record"""
        if (self.copyImg == None):
            self.log.warning("Not pasting picture: no picture available")
        self.onImageDropped(self.copyImg)
        

    def onImageDropped(self, img):
        """called when image dropped"""
        self.img = img
        self.record.modified(True)      # causes record to be saved
        self.imageDecode()              # update record
        self.resizeImage()              # update on screen


    #-- interface --------------------------------------------------------------
    def resizeImage(self):
        """puts the image in a label on the screen at the correct size"""
        if (self.img != None):
            size = (self.labelImage.contentsRect().width(), \
                    self.labelImage.contentsRect().height())
            img = self.img.smoothScale( size[0], size[1] , QImage.ScaleMin )
            self.pix.convertFromImage(img)
            self.labelImage.setPixmap(self.pix)

        else:
            self.log.debug("No image set: clearing label")
            self.labelImage.clear()


    def getRecord(self):
        """reads the displayed image back into the record and returns it"""
        self.log.debug("Returning record: %d, %s", self.record['number'], self.record['mediaLabel'])

        # create empty data if none set
        if (self.record == None): return None
        # only update picture data if it was modified
        return self.record


    def imageDecode(self):
        """decodes internal image to be saved in the record"""
        #-- handle embedded image
        if (pref['embeddedImage']):
            # extract image from QImage 
            data = IODeviceBuffer()
            
            if (self.img == None):
                self.log.debug("Saving empty image")
                self.record['embeddedImage'] = ""
                self.record['picture'] = u""                
            elif ( self.img.save(data, pref['imageFormat']) ):
                self.log.debug("Saving embedded image: %d bytes", data.size())
                self.record['embeddedImage'] = data.readAll()
                self.record['picture'] = ".%s"%pref['imageFormat'].lower()
                data.close()            # free memory
            else:
                self.log.error("Could not save image to record field")
                self.record['embeddedImage'] = ""
                self.record['picture'] = u""

        #-- handle save image as link
        else:
            pathToPicture = self.basedir # assume relative path
            
            # absolute picture path or linked image
            if ((pref['filenamePrefix'][0] in ('/', )) or
                (pref['filenamePrefix'][:7] in ('http://',))):
                pathToPicture = ""

            # build filename from prefix (including suffix)
            try:
                filename = (pref['filenamePrefix']%self.record['number'])+\
                           ".%s"%pref['imageFormat'].lower()
            except TypeError, e:
                self.log.error("Invalid filename prefix: %s", unicode(e))
                self.record['embeddedImage'] = ""
                self.record['picture'] = u""

            # create target directory
            abs_filename = fjoin(pathToPicture, filename)
            filedir = dirname(abs_filename)
            if (not fexists(filedir)):
                self.log.info("Creating target directory %s", filedir)
                try:
                    os.makedirs(filedir)
                except IOError, e:
                    self.log.error("Could not create target directory: %s", unicode(e))
                    self.record['embeddedImage'] = ""
                    self.record['picture'] = u""

            # test if file exists
            if (fexists(abs_filename)):
                self.log.warning("Target file %s already exists: overwriting", abs_filename)

            # copy data to target file
            data = IODeviceBuffer()
            # copy to target buffer is successfull
            if (self.img == None):
                self.log.info("Saving empty image")
                self.record['embeddedImage'] = ""
                self.record['picture'] = u""                              
                
            elif (self.img.save(data, pref['imageFormat'])):
                self.log.debug("Saving linked image %s: %d bytes", abs_filename, data.size())
                try:
                    f = open(abs_filename, "wb")
                    f.write(data.readAll())
                    f.close
                    data.close()        # free memory
                except IOError, e:
                    self.log.error("Could not write data to disk: %s", unicode(e))
                    self.record['embeddedImage'] = ""
                    self.record['picture'] = u""
                else:
                    self.record['embeddedImage'] = str(splitext(filename)[1]) # MUST be ASCII string
                    self.record['picture'] = filename                
            # could not save data to target buffer
            else:
                self.log.error("Could not save image to intermediate buffer")
                self.record['embeddedImage'] = ""
                self.record['picture'] = u""

        # notify of modification
        self.emit(PYSIGNAL("modified"), ())
        
        
    def setRecord(self, rec):
        """sets a record and displays the picture"""
        self.log.debug("Setting record: %d, %s", rec['number'], rec['mediaLabel'])
        # remember record
        self.record = rec

        #-- read record fields and put picture, if available on streen
        if (self.record == None):
            self.img = None
            
        # record has embedded image   
        elif (len(self.record['embeddedImage']) > MIN_PICTURE_SIZE):
            self.log.debug("Loading embedded image")
            self.img = QImage()
            ok = self.img.loadFromData(self.record['embeddedImage'])
            if (not ok):
                self.log.error("Could not identify embedded image data")
                self.img = None
                
        # try loading picture from reference
        elif (self.record['picture']):
            self.log.debug("Loading linked image: %s", self.record['picture'])

            pathToPicture = self.basedir # assume relative path
            
            # absolute picture path or linked image
            if ((self.record['picture'][0] in ('/',)) or
                (self.record['picture'][:7] in ('http://',))):
                pathToPicture = ""
                
            # load picture
            f = openResource(fjoin(pathToPicture, self.record['picture']))
            if (f != None):
                self.img = QImage()
                ok = self.img.loadFromData(f.read())
                f.close()
                if (not ok):
                    self.log.error("Could not identify embedded image data")
                    self.img = None
            # could not access file
            else:
                self.img = None
                
        # new record: display no picture
        else:
            self.img = None

        # at this point the image can be:
        # - from http (or whatever openResource loads)
        # - from (correctly) linked image
        # - from temp linked image
        # - embedded data
        # => 2 of 4 reasonst which need explicid decoding
        # image is automatically decoded on drag&drop
        # main program loop must image decode manually when
        # image is loaded from script
        self.resizeImage()




#-- testroutine ----------------------------------------------------------------
if (__name__ == "__main__"):
    from qt import QApplication, QObject, SIGNAL, SLOT
    from AMCFile33 import AMCFile33
    import sys

    a = QApplication(sys.argv)
    QObject.connect(a,SIGNAL("lastWindowClosed()"),a,SLOT("quit()"))

    # need logging handlers for testing
    log = logging.getLogger("")
    hdlr = logging.StreamHandler()
    formatter = logging.Formatter('[%(name)s] %(levelname)s: %(message)s')
    hdlr.setFormatter(formatter)
    log.addHandler(hdlr) 
    log.setLevel(pref['logLevel'])

    # read sample record
    f = AMCFile33(u"/home/merkosh/Projekte/Film- Mp3- Hörspiellisten/Filme/LMC/Filme.amc", "rb")
    rec = f.read()[-1]

    w = PicturePreview(u"/home/merkosh/Projekte/Film- Mp3- Hörspiellisten/Filme/LMC")
    w.setRecord(rec)

    a.setMainWidget(w)
    w.show()

    a.exec_loop()

    tmp = w.getRecord()
    print "len embedded: ",len(tmp['embeddedImage'])
    print "picture:  ",tmp['picture']
