//  Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <config.h>

#include <gtkmm/builder.h>
#include "listerola-window.h"
#include "boxcompose.h"
#include "timing.h"
#include "file.h"
#include "event-box.h"
#include "host-box.h"
#include "hole-box.h"
#include "host-list.h"
#include "host.h"
#include "event.h"
#include "event-date-region.h"
#include "hosted-event-list.h"
#include "hosted-event.h"
#include "listerola.h"

const double first_row_height_ratio = 0.5;
const double first_column_width_ratio = 0.5;

ListerolaWindow::ListerolaWindow (Listerola &l, ListerolaViewOptions &o,
                                  Pango::AttrList event_attrs,
                                  Pango::AttrList host_attrs,
                                  Pango::AttrList header_attrs)
  :listerola(l), opts(o)
{
  event_text_attributes = event_attrs;
  host_text_attributes = host_attrs;
  header_text_attributes = header_attrs;
  guint16 r = opts.get_event_foreground_colour().get_red_p() * 65535;
  guint16 g = opts.get_event_foreground_colour().get_green_p() * 65535;
  guint16 b = opts.get_event_foreground_colour().get_blue_p() * 65535;
  Pango::Attribute event_fg = 
    Pango::Attribute::create_attr_foreground(r, g, b);
  event_text_attributes.insert(event_fg);
  r = opts.get_header_foreground_colour().get_red_p() * 65535;
  g = opts.get_header_foreground_colour().get_green_p() * 65535;
  b = opts.get_header_foreground_colour().get_blue_p() * 65535;
  Pango::Attribute header_fg = 
    Pango::Attribute::create_attr_foreground(r, g, b);
  header_text_attributes.insert(header_fg);
  r = opts.get_host_foreground_colour().get_red_p() * 65535;
  g = opts.get_host_foreground_colour().get_green_p() * 65535;
  b = opts.get_host_foreground_colour().get_blue_p() * 65535;
  Pango::Attribute host_fg = 
    Pango::Attribute::create_attr_foreground(r, g, b);
  host_text_attributes.insert(host_fg);

  Glib::RefPtr < Gtk::Builder > xml =
    Gtk::Builder::create_from_file (File::get_glade_path () + "/listerola.gtk");

  xml->get_widget ("window", window);
  window->signal_size_allocate().connect
    (sigc::mem_fun(*this, &ListerolaWindow::on_window_resized));
  window->signal_delete_event().connect
    (sigc::mem_fun(*this, &ListerolaWindow::on_delete_event));

  xml->get_widget ("listerola_viewport", listerola_viewport);
  xml->get_widget ("listing_hbox", listing_hbox);
  xml->get_widget ("listing_vbox", listing_vbox);
  scroll_hbox = new Gtk::HBox();
  listerola_viewport->add(*manage(scroll_hbox));
  scroll_vbox = new Gtk::VBox();
  scroll_hbox->add(*manage(scroll_vbox));

  xml->get_widget ("header_vbox", header_vbox);
  xml->get_widget ("header_hbox", header_hbox);
  xml->get_widget ("ad_hbox", ad_hbox);
  xml->get_widget ("ad_vbox", ad_vbox);
  xml->get_widget ("menubar", menubar);
  xml->get_widget ("title_label", title_label);
  xml->get_widget ("fullscreen_menuitem", fullscreen_menuitem);
  fullscreen_menuitem->signal_activate().connect
    (sigc::mem_fun(*this, &ListerolaWindow::on_fullscreen_activated));
  xml->get_widget ("quit_menuitem", quit_menuitem);
  quit_menuitem->signal_activate().connect
    (sigc::mem_fun(*this, &ListerolaWindow::on_quit_activated));
  xml->get_widget ("preferences_menuitem", preferences_menuitem);
  preferences_menuitem->signal_activate().connect
    (sigc::mem_fun(*this, &ListerolaWindow::on_preferences_activated));
  xml->get_widget ("about_menuitem", about_menuitem);
  about_menuitem->signal_activate().connect
    (sigc::mem_fun(*this, &ListerolaWindow::on_about_activated));

  calculate_heights_and_widths();
  update_scrolling_speed (opts.get_scroll_speed());
    
  second_handler = Timing::instance ().register_timer
      (sigc::mem_fun (*this, &ListerolaWindow::on_second_passed), 1000);

  window->set_icon_from_file(File::get_data_path() + "/listerola.png");

  window_is_fullscreen = false;
  if (opts.get_start_in_fullscreen())
    on_fullscreen_activated();

  listerola.options_changed.connect
    (sigc::mem_fun (*this, &ListerolaWindow::on_column_scrolled));
  listerola.listing_file_changed.connect
    (sigc::mem_fun (*this, &ListerolaWindow::on_genre_colours_changed));
  listerola.listing_file_changed.connect
    (sigc::mem_fun (*this, &ListerolaWindow::on_host_names_changed));
  opts.scroll_speed_changed.connect
    (sigc::mem_fun (*this, &ListerolaWindow::on_scroll_speed_changed));
  opts.background_colour_changed.connect
    (sigc::mem_fun (*this, &ListerolaWindow::on_background_colour_changed));
  opts.host_background_colour_changed.connect
    (sigc::mem_fun (*this, &ListerolaWindow::on_column_scrolled));
  opts.header_background_colour_changed.connect
    (sigc::mem_fun (*this, &ListerolaWindow::on_column_scrolled));
  opts.show_title_changed.connect
    (sigc::mem_fun (*this, &ListerolaWindow::update_title));
  opts.title_changed.connect
    (sigc::mem_fun (*this, &ListerolaWindow::update_title));
  opts.title_background_colour_changed.connect
    (sigc::mem_fun (*this, &ListerolaWindow::update_title));

  opts.event_foreground_colour_changed.connect
    (sigc::mem_fun (*this, &ListerolaWindow::on_column_scrolled));
  opts.header_foreground_colour_changed.connect
    (sigc::mem_fun (*this, &ListerolaWindow::on_column_scrolled));
  opts.host_foreground_colour_changed.connect
    (sigc::mem_fun (*this, &ListerolaWindow::on_column_scrolled));
  opts.title_foreground_colour_changed.connect
    (sigc::mem_fun (*this, &ListerolaWindow::update_title));
  opts.genre_colours_changed.connect
    (sigc::mem_fun(*this, &ListerolaWindow::on_genre_colours_changed));
  opts.show_host_numbers_changed.connect
    (sigc::mem_fun(*this, &ListerolaWindow::on_column_scrolled));

  update_background_colour();
  update_title();
}

void ListerolaWindow::update_title()
{
  bool resize_box = false;
  if (opts.get_show_title() == false)
    {
      if (title_label->get_no_show_all() == false)
        {
          title_label->set_visible(false);
          title_label->set_no_show_all(true);
          resize_box = true;
        }
    }
  else
    {
      if (title_label->get_no_show_all() == true)
        {
          title_label->set_visible(true);
          title_label->set_no_show_all(false);
          resize_box = true;
        }
      std::string title;
      if (opts.get_title() == "")
        title = "Listerola";
      else
        title = opts.get_title();
      title_label->set_text(title);
      guint16 r = opts.get_title_background_colour().get_red_p() * 65535;
      guint16 g = opts.get_title_background_colour().get_green_p() * 65535;
      guint16 b = opts.get_title_background_colour().get_blue_p() * 65535;
      Pango::AttrList attrs;
      Pango::Attribute bg = Pango::Attribute::create_attr_background(r, g, b);
      attrs.insert(bg);
      Pango::Attribute scale = Pango::Attribute::create_attr_scale(8.0);
      attrs.insert(scale);
      r = opts.get_title_foreground_colour().get_red_p() * 65535;
      g = opts.get_title_foreground_colour().get_green_p() * 65535;
      b = opts.get_title_foreground_colour().get_blue_p() * 65535;
      Pango::Attribute fg = Pango::Attribute::create_attr_foreground(r, g, b); 
      attrs.insert(fg);
      Pango::Attribute weight = 
        Pango::Attribute::create_attr_weight(Pango::WEIGHT_BOLD);
      attrs.insert(weight);
      title_label->set_attributes(attrs);
    }
  window->show();
  if (resize_box)
    {
      while (g_main_context_iteration(NULL, FALSE)); //doEvents
      on_column_scrolled();
    }

}
void ListerolaWindow::update_background_colour()
{
  window->modify_bg(Gtk::STATE_NORMAL, opts.get_background_colour());
  menubar->modify_bg(Gtk::STATE_NORMAL, opts.get_background_colour());
}

void ListerolaWindow::calculate_heights_and_widths()
{

  double total_height = listing_vbox->get_allocation().get_height();
  double total_width = listing_vbox->get_allocation().get_width();
  row_height = total_height / (listerola.get_num_rows() + first_row_height_ratio);
  header_row_height = row_height * first_row_height_ratio;
  double total_columns = listerola.get_duration() / listerola.get_interval();
  double interval_width = total_width / (total_columns + first_column_width_ratio);

  hour_width = interval_width * (1/ listerola.get_interval());
  channel_width = interval_width * first_column_width_ratio;

  if (row_height == 0.0)
    row_height = 1.0;
  if (header_row_height == 0.0)
    header_row_height = 1.0;
  if (hour_width == 0.0)
    hour_width = 1.0;
  if (channel_width == 0.0)
    channel_width = 1.0;

  //what should the text scales be?

  //let's presume a normal text height of 12 or so.
  //fixme: find out how tall the default font is.
  double default_text_height = 12.0;
  double text_row_height = row_height;
  if (text_row_height < 0)
    text_row_height = 1;
  double scale = 1;
  //if (row_height > 48)
    {
      double target_text_height = text_row_height / 4.0;
      scale = target_text_height / default_text_height;
    }
  if (scale < 0.5)
    scale = 0.5;
  if (scale > 1.25)
    scale = 1.25;
  update_scale_attribute(event_text_attributes, scale);

}

void ListerolaWindow::update_scale_attribute(Pango::AttrList &attrs, double s)
{
  Pango::Attribute attr = Pango::Attribute::create_attr_scale(s);
  attr.set_end_index(255);
  attrs.change(attr);
}

void ListerolaWindow::fill_headers()
{
  header_hbox->children ().erase (header_hbox->children ().begin (),
                                  header_hbox->children ().end ());
  EventDate nowdate;
  timebox =
    new FrameBox (Box::ucompose(header_text_attributes, "%1", nowdate.to_string(false)), 
                  Gdk::Color("black"), header_row_height, channel_width);
  timebox->set_background_colour(opts.get_header_background_colour());
  header_hbox->pack_start(*timebox, Gtk::PACK_SHRINK, 0);
  EventDate then = listerola.get_current_date();
  for (double i = 0; i < listerola.get_duration(); i+= listerola.get_interval())
    {
      double width = hour_width * listerola.get_interval();
      if (i + listerola.get_interval() > listerola.get_duration())
        width = listerola.get_duration() + listerola.get_interval() - i;
      FrameBox * interval =
        new FrameBox (Box::ucompose(header_text_attributes, "%1", then.to_string()), 
                      Gdk::Color("white"), header_row_height, width);
      interval->set_background_colour(opts.get_header_background_colour());
      header_hbox->pack_start(*interval, Gtk::PACK_SHRINK, 0);
      then.add_hours(listerola.get_interval());
    }
  header_hbox->show_all();
}

bool
ListerolaWindow::on_second_passed()
{
  EventDate nowdate;

  if (listerola.flip_ahead_to_next_interval())
    on_column_scrolled();

  if (nowdate.subtract(listerola.get_last_refresh_date()) / 3600.0 > 
      opts.get_refresh_interval() && opts.get_refresh_interval() > 0.0)
    listerola.reload_listing_file();

  Gtk::HBox *contents = Box::ucompose(header_text_attributes, "%1", nowdate.to_string(false));
  timebox->set_box_contents(contents);
  timebox->show_all();
  return Timing::CONTINUE;
}

      
void ListerolaWindow::on_column_scrolled()
{
  update_scroll_box_size (listerola.get_num_rows(), listerola.get_duration());
}

void
ListerolaWindow::hide ()
{
  window->hide ();
}

void
ListerolaWindow::show ()
{
  window->show_all ();
  return;
}

void
ListerolaWindow::update_scrolling_speed (double tick_speed)
{
  scroll_tick_handler.disconnect ();
  if (tick_speed > 0.0)
    scroll_tick_handler = Timing::instance ().register_timer
      (sigc::mem_fun (*this, &ListerolaWindow::on_vertical_scroll_tick),
       int (tick_speed));
}

Gtk::HBox * ListerolaWindow::get_next_row ()
{
  listerola.next();

  Gtk::HBox * box = new Gtk::HBox ();
  HostedEventList events = listerola.get_events();
  if (events.size() == 0)
    {
      Gtk::HBox *box = new Gtk::HBox();
      HoleBox *hole = new HoleBox(opts.get_background_colour(),listerola.get_duration() * hour_width, row_height);
      box->pack_start (*hole, Gtk::PACK_SHRINK, 0);
      return box;
    }

  HostBox *hostbox = new HostBox (events.get_host(), true, 
                                  opts.get_show_host_numbers(), 
                                  double (row_height), 
                                  channel_width, host_text_attributes);
  hostbox->set_background_colour(opts.get_host_background_colour());
  box->pack_start (*hostbox, Gtk::PACK_SHRINK, 0);
  EventDate now = listerola.get_current_date();
  for (HostedEventList::iterator it = events.begin (); it != events.end (); 
       it++)
    {
      Event *event = *it;
      if (listerola.get_current_date().get_ending_date() < event->get_date())
        break;
      double duration = event->get_date().get_duration();
      duration -= listerola.overlap_end(event);
      duration -= listerola.overlap_start(event);

      double diff = event->get_date().subtract(now) ;
      if (diff > 5.0) //seconds
        {
          //hey we found a hole.
          double hole_duration = diff / 60.0 / 60.0;
          double hole_width = hour_width * hole_duration;
          if (hole_width >= 1.0)
            {
              HoleBox *hole = new HoleBox(opts.get_background_colour(), 
                                          hole_width, row_height);
              box->pack_start (*hole, Gtk::PACK_SHRINK, 0);
            }
        }
      HostedEvent *he = new HostedEvent(events.get_host(), event);
      EventBox * ebox = new EventBox (he, double (row_height), 
                                      hour_width * duration,
                                      event_text_attributes);

      ebox->set_background_colour(opts.get_genre_colour(event->get_genre()));
      if (listerola.overlap_start(event) > 0.05)
        ebox->set_show_left_arrow(true);
      if (listerola.overlap_end(event) > 0.05)
        ebox->set_show_right_arrow(true);

      box->pack_start (*ebox, Gtk::PACK_SHRINK, 0);
      if (event->get_date ().
          straddles (listerola.get_current_date ().get_ending_date ()))
        break;
      if (listerola.get_current_date().get_ending_date() < event->get_date())
        break;
      now.add_hours(duration);
    }

  return box;
}

void
ListerolaWindow::on_row_scrolled (Gtk::HBox *row)
{
  Gtk::HBox *newrow = get_next_row();
  scroll_vbox->pack_start (*newrow, Gtk::PACK_SHRINK, 0);
  scroll_vbox->children().remove(*row);
  listerola_viewport->show_all();
  while (g_main_context_iteration(NULL, FALSE)); //doEvents
  listerola_viewport->get_vadjustment()->set_value(0);
}

bool
ListerolaWindow::on_vertical_scroll_tick ()
{
  listerola_viewport->get_vadjustment()->set_value(listerola_viewport->get_vadjustment()->get_value() + (opts.get_vertical_scroll_step() * 1.0));
  if (int (listerola_viewport->get_vadjustment()->get_value ()) % row_height == 0)
    {
      bool retval = Timing::CONTINUE;
      if (opts.get_scroll_row_pause() > 0.0)
	retval = on_vertical_scroll_pause ();
  
      Gtk::HBox *row = NULL;
      if (scroll_vbox->children().size() > 0)
        row = dynamic_cast<Gtk::HBox*>(scroll_vbox->children().front().get_widget());
      if (row)
        on_row_scrolled(row);
      return retval;
    }
  return Timing::CONTINUE;
}

bool
ListerolaWindow::on_vertical_scroll_pause ()
{
  scroll_tick_handler.disconnect ();
  scroll_tick_handler = Timing::instance ().register_timer
    (sigc::mem_fun (*this, &ListerolaWindow::on_vertical_scroll_resume),
     int (opts.get_scroll_row_pause()));
  return Timing::STOP;
}

bool
ListerolaWindow::on_vertical_scroll_resume ()
{
  update_scrolling_speed (opts.get_scroll_speed());
  return Timing::STOP;
}

void
ListerolaWindow::update_scroll_box_size (double rows, double columns)
{
  guint16 r = opts.get_event_foreground_colour().get_red_p() * 65535;
  guint16 g = opts.get_event_foreground_colour().get_green_p() * 65535;
  guint16 b = opts.get_event_foreground_colour().get_blue_p() * 65535;
  Pango::Attribute event_fg = 
    Pango::Attribute::create_attr_foreground(r, g, b);
  event_text_attributes.change(event_fg);
  r = opts.get_header_foreground_colour().get_red_p() * 65535;
  g = opts.get_header_foreground_colour().get_green_p() * 65535;
  b = opts.get_header_foreground_colour().get_blue_p() * 65535;
  Pango::Attribute header_fg = 
    Pango::Attribute::create_attr_foreground(r, g, b);
  header_text_attributes.change(header_fg);
  r = opts.get_host_foreground_colour().get_red_p() * 65535;
  g = opts.get_host_foreground_colour().get_green_p() * 65535;
  b = opts.get_host_foreground_colour().get_blue_p() * 65535;
  Pango::Attribute host_fg = 
    Pango::Attribute::create_attr_foreground(r, g, b);
  host_text_attributes.change(host_fg);

  calculate_heights_and_widths();
  fill_headers();
  scroll_vbox->children ().erase (scroll_vbox->children ().begin (),
				  scroll_vbox->children ().end ());

  for (int i = 0; i < rows + 1; i++)
    {
      Gtk::HBox * box = get_next_row ();
      if (box)
        scroll_vbox->pack_start (*box, Gtk::PACK_SHRINK, 0);
    }
  Gtk::Allocation box = listerola_viewport->get_allocation();
  Gtk::Allocation allocation(box.get_x(), box.get_y(), channel_width +
				      (listerola.get_duration() * hour_width),
				      row_height * rows);
  listerola_viewport->size_allocate(allocation);
  Gtk::Adjustment * a = listerola_viewport->get_vadjustment ();
  a->set_upper(row_height * rows);
  listerola_viewport->show_all ();
}

void
ListerolaWindow::on_window_resized(Gtk::Allocation box)
{
  static Gtk::Allocation prev_allocation;
  if (prev_allocation.get_height() == box.get_height() &&
      prev_allocation.get_width() == box.get_width())
    return;
  update_scroll_box_size (listerola.get_num_rows(), listerola.get_duration());
  prev_allocation = box;
}
    
void ListerolaWindow::on_fullscreen_activated()
{
  if (window_is_fullscreen)
    {
      window->unfullscreen();
      Gdk::Cursor blank(Gdk::ARROW);
      window->get_window()->set_cursor(blank);
    }
  else
    {
      window->fullscreen();
      Gdk::Cursor blank(Gdk::BLANK_CURSOR);
      window->get_window()->set_cursor(blank);
    }
  window_is_fullscreen = !window_is_fullscreen;
}

void ListerolaWindow::on_preferences_activated()
{
  if (window_is_fullscreen)
    on_fullscreen_activated();
  open_preferences_requested.emit(get_uncoloured_genres(),
                                  listerola.get_host_names());
}

void ListerolaWindow::on_about_activated()
{
  Gtk::AboutDialog *dialog = new Gtk::AboutDialog;
  dialog->set_name(PACKAGE_NAME);
  dialog->set_version(PACKAGE_VERSION);
  dialog->set_copyright("Copyright (C) 2010 Ben Asselstine");
  dialog->set_comments("This software is licensed under the GNU GPL version 3.");
  dialog->set_logo(Gdk::Pixbuf::create_from_file(File::get_data_path() + "/listerola.png"));
  dialog->set_icon_from_file(File::get_data_path() + "/listerola.png");
  dialog->run();
  delete dialog;
}

void ListerolaWindow::on_quit_activated()
{
  on_delete_event(NULL);
}
    
bool ListerolaWindow::on_delete_event(GdkEventAny *e)
{
  if (window_is_fullscreen)
    on_fullscreen_activated();
  window->hide();
  quit_requested.emit();
  return false;
}
    
void ListerolaWindow::on_scroll_speed_changed()
{
  update_scrolling_speed(opts.get_scroll_speed());
}
  
void ListerolaWindow::on_background_colour_changed()
{
  update_background_colour();
}

std::list<std::string> ListerolaWindow::get_uncoloured_genres()
{
  std::list<std::string> uncoloured_genres;
  std::list<std::string> all_genres = listerola.get_genres();
  std::list<std::string> coloured_genres = opts.get_genres();
  for (std::list<std::string>::iterator it = all_genres.begin(); 
       it != all_genres.end(); it++)
    {
      if (std::find(coloured_genres.begin(), coloured_genres.end(), *it) ==
          coloured_genres.end())
        uncoloured_genres.push_back(*it);
    }
  uncoloured_genres.sort();
  return uncoloured_genres;
}
    
void ListerolaWindow::on_genre_colours_changed()
{
  update_genres_in_preferences_requested.emit(get_uncoloured_genres());
}

void ListerolaWindow::on_host_names_changed()
{
  update_hosts_in_preferences_requested.emit(listerola.get_host_names());
}
